package ohd.hseb.hefs.mefp.tools.canonical;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.StringTokenizer;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.pe.tools.SortedCollection;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.xml.CollectionXMLReader;
import ohd.hseb.hefs.utils.xml.CollectionXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReadable;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLReaderFactory;
import ohd.hseb.hefs.utils.xml.XMLWritable;
import ohd.hseb.hefs.utils.xml.XMLWriter;

/**
 * A list of canonical events designed to store all events defined in either a base event file or a modulation event
 * file. It extends SortedCollection order to ensure that it is a sorted list. Do NOT modify the list elements in place!
 * Any such modification may bread the sorting.
 * 
 * @author hank.herr
 */
public class CanonicalEventList extends SortedCollection<CanonicalEvent> implements XMLReadable, XMLWritable
{
    private String _xmlTagName = "canonicalEventList";

    private CanonicalEventType _eventType;

    private int _periodUnitsInHours;

    private boolean _precipitationFlag;

    public CanonicalEventList(final boolean precipitation)
    {
        this(precipitation, CanonicalEventType.BASE);
    }

    /**
     * Copy constructor.
     */
    public CanonicalEventList(final boolean precipitation, final CanonicalEventList base)
    {
        super();
        _eventType = base._eventType;
        _precipitationFlag = precipitation;
        determinePeriodUnitsInHours();
        addAll(base);
    }

    public CanonicalEventList(final boolean precipitation, final CanonicalEventType eventType)
    {
        super();
        _eventType = eventType;
        _precipitationFlag = precipitation;
        determinePeriodUnitsInHours();
    }

    public CanonicalEventList(final int periodUnitsInHours, final CanonicalEventType eventType)
    {
        super();
        _eventType = eventType;
        _periodUnitsInHours = periodUnitsInHours;
    }

    public CanonicalEventList(final boolean precipitation, final File inputFile) throws Exception
    {
        super();
        _precipitationFlag = precipitation;
        determinePeriodUnitsInHours();
        readFromFile(inputFile);
    }

    /**
     * For testing purposes.
     * 
     * @param events all events in this list
     */
    public CanonicalEventList(final CanonicalEvent... events)
    {
        super(events);
    }

    /**
     * Sets the {@link #_periodUnitsInHours} based on the flag: 6h if true, 24h if false.
     * 
     * @param precipitation Indicates if the events are for precipitation.
     */
    public void determinePeriodUnitsInHours()
    {
        setPeriodUnitsInHours(CanonicalEvent.determineCanonicalEventPeriodUnitInHours(_precipitationFlag));
    }

    /**
     * @param numberOfForecastDays The number of forecast days used for a type of forecast; e.g. rfc, gfs, cfs.
     * @return The number of canonical events that can be applied based on the number of forecast days. This assumes
     *         that the list has been sorted by end lead period.
     */
    public int determineNumberOfApplicableEvents(final int numberOfForecastDays)
    {
        if(numberOfForecastDays == 0)
        {
            return 0;
        }

        int results = 0;
        for(final CanonicalEvent evt: this)
        {
            if(evt.getEndLeadPeriod() * this._periodUnitsInHours <= numberOfForecastDays * 24)
            {
                results++;
            }
            else
            {
                break;
            }
        }
        return results;
    }

    /**
     * @param numberOfEvents
     * @return A {@link CanonicalEventList} containing the first nubmerOfEvents events starting from the beginning of
     *         this list.
     */
    public CanonicalEventList subList(final int numberOfEvents)
    {
        final CanonicalEventList results = new CanonicalEventList(this._periodUnitsInHours, this._eventType);
        results.addAll(ListTools.createSubList(this, numberOfEvents));
        return results;
    }

    /**
     * @param numberOfEvents
     * @return A {@link CanonicalEventList} containing all events in the list starting with firstEventIndex.
     */
    public CanonicalEventList subListAfter(final int firstEventIndex)
    {
        final CanonicalEventList results = new CanonicalEventList(this._periodUnitsInHours, this._eventType);
        results.addAll(ListTools.createSubList(this, firstEventIndex, true, -1));
        return results;
    }

    @SuppressWarnings("unused")
    private void setEventType(final CanonicalEventType eventType)
    {
        _eventType = eventType;
    }

    public void setPeriodUnitsInHours(final int hours)
    {
        this._periodUnitsInHours = hours;
    }

    public int getPeriodUnitsInHours()
    {
        return this._periodUnitsInHours;
    }

    public boolean isBaseEventType()
    {
        return _eventType == CanonicalEventType.BASE;
    }

    public boolean isModulationEventType()
    {
        return _eventType == CanonicalEventType.MODULATION;
    }

    public boolean isPrecipitation()
    {
        return _precipitationFlag;
    }

    public void setToBaseEventType()
    {
        _eventType = CanonicalEventType.BASE;
    }

    public void setToModulationEventType()
    {
        _eventType = CanonicalEventType.MODULATION;
    }

    public void readFromFile(final File inputFile) throws Exception
    {
        this.clear();

        if(!inputFile.exists())
        {
            throw new Exception("Input file, " + inputFile.getAbsolutePath() + ", does not exist.");
        }
        if(!inputFile.canRead())
        {
            throw new Exception("Input file, " + inputFile.getAbsolutePath() + ", exists but cannot be read.");
        }
        final FileReader fileReader = new FileReader(inputFile);
        final BufferedReader buffReader = new BufferedReader(fileReader);

        try
        {
            //Read two header lines
            String line = buffReader.readLine();
            final StringTokenizer tokenizer = new StringTokenizer(line, " ");
            final String eventType = tokenizer.nextToken();
            if((!line.trim().equals(eventType + " Events (Units are 6-hr Periods)"))
                && !line.trim().equals(eventType + " Events (Units are Days)"))
            {
                throw new Exception("Unexpected first header line in canonical event file: '" + line + "'");
            }
            line = buffReader.readLine();
            if(!line.trim().equals("Event   Start   Stop    Dur     NCFSmems"))
            {
                throw new Exception("Unexpected second header line in canonical event file: '" + line + "'");
            }

            _eventType = CanonicalEventType.forName(eventType);

            line = buffReader.readLine();
            while(line != null)
            {
                this.add(new CanonicalEvent(line));
                line = buffReader.readLine();
            }
        }
        finally
        {
            buffReader.close();
        }
    }

    public void writeToFile(final File outputFile) throws IOException
    {
        final FileWriter fileWriter = new FileWriter(outputFile);
        fileWriter.write(_eventType + " Events (Units are " + this._periodUnitsInHours + "-hr Periods)\n");
        fileWriter.write("Event   Start   Stop    Dur     NCFSmems\n");
        for(final CanonicalEvent event: this)
        {
            fileWriter.write(event.writeToLine() + "\n");
        }
        fileWriter.close();
    }

    /**
     * @param ts The time series for which to calculate events.
     * @param t0 The T0 to assume for the event, since it may not match the time series (for observed data in
     *            particular).
     * @return A mapping of {@link CanonicalEvent} to the computed event value as a {@link Double}.
     */
    public LinkedHashMap<CanonicalEvent, Double> computeEvents(final TimeSeriesArray ts, final long t0)
    {
        //TODO There might be efficiency to gain by having the computation done here for all canonical events at once, in one big loop,
        //without computing them independently.
        final LinkedHashMap<CanonicalEvent, Double> results = new LinkedHashMap<CanonicalEvent, Double>();
        for(final CanonicalEvent event: this)
        {
            results.put(event, event.computeEvent(ts, t0, _periodUnitsInHours));
        }
        return results;
    }

    /**
     * @param ts The time series for which to calculate events.
     * @param t0 The T0 to assume for the event, since it may not match the time series (for observed data in
     *            particular).
     * @param doNotAllowMissing If true, then if any time series value is missing when computing an event, no event
     *            value will be computed for that event. It will be stored as {@link Double#NaN} in the returned map. If
     *            false, the missing values are skipped and computation continues. Only if all values to be used in the
     *            computation of an event are missing, will the event value be {@link Double#NaN}.
     * @param useMemberSizeLimit Flag passed through to
     *            {@link CanonicalEvent#computeEvent(List, long, int, boolean, boolean)}.
     * @return A mapping of {@link CanonicalEvent} to the computed event value as a {@link Double}.
     */
    public LinkedHashMap<CanonicalEvent, Double> computeEvents(final List<TimeSeriesArray> ts,
                                                               final long t0,
                                                               final boolean doNotAllowMissing,
                                                               final boolean useMemberSizeLimit)
    {
        final LinkedHashMap<CanonicalEvent, Double> results = new LinkedHashMap<CanonicalEvent, Double>();
        for(final CanonicalEvent event: this)
        {
            results.put(event, event.computeEvent(ts, t0, _periodUnitsInHours, doNotAllowMissing, useMemberSizeLimit));
        }
        return results;
    }

    public void writeToModelParameterFile(final String stream) throws Exception
    {
        throw new Exception("This hasn't been coded yet!!!!!!!");
    }

    public void setXMLTagName(final String name)
    {
        this._xmlTagName = name;
    }

    /**
     * Re-index elements.
     */
    public void indexElements()
    {
        int i = 1;
        for(final CanonicalEvent event: this)
        {
            event.setEventNumber(i);
            i++;
        }
    }

    public String getXMLTagName()
    {
        return _xmlTagName;
    }

    /**
     * @throws Exception If any event's {@link CanonicalEvent#validate()} throws an exception. Only this first exception
     *             is caught.
     */
    public void validate() throws Exception
    {
        for(final CanonicalEvent evt: this)
        {
            evt.validate();
        }
    }

    @Override
    public XMLWriter getWriter()
    {
        return new CollectionXMLWriter(getXMLTagName(), this);
    }

    @Override
    public XMLReader getReader()
    {
        //The reader must return instances of CanonicalEvent.  Additionally, the finalizeReading resorts the list
        //via informChanged() and then indexes the elements.
        return new CollectionXMLReader<CanonicalEvent>(getXMLTagName(), this, new XMLReaderFactory<CanonicalEvent>()
        {
            @Override
            public CanonicalEvent get()
            {
                return new CanonicalEvent();
            }
        })
        {
            @Override
            public void finalizeReading() throws XMLReaderException
            {
                super.finalizeReading();
                informChanged();
                indexElements();
            }
        };
    }

    @Override
    public boolean equals(final Object other)
    {
        if(!super.equals(other))
        {
            return false;
        }
        if(!(other instanceof CanonicalEventList))
        {
            return false;
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        return super.hashCode() ^ _eventType.hashCode() ^ _xmlTagName.hashCode()
            ^ new Integer(_periodUnitsInHours).hashCode();
    }

    @Override
    public CanonicalEvent[] toArray()
    {
        final CanonicalEvent[] results = new CanonicalEvent[size()];
        for(int i = 0; i < size(); i++)
        {
            results[i] = get(i);
        }
        return results;
    }
}
