package ohd.hseb.hefs.mefp.tools.canonical;

import java.util.Set;

import javax.swing.JTable;

import ohd.hseb.hefs.utils.gui.jtable.GenericTable;
import ohd.hseb.hefs.utils.gui.jtable.TableTools;
import ohd.hseb.hefs.utils.gui.jtable.models.AbstractToolTipTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.AddDeleteRowTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.AddDeleteTableModelWrapper;
import ohd.hseb.hefs.utils.gui.jtable.models.PreferredWidthsTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.TableAwareTableModel;

import com.google.common.collect.Sets;

/**
 * An {@link AddDeleteTableModelWrapper} that allows for displaying the canonical events in an editable table via a
 * {@link GenericTable}.
 * 
 * @author hank.herr
 */
@SuppressWarnings("serial")
public class CanonicalEventTableModel extends AddDeleteTableModelWrapper
{
    private static final String[] COLUMN_NAMES = {"Event Number", "Start", "End", "Length", "Lagged Members"};
    private static final String ADD_STRING = "Add a canonical event.";
    private static final String DELETE_STRING = "Remove this canonical event.";

    /**
     * Attribute allows for the {@link #getDisplayedEvents()} method to return it so that external tools can see which
     * object is currently displayed.
     */
    private final CanonicalEventList _displayedEvents;

    public CanonicalEventTableModel(final CanonicalEventList events)
    {
        super(new InnerTableModel(events), ADD_STRING, DELETE_STRING);
        _displayedEvents = events;
    }

    /**
     * @return {@link CanonicalEventList} containing the displayed events in the current table.
     */
    public CanonicalEventList getDisplayedEvents()
    {
        return _displayedEvents;
    }

    private static class InnerTableModel extends AbstractToolTipTableModel implements PreferredWidthsTableModel,
    AddDeleteRowTableModel, TableAwareTableModel
    {
        private final CanonicalEventList _data;

        private final Set<JTable> _tables;

        public InnerTableModel(final CanonicalEventList events)
        {
            _data = events;
            _tables = Sets.newHashSet();
        }

        @Override
        public int getRowCount()
        {
            return _data.size();
        }

        @Override
        public int getColumnCount()
        {
            return 5;
        }

        @Override
        public Class<?> getColumnClass(final int col)
        {
            return Integer.class;
        }

        @Override
        public String getColumnName(final int col)
        {
            if(col == 3)
            {
                return COLUMN_NAMES[col] + String.format(" (%dhrs)", _data.getPeriodUnitsInHours());
            }
            else if(0 <= col && col < COLUMN_NAMES.length)
            {
                return COLUMN_NAMES[col];
            }
            return null;
        }

        @Override
        public Integer getValueAt(final int row, final int column)
        {
            final CanonicalEvent event = _data.get(row);
            if(column == 0)
            {
                return event.getEventNumber();
            }
            else if(column == 1)
            {
                return event.getStartLeadPeriod();
            }
            else if(column == 2)
            {
                return event.getEndLeadPeriod();
            }
            else if(column == 3)
            {
                return event.getDuration();
            }
            else if(column == 4)
            {
                return event.getNumberOfLaggedEnsembleMembers();
            }
            return null;
        }

        @Override
        public boolean isCellEditable(final int row, final int col)
        {
            return col != 3 && col != 0;
        }

        @Override
        public void setValueAt(Object aValue, final int row, final int col)
        {
            // Cast to int if able.
            if(!(aValue instanceof Integer))
            {
                if(aValue instanceof Number)
                {
                    aValue = ((Number)aValue).intValue();
                }
                else
                {
                    return;
                }
            }
            final int value = (Integer)aValue;

            final CanonicalEvent event = _data.get(row);
            if(col == 1)
            {
                event.setStartLeadPeriod(value);
            }
            else if(col == 2)
            {
                event.setEndLeadPeriod(value);
            }
            else if(col == 4)
            {
                event.setNumberOfLaggedEnsembleMembers(value);
            }
            else
            {
                return;
            }

            _data.informChanged(row);
            _data.indexElements();

            fireTableDataChanged();

            final int index = _data.indexOf(event);
            for(final JTable table: _tables)
            {
                TableTools.selectRowAndScrollToIt(table, index);
            }

        }

        @Override
        public Integer getPreferredWidth(final int column)
        {
            return 40;
        }

        @Override
        public void addRow()
        {
            final CanonicalEvent e = new CanonicalEvent();
            if(!_data.isEmpty()) //If empty, the default -1 value is kept.
            {
                e.setEndLeadPeriod(_data.last().getEndLeadPeriod() + 1);
            }
            _data.add(e);
            _data.indexElements();
            this.fireTableDataChanged();
            for(final JTable table: _tables)
            {
                table.setRowSelectionInterval(_data.size() - 1, _data.size() - 1);
            }
        }

        @Override
        public void deleteRow(final int indexOfRowClicked, final int[] allTableSelectedIndices)
        {
            _data.remove(indexOfRowClicked);
            _data.indexElements();
            this.fireTableDataChanged();
        }

        @Override
        public String getColumnHeaderToolTip(final int column)
        {
            if(column == 3)
            {
                return String.format("Length in %d hour increments.", _data.getPeriodUnitsInHours());
            }
            if(0 <= column && column < COLUMN_NAMES.length)
            {
                return COLUMN_NAMES[column];
            }
            return null;
        }

        @Override
        public String getCellToolTip(final int row, final int col)
        {
            return null;
        }

        @Override
        public void addTable(final JTable table)
        {
            _tables.add(table);
        }

        @Override
        public void removeTable(final JTable table)
        {
            _tables.remove(table);
        }
    }
}