package ohd.hseb.hefs.mefp.tools.canonical;

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Set;
import java.util.TreeMap;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesHeader;
import ohd.hseb.hefs.utils.datetime.AstronomicalJulianDay;
import ohd.hseb.hefs.utils.datetime.DateTools;
import ohd.hseb.hefs.utils.effect.Effect;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tsarrays.LaggedEnsemble;
import ohd.hseb.hefs.utils.xml.ArrayXMLReader;
import ohd.hseb.hefs.utils.xml.ArrayXMLWriter;
import ohd.hseb.hefs.utils.xml.CompositeXMLWriter;
import ohd.hseb.hefs.utils.xml.ListXMLReader;
import ohd.hseb.hefs.utils.xml.XMLReadable;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderFactory;
import ohd.hseb.hefs.utils.xml.XMLWritable;
import ohd.hseb.hefs.utils.xml.XMLWriter;
import ohd.hseb.hefs.utils.xml.vars.XMLLong;
import ohd.hseb.util.misc.HCalendar;

/**
 * Wraps a {@link LinkedHashMap} mapping milliseconds to a float[] containing computed event values. The milliseconds
 * always represent the COMPUTATIONAL 12Z T0, or 0th time step, that corresponds to the canonical event computations.
 * When looking at reforecasts, the computational time is what the T0 would be for that data to be used in an
 * operational forecast and accounts for any operational lag in receiving forecasts. It provides a tool to take a julian
 * day and compute its milliseconds using a 12Z time of day. It also specifies an {@link XMLReader} and
 * {@link XMLWriter}, which each use three tag names: a top-level tag name, a tag name for forecast time values, and a
 * tag name for individual event values for a forecast time. Each tag name can be set externally via the appropriate set
 * method.<br>
 * <br>
 * NOTE: {@link AstronomicalJulianDay#computeMillis(double)} can be used to compute milliseconds from an astronomical
 * julian day, which is how things are stored in the binary file. Once stored here, all access should be via
 * milliseconds.
 * 
 * @author Hank.Herr
 */
public class CanonicalEventValues implements XMLReadable, XMLWritable
{
    /**
     * Straight forecast time to values map, kept as a TreeMap which sorts the keys by forecast computation time.
     * This is necessary for the getFirst* and getLast* methods below.
     */
    private final TreeMap<Long, float[]> _forecastComputationTimeToEventValuesMap = new TreeMap<Long, float[]>();

    private final CanonicalEventList _eventsToCompute;
    private String _topTagName = "canonicalEventValues";
    private String _forecastTimeValuesTagName = "forecastTimeValues";
    private String _eventValueTagName = "eventValue";

    public CanonicalEventValues(final CanonicalEventList eventsToCompute)
    {
        _eventsToCompute = eventsToCompute;
    }

    public void copyValues(final CanonicalEventValues base)
    {
        if(!_eventsToCompute.equals(base.getEventsToCompute()))
        {
            throw new IllegalArgumentException("Base canonical event list does not have the same events to compute as this.");
        }
        for(final Long millis: base.getComputationalTimes())
        {
            _forecastComputationTimeToEventValuesMap.put(millis, base.getEventValues(millis));
        }
    }

    /**
     * Compute the canonical event values for list of time series, typically forecast time series. For each time series,
     * the canonical event is computed relative to the first 12Z time found AFTER the forecast time. Usually, this will
     * be the forecast time itself, but, for GEFS, for example, the forecast times are always 0Z, so the 12Z
     * determination is required. The events are stored based on the computational 12Z time.
     * 
     * @param timeSeries {@link Collection} of {@link TimeSeriesArray} instances specifying forecast time series.
     * @param lagInHours The operational lag to apply when computing the events. When applied to reforecast time series,
     *            the lag may be needed to account for a lag operationally, and is applied by adding the provided number
     *            of hours to the time series forecast time BEFORE finding the first 12Z time after it, which becomes
     *            the computational time. The computational time would then effectively be the T0 at which the
     *            reforecast would have been used operationally.
     */
    public void computeEvents(final Collection<TimeSeriesArray> timeSeries, final int lagInHours)
    {
        _forecastComputationTimeToEventValuesMap.clear();
        for(final TimeSeriesArray ts: timeSeries)
        {
            final long computationTime = DateTools.findFirstExact12ZTimeAfterOrOn(ts.getHeader().getForecastTime()
                + lagInHours * HCalendar.MILLIS_IN_HR);
            final LinkedHashMap<CanonicalEvent, Double> computedValues = _eventsToCompute.computeEvents(ts,
                                                                                                        computationTime);
            putEventValues(computationTime, computedValues);
        }
    }

    /**
     * Creates two sublists of {@link #_eventsToCompute}. The first list has size numberOfEventsForFirstTimeSeries and
     * is used to compute events for firstTimeSeries. The second list contains the rest of the events and is used to
     * compute events for secondTimeSeries. Computation is done by navigating both collections simultaneously in order
     * and finding shared {@link TimeSeriesHeader#getForecastTime()} returned times. When a time is found that is common
     * to both, those two {@link TimeSeriesArray} instances are used to compute events and the two returned lists of
     * event values are combined and stored with the computational forecast time, or the 0th time step of the
     * computations.<br>
     * <br>
     * Note that this calls {@link CanonicalEventList#computeEvents(List, long, boolean, boolean)} passing in true for
     * using the ensemble size specified by {@link CanonicalEvent#getNumberOfLaggedEnsembleMembers()}. This will have to
     * change if we ever do NOT want to use that limit!<br>
     * <br>
     * Note that the canonical events are always computed starting at 12Z, so the forecast times under which values are
     * computed will not match the data forecast times if those tiimes are not 12Z.
     * 
     * @param <G> Ensemble of time series must be a {@link List} of {@link TimeSeriesArray} instances. Instances of
     *            {@link LaggedEnsemble} is a valid choice.
     * @param <E> The class storing the ensembles G must be a subclass of {@link Collection}.
     * @param numberOfEventsForFirstTimeSeries The number of events within {@link #_eventsToCompute} to compute using
     *            the time series within firstTimeSeries. The rest are computed using those in secondTimeSeries.
     * @param firstTimeSeries A {@link Collection} of {@link List}s of {@link TimeSeriesArray} instances. Each list is
     *            an ensemble. A {@link List} of {@link LaggedEnsemble} instances can be passed in, for example.
     * @param secondTimeSeries See firstTimeSeries.
     * @param lagInHours The operational lag to apply when computing the events. When applied to reforecast time series,
     *            the lag may be needed to account for a lag operationally, and is applied by adding the provided number
     *            of hours to the time series forecast time BEFORE finding the first 12Z time after it, which becomes
     *            the computational time. The computational time would then effectively be the T0 at which the
     *            reforecast would have been used operationally.
     */
    public <G extends List<TimeSeriesArray>, E extends Collection<G>> void computeEvents(final int numberOfEventsForFirstTimeSeries,
                                                                                         final E firstTimeSeries,
                                                                                         final E secondTimeSeries,
                                                                                         final int lagInHours)
    {
        final CanonicalEventList firstEvents = _eventsToCompute.subList(numberOfEventsForFirstTimeSeries);
        final CanonicalEventList secondEvents = _eventsToCompute.subListAfter(numberOfEventsForFirstTimeSeries);

        final Iterator<G> firstIter = firstTimeSeries.iterator();
        final Iterator<G> secondIter = secondTimeSeries.iterator();
        List<TimeSeriesArray> firstEns = null;
        List<TimeSeriesArray> secondEns = null;
        long forecastTime;

        //Loop over all firstIter time series.
        while(firstIter.hasNext())
        {
            firstEns = firstIter.next();
            forecastTime = firstEns.get(0).getHeader().getForecastTime();

            //If the secondTS has not been initially set yet, or if its forecast time is before the firstTS
            //forecast time, then iterate through secondIter until we find a time series equal or after
            //firstTS.  Break out if we reach the end of secondIter.
            while((secondEns == null) || (forecastTime > secondEns.get(0).getHeader().getForecastTime()))
            {
                if(!secondIter.hasNext())
                {
                    //If the second set of time series runs out without finding any time series that satisfies
                    //the while condition, then stop. This is one exit condition: secondIter runs out!
                    return;
                }
                secondEns = secondIter.next();
            }

            //If the second time series found has the same forecast time as the first, then events can be computed
            //for the both of them and combined into a single computedValues map.
            if((secondEns != null) && (secondEns.get(0).getHeader().getForecastTime() == forecastTime))
            {
                //Computation time is the forecast time shifted to the first 12Z slot after its beginning.
                final long computationTime = DateTools.findFirstExact12ZTimeAfterOrOn(forecastTime + lagInHours
                    * HCalendar.MILLIS_IN_HR);

                final LinkedHashMap computedValues = firstEvents.computeEvents(firstEns, computationTime, false, true);
                computedValues.putAll(secondEvents.computeEvents(secondEns, computationTime, false, true));
                putEventValues(computationTime, computedValues);
            }
        }
    }

    /**
     * Compute the canonical event values from a single long time series, typically observed historical, for specified
     * forecast times.
     * 
     * @param longTimeSeries The time series.
     * @param computationalTimes The forecast times for which to compute the event values. This should be computational
     *            12Z times.
     * @param lagToApply Should always be 0. Specifies the lag (hours) to apply when computing events from the
     *            longTimeSeries based on given forecast times. For example, if the lag is 24-hours (as for CFSv2
     *            forecasts), then for a given forecast time, the events to pair with it from the longTimeSeries are
     *            computed relative to the forecast time + 24-hours. In general, the computational times should
     *            represent the forecast times to use after accounting for lags, so this should always be 0. I'll leave
     *            it in just in case, however!
     */
    public void computeEvents(final TimeSeriesArray longTimeSeries,
                              final Collection<Long> computationalTimes,
                              final int lagToApply)
    {
        //NOTE:Note that GEFS has a 12-hour lag; its reforecasts are at 0Z, but the used
        //            computation times are at 12Z. However, that is handled internally, because the forecast event values
        //            herein are stored at the 12Z computation times, so that when passed to this method, the 12Z times are
        //            provided.

        for(final Long t0: computationalTimes)
        {
            //XXX Do we need to store the values if the resulting event values are all missing?  
            //Currently, this will still record the values for the T0.  It may be useful to know that
            //event computation was attempted, but failed due to missing data, and saving all missings would
            //allow for that.  If we did not save it, we may wonder if it event attempted computation.  
            //On the other hand, do we care to know whether computation was attempted?
            final LinkedHashMap<CanonicalEvent, Double> computedValues = _eventsToCompute.computeEvents(longTimeSeries,
                                                                                                        t0
                                                                                                            + lagToApply
                                                                                                            * HCalendar.MILLIS_IN_HR);
            putEventValues(t0, computedValues);
        }
    }

    /**
     * @param computationalTime The computational forecast time used; i.e., the 12Z T0 basis time for event
     *            computations.
     * @param values The values.
     */
    public void putEventValues(final Long computationalTime, final float[] values)
    {
        _forecastComputationTimeToEventValuesMap.put(computationalTime, values);
    }

    /**
     * @param computationalTime The computational forecast time used; i.e., the 12Z T0 basis time for event
     *            computations.
     * @param computedValues {@link LinkedHashMap} of computed values for all events. The values are stored as doubles.
     */
    private void putEventValues(final Long computationalTime, final LinkedHashMap<CanonicalEvent, Double> computedValues)
    {
        final float[] eventValues = new float[computedValues.size()];
        for(int i = 0; i < computedValues.size(); i++)
        {
            eventValues[i] = computedValues.get(_eventsToCompute.get(i)).floatValue();
        }
        _forecastComputationTimeToEventValuesMap.put(computationalTime, eventValues);
    }

    /**
     * Clears the {@link #_forecastComputationTimeToEventValuesMap} event values mapping.
     */
    public void clearEventValues()
    {
        this._forecastComputationTimeToEventValuesMap.clear();
    }

    public boolean isEmpty()
    {
        return getComputationalTimes().isEmpty();
    }

    /**
     * @return The events to compute. The size of the list will match the size of float[] returned via the get methods.
     *         The order of the float values has a one-to-one correspondence to {@link CanonicalEventList}.
     */
    public CanonicalEventList getEventsToCompute()
    {
        return _eventsToCompute;
    }

    /**
     * You may need to use {@link DateTools#findFirst12ZTimeAfterOrEqualTo(long)} to determine the passed in forecast
     * time because that is used to determine the forecast times at which events are stored.
     * 
     * @param computationalTime The computational forecast time used; i.e., the 12Z T0 basis time for event
     *            computations.
     * @return Array of values, whose order corresponds to the return of {@link #getEventsToCompute()}.
     */
    public float[] getEventValues(final long computationalTime)
    {
        return _forecastComputationTimeToEventValuesMap.get(computationalTime);
    }

    /**
     * You may need to use {@link DateTools#findFirst12ZTimeAfterOrEqualTo(long)} to determine the passed in forecast
     * time because that is used to determine the forecast times at which events are stored.
     * 
     * @param event {@link CanonicalEvent} for which to acquire the value.
     * @param computationalTime The computational forecast time used; i.e., the 12Z T0 basis time for event
     *            computations.
     * @return A single canonical event value; uses {@link #_eventsToCompute} and
     *         {@link #_forecastComputationTimeToEventValuesMap} .
     */
    public float getEventValue(final CanonicalEvent event, final long computationalTime)
    {
        final int index = _eventsToCompute.indexOf(event);
        return _forecastComputationTimeToEventValuesMap.get(computationalTime)[index];
    }

    /**
     * @return Set of forecastTimes, each being the computation 12Z T0 for which the events were computed (not
     *         necessarily the true forecast times). {@link #getEventValues(long)}.
     */
    public Set<Long> getComputationalTimes()
    {
        return _forecastComputationTimeToEventValuesMap.keySet();
    }

    public long getFirstComputationalTime()
    {
        return ListTools.first(_forecastComputationTimeToEventValuesMap.keySet());
    }

    public long getLastcomputationalTime()
    {
        return ListTools.last(_forecastComputationTimeToEventValuesMap.keySet());
    }

    public int getNumberOfComputationalTimesWithValues()
    {
        return getComputationalTimes().size();
    }

    public String getTopTagName()
    {
        return _topTagName;
    }

    public void setTopTagName(final String topTagName)
    {
        _topTagName = topTagName;
    }

    public String getForecastTimeValuesTagName()
    {
        return _forecastTimeValuesTagName;
    }

    public void setForecastTimeValuesTagName(final String forecastTimeValuesTagName)
    {
        _forecastTimeValuesTagName = forecastTimeValuesTagName;
    }

    public String getEventValueTagName()
    {
        return _eventValueTagName;
    }

    public void setEventValueTagName(final String eventValueTagName)
    {
        _eventValueTagName = eventValueTagName;
    }

    public void writeToByteStream(final DataOutputStream stream) throws IOException
    {
        stream.writeInt(_forecastComputationTimeToEventValuesMap.keySet().size());
        for(final Long millis: _forecastComputationTimeToEventValuesMap.keySet())
        {
            final float[] values = _forecastComputationTimeToEventValuesMap.get(millis);
            stream.writeLong(millis);
            stream.writeInt(values.length);
            for(int i = 0; i < values.length; i++)
            {
                stream.writeFloat(values[i]);
            }
        }
    }

    public void readFromByteStream(final DataInputStream stream) throws IOException
    {
        _forecastComputationTimeToEventValuesMap.clear();
        final int numberToRead = stream.readInt();
        for(int i = 0; i < numberToRead; i++)
        {
            final long forecastTime = stream.readLong();
            final int numberOfValues = stream.readInt();
            final float[] values = new float[numberOfValues];
            for(int j = 0; j < numberOfValues; j++)
            {
                values[j] = stream.readFloat();
            }

            _forecastComputationTimeToEventValuesMap.put(forecastTime, values);
        }
    }

    @Override
    public XMLWriter getWriter()
    {
        final List<ArrayXMLWriter> forecastTimeWriters = new ArrayList<ArrayXMLWriter>(_forecastComputationTimeToEventValuesMap.size());
        for(final Long millis: _forecastComputationTimeToEventValuesMap.keySet())
        {
            final ArrayXMLWriter writer = new ArrayXMLWriter(getForecastTimeValuesTagName(),
                                                             getEventValueTagName(),
                                                             _forecastComputationTimeToEventValuesMap.get(millis));
            writer.setUseDelimiter(true);
            writer.setNumberFormatter(new DecimalFormat("0.00000"));
            writer.addAttribute(new XMLLong("milliseconds", millis), true);
            forecastTimeWriters.add(writer);
        }
        return new CompositeXMLWriter(getTopTagName(), forecastTimeWriters);
    }

    @Override
    public XMLReader getReader()
    {
        //The time series are read in as a list, with each element of the list being a ArrayXMLReader
        //with a single attribute: milliseconds.  When done reading the list, the _millisecondsToEventValuesMap
        //is populated based on the contents of the list by retrieving the milliseconds and the array of numbers.
        final ListXMLReader julianDayReaders = new ListXMLReader<ArrayXMLReader>(getTopTagName(),
                                                                                 new XMLReaderFactory<ArrayXMLReader>()
                                                                                 {
                                                                                     @Override
                                                                                     public ArrayXMLReader get()
                                                                                     {
                                                                                         final ArrayXMLReader julianDayReader = new ArrayXMLReader(getForecastTimeValuesTagName(),
                                                                                                                                                   getEventValueTagName(),
                                                                                                                                                   new float[_eventsToCompute.size()]);
                                                                                         julianDayReader.addAttribute(new XMLLong("milliseconds"),
                                                                                                                      true);
                                                                                         return julianDayReader;
                                                                                     }
                                                                                 },
                                                                                 new Effect<List<ArrayXMLReader>>()
                                                                                 {
                                                                                     @Override
                                                                                     public void perform(final List<ArrayXMLReader> input)
                                                                                     {
                                                                                         final List<ArrayXMLReader> readers = input;
                                                                                         for(final ArrayXMLReader reader: readers)
                                                                                         {
                                                                                             _forecastComputationTimeToEventValuesMap.put(((XMLLong)reader.getAttributes()
                                                                                                                                                          .retrieve("milliseconds")
                                                                                                                                                          .getStorageObject()).get(),
                                                                                                                                          (float[])reader.getStorageObject());
                                                                                         }
                                                                                     }
                                                                                 });
        return julianDayReaders;
    }

 // Dumps a events .bin file to a text file.

    public static void main(final String[] args)
    {
        final DecimalFormat nf = new DecimalFormat("0.000000");
       
       
        try
        {
            final String inFileName = args[0];
            final String outFileName = args[0] + ".dump";
           
            final BufferedInputStream bufStream = new BufferedInputStream(new FileInputStream(new File(args[0])));
            final DataInputStream dataStream = new DataInputStream(bufStream);

            final FileWriter writer = new FileWriter(new File(outFileName));

            int numberToRead = dataStream.readInt();
            for(int i = 0; i < numberToRead; i++)
            {
                System.out.println("####>> " + i + " of " + numberToRead);
                final long forecastTime = dataStream.readLong();
                final int numberOfValues = dataStream.readInt();
                final double[] values = new double[numberOfValues];
                for(int j = 0; j < numberOfValues; j++)
                {
                    values[j] = dataStream.readFloat();
                }
               
                String outStr = HCalendar.buildDateStr(forecastTime, HCalendar.DEFAULT_DATE_FORMAT);
                for (final double value : values)
                {
                    outStr += "," + nf.format(value);
                }
               
                writer.write(outStr + "\n");
            }
           
            writer.write("===========================================================================\n");
           
             numberToRead = dataStream.readInt();
            for(int i = 0; i < numberToRead; i++)
            {
                System.out.println("####>> " + i + " of " + numberToRead);
                final long forecastTime = dataStream.readLong();
                final int numberOfValues = dataStream.readInt();
                final double[] values = new double[numberOfValues];
                for(int j = 0; j < numberOfValues; j++)
                {
                    values[j] = dataStream.readFloat();
                }
               
                String outStr = HCalendar.buildDateStr(forecastTime, HCalendar.DEFAULT_DATE_FORMAT);
                for (final double value : values)
                {
                    outStr += "," + nf.format(value);
                }

                writer.write(outStr + "\n");
            }
          
            writer.close();
        }
        catch(final Exception e)
        {
            e.printStackTrace();
        }
    }
}
