package ohd.hseb.hefs.mefp.tools.canonical;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Iterator;
import java.util.List;
import java.util.TimeZone;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Lists;

import ohd.hseb.hefs.utils.BinaryPredicate;
import ohd.hseb.util.misc.HCalendar;

/**
 * This is an implementation of the MEFP data gathering algorithm. The algorithm incorporates a window that grows five
 * days at a time. The mechanism for checking if the window is sufficient for a particular canonical event is provided
 * as a {@link BinaryPredicate} to the constructor. The data gathered for all events at once, and runs until all events
 * have enough data or a maximum window width is reached. Note that once enough values are gathered for a particular
 * event, that list of values will no longer be added to.<br>
 * <br>
 * To use this class, construct an instance first. Note that it will populate a map of days of year to values upon
 * construction, which may take some time. Then, for each desired day-of-year, call the {@link #gatherEventValues(int)}
 * followed by the get methods to acquire the values gathered.
 * 
 * @author hank.herr
 */
public class CanonicalEventValuesGatherer
{

    /**
     * Returns if enough events were found.
     */
    private BinaryPredicate<List<Float>, List<Float>> _listAcceptanceCheck;

    /**
     * The initial, minimum window width.
     */
    private final int _startingWindowWidth;

    /**
     * The largest allowed window width.
     */
    private final int _maximumWindowWidth;

    /**
     * The canonical events for which events were computed; dictates the number of events for which values are gathered,
     * and the size of all arrays stored herein.
     */
    private final CanonicalEventList _events;

    /**
     * Since values are accessed during parameter estimation by day of year, this mapping will allow for quick access.
     */
    private final ArrayListMultimap<Integer, float[]> _dayOfYearToFcstValuesMap = ArrayListMultimap.create();

    /**
     * Since values are accessed during parameter estimation by day of year, this mapping will allow for quick access.
     */
    private final ArrayListMultimap<Integer, long[]> _dayOfYearToT0sMap = ArrayListMultimap.create();

    /**
     * Since values are accessed during parameter estimation by day of year, this mapping will allow for quick access.
     */
    private final ArrayListMultimap<Integer, float[]> _dayOfYearToObsValuesMap = ArrayListMultimap.create();

    /**
     * Records the gathered forecast event values after {@link #gatherEventValues(int)} is called. It only stores the
     * most recent results (one day at a time).
     */
    private final List<Float>[] _gatheredFcstEventValues;

    /**
     * Records the T0s corresponding to {@link #_gatheredFcstEventValues}. This is generally only useful for interactive
     * viewing of the event values, since the T0s are thrown out after gathering during parameter estimation.
     */
    private final List<Long>[] _gatheredFcstT0s;

    /**
     * Records the gathered observed event values after {@link #gatherEventValues(int)} is called. It only stores the
     * most recent results (one day at a time).
     */
    private final List<Float>[] _gatheredObsEventValues;

    /**
     * Records if enough events were found based on {@link #_listAcceptanceCheck}.
     */
    private final boolean[] _enoughEventsFlags;

    /**
     * Records the last window widths used for each event; the one that yielded enough events or the maximum window.
     */
    private final int[] _windowWidthsUsed; //I don't know if we'll need this in the future or not.

    private final Calendar _workingCalendar = Calendar.getInstance(TimeZone.getTimeZone("GMT"));

    private int _workingEventNumber;

    /**
     * This calls
     * {@link #CanonicalEventValuesGatherer(CanonicalEventValues, CanonicalEventValues, BinaryPredicate, int, int, int, int)}
     * passing in the fixed window width provided as both the starting and ending window width.
     * 
     * @param fixedWindowWidth The fixed window width to use. Both {@link #_startingWindowWidth} and
     *            {@link #_maximumWindowWidth} are set to this value and the {@link #gatherEventValues(int)} method
     *            exits immediately after first iteration of the window width.
     */
    public CanonicalEventValuesGatherer(final CanonicalEventValues forecastValues,
                                        final CanonicalEventValues observedValues,
                                        final BinaryPredicate<List<Float>, List<Float>> listAcceptanceChecker,
                                        final int fixedWindowWidth,
                                        final int firstYear,
                                        final int lastYear)
    {
        this(forecastValues,
             observedValues,
             listAcceptanceChecker,
             fixedWindowWidth,
             fixedWindowWidth,
             firstYear,
             lastYear);
    }

    /**
     * This calls
     * {@link #CanonicalEventValuesGatherer(CanonicalEventValues, CanonicalEventValues, BinaryPredicate, int, int, int, int, int)}
     * passing in 1 as the days between T0s.
     */
    public CanonicalEventValuesGatherer(final CanonicalEventValues forecastValues,
                                        final CanonicalEventValues observedValues,
                                        final BinaryPredicate<List<Float>, List<Float>> listAcceptanceChecker,
                                        final int startingWindowWidth,
                                        final int maximumWindowWidth,
                                        final int firstYear,
                                        final int lastYear)
    {
        this(forecastValues,
             observedValues,
             listAcceptanceChecker,
             startingWindowWidth,
             maximumWindowWidth,
             firstYear,
             lastYear,
             1);
    }

    /**
     * The {@link CanonicalEventValues#getEventsToCompute()} returned list will be recorded herein for forecast values
     * after making sure it matches the observed values.
     * 
     * @param forecastValues The canonical event values computed based on forecasts.
     * @param observedValues The canonical event values computed based on observations. The call
     *            {@link CanonicalEventValues#getComputationalTimes()} must match that returned for forecastValues.
     * @param listAcceptanceChecker {@link BinaryPredicate} that returns true if the list canonical event values for
     *            forecasts and observed satisfies conditions required to be used for parameter estimation.
     * @param startingWindowWidth The initial, minimum window width to use.
     * @param maximumWindowWidth The maximum allowable window width. The window grows 5 days at a time.
     * @param firstYear The first year for which to gather events. Pass in -1 to provide an unlimited lower bound.
     * @param lastYear The last year for which to gather events. Pass in -1 to provide an unlimited upper bound.
     * @param daysBetweenTOs The number of days between T0s for which you want to include reforecasts.
     */
    public CanonicalEventValuesGatherer(final CanonicalEventValues forecastValues,
                                        final CanonicalEventValues observedValues,
                                        final BinaryPredicate<List<Float>, List<Float>> listAcceptanceChecker,
                                        final int startingWindowWidth,
                                        final int maximumWindowWidth,
                                        final int firstYear,
                                        final int lastYear,
                                        final int daysBetweenT0s)
    {
        if((startingWindowWidth < 0) || (maximumWindowWidth < 0))
        {
            throw new IllegalArgumentException("The window width arguments cannot be negative.");
        }
        if(!forecastValues.getEventsToCompute().equals(observedValues.getEventsToCompute()))
        {
            throw new IllegalArgumentException("The forecast and observed events to compute are not identical.");
        }
        if(forecastValues.isEmpty())
        {
            throw new IllegalArgumentException("There are no forecast canonical event values to gather (list is empty).");
        }
        if(observedValues.isEmpty())
        {
            throw new IllegalArgumentException("There are no observed canonical event values to gather (list is empty).");
        }
        if(!forecastValues.getComputationalTimes().equals(observedValues.getComputationalTimes()))
        {
            throw new IllegalArgumentException("The observed values were not computed for the same forecast times as the forecast values.");
        }
        _events = forecastValues.getEventsToCompute();

        //Create used lists for the forecast and event values.
        //XXX New code created for the GEFS sensitivity analysis.
        CanonicalEventValues usedFcstValues = forecastValues;
        if(daysBetweenT0s > 1)
        {
            usedFcstValues = new CanonicalEventValues(_events);
            for(long workingTime =
                                 forecastValues.getFirstComputationalTime(); workingTime <= forecastValues.getLastcomputationalTime(); workingTime +=
                                                                                                                                                   daysBetweenT0s
                                                                                                                                                       * 24
                                                                                                                                                       * HCalendar.MILLIS_IN_HR)
            {
                final float[] values = forecastValues.getEventValues(workingTime);
                if(values != null)
                {
                    usedFcstValues.putEventValues(workingTime, values);
                }
            }
        }

        //Prepare storage objects for the values found for each event.
        _gatheredFcstEventValues = new ArrayList[_events.size()];
        _gatheredFcstT0s = new ArrayList[_events.size()];
        _gatheredObsEventValues = new ArrayList[_events.size()];
        for(int i = 0; i < _events.size(); i++)
        {
            _gatheredFcstEventValues[i] = Lists.newArrayList();
            _gatheredFcstT0s[i] = Lists.newArrayList();
            _gatheredObsEventValues[i] = Lists.newArrayList();
        }
        _enoughEventsFlags = new boolean[this._events.size()];
        _windowWidthsUsed = new int[_events.size()];

        //Populate.
        _listAcceptanceCheck = listAcceptanceChecker;
        _startingWindowWidth = startingWindowWidth;
        _maximumWindowWidth = maximumWindowWidth;
        populateDayOfYearMaps(usedFcstValues, observedValues, firstYear, lastYear);
    }

    protected void setListAcceptanceCheck(final BinaryPredicate<List<Float>, List<Float>> predicate)
    {
        _listAcceptanceCheck = predicate;
    }

    /**
     * Populate {@link #_dayOfYearToFcstValuesMap} and {@link #_dayOfYearToObsValuesMap} based on the given
     * {@link CanonicalEventValues} instances. Both maps need to contain event values for forecast times that overlap.
     * Note that this method does some argument checking to make sure that the events computed are identical for both
     * arguments and that neither list is empty.
     * 
     * @param forecastValues The canonical event values computed based on forecasts.
     * @param observedValues The canonical event values computed based on observations.
     * @param firstYear The first year for which to gather events. Pass in -1 to provide an unlimited lower bound.
     * @param lastYear The last year for which to gather events. Pass in -1 to provide an unlimited upper bound.
     */
    private void populateDayOfYearMaps(final CanonicalEventValues forecastValues,
                                       final CanonicalEventValues observedValues,
                                       final int firstYear,
                                       final int lastYear)
    {
        //Loop through the forecast times one at a time.  The observed times are advanced in order
        //to keep up with the forecast times.  
        final Iterator<Long> observedTimesIter = forecastValues.getComputationalTimes().iterator();
        long obsTime = observedTimesIter.next();
        boolean atLeastOneEventValueGathered = false;
        for(final Long fcstTime: forecastValues.getComputationalTimes())
        {
            //Iterate the obsTime until it equals or exceeds forecast time.  If it cannot be iterated
            //enough before running out, break out of the loop... we are done.  In reality, this should
            //not be necessary since both the forecast value times and observed value times should
            //be identical.
            while((obsTime < fcstTime) && (observedTimesIter.hasNext()))
            {
                obsTime = observedTimesIter.next();
            }
            if(obsTime < fcstTime)
            {
                break;
            }
            if(obsTime == fcstTime)
            {
                //Only add to the map if the forecast time is within the specified years.
                final int year = HCalendar.computeCalendarFromMilliseconds(fcstTime).get(Calendar.YEAR);
                if(((firstYear < 0) || (year >= firstYear)) && ((lastYear < 0) || (year <= lastYear)))
                {
                    atLeastOneEventValueGathered = true;
                    addToDayOfYearMaps(fcstTime,
                                       forecastValues.getEventValues(fcstTime),
                                       observedValues.getEventValues(obsTime));
                }
            }
        }
        if(!atLeastOneEventValueGathered)
        {
            throw new IllegalArgumentException("From the provided forecast and observed data, and given the initial year of "
                + firstYear + " and last year of " + lastYear
                + ", no event values could be gathered.  It maybe that the initial and last year define a time period for which there are no forecasts.");
        }
    }

    /**
     * Records the values in the {@link #_dayOfYearToAllValuesMap}. Note that any day that shows up as the 366th day of
     * year is not used. Based on the code chkwin.f in epp3_precip_parms/epp3_temp_parms, the 366th day never results in
     * a return of 1 to include the value, because looping occurs at the 365th day.
     * 
     * @param forecastTime The forecast time in millis
     * @param eventValues the event values.
     */
    private void addToDayOfYearMaps(final long forecastTime, final float[] fcstValues, final float[] obsValues)
    {
        _workingCalendar.setTimeInMillis(forecastTime);
        final int dayOfYear = _workingCalendar.get(Calendar.DAY_OF_YEAR);
        _dayOfYearToFcstValuesMap.put(dayOfYear, fcstValues);

        //All values were computed from the same forecast time, but I still need one per value.
        final long[] t0s = new long[fcstValues.length];
        Arrays.fill(t0s, forecastTime);
        _dayOfYearToT0sMap.put(dayOfYear, t0s);

        _dayOfYearToObsValuesMap.put(dayOfYear, obsValues);
    }

    /**
     * @param dayOfYear Following {@link Calendar} day-of-year, counting starts at 1.
     * @return List of arrays, each of which specifies the canonical event values for one forecast time. The list will
     *         contain the values across all instances of the specified day-of-year.
     */
    private List<float[]> getFcstEventValues(final int dayOfYear)
    {
        return this._dayOfYearToFcstValuesMap.get(dayOfYear);
    }

    private List<long[]> getFcstT0s(final int dayOfYear)
    {
        return this._dayOfYearToT0sMap.get(dayOfYear);
    }

    /**
     * @param dayOfYear Following {@link Calendar} day-of-year, counting starts at 1.
     * @return List of arrays, each of which specifies the canonical event values for one forecast time. The list will
     *         contain the values across all instances of the specified day-of-year.
     */
    private List<float[]> getObsEventValues(final int dayOfYear)
    {
        return this._dayOfYearToObsValuesMap.get(dayOfYear);
    }

    /**
     * See chkwin Fortran code. The bounds are determined based on a 365 day year.
     * 
     * @param dayOfYear Base day of year.
     * @param windowWidth The width of the window, diameter.
     * @return The lower bound of the window, which is the day of year minus one half the window width, round down.
     */
    private int determineWindowLowerBound(final int dayOfYear, final int windowWidth)
    {
        int bound;
        bound = dayOfYear - (windowWidth / 2);
        if(bound <= 0)
        {
            bound += 365;
        }
        return bound;
    }

    /**
     * See chkwin Fortran code. The bounds are determined based on a 365 day year.
     * 
     * @param dayOfYear Base day of year.
     * @param windowWidth The width of the window, diameter.
     * @return The upper bound of the window, which is the day of year plus one half the window width, round down.
     */
    private int determineWindowUpperBound(final int dayOfYear, final int windowWidth)
    {
        int bound;
        bound = dayOfYear + (windowWidth / 2);
        if(bound > 365)
        {
            bound -= 365;
        }
        return bound;
    }

    /**
     * Adds to the {@link #_gatheredFcstEventValues} and {@link #_gatheredObsEventValues} lists, keeping both in synch
     * so that forecasts and corresponding observed have the same index. Events are only added if the number is not
     * missing (i.e., NaN), and if the {@link #_enoughEventsFlags} flag for the event is false.
     */
    private void appendValues(final float[] fcstCanonicalEventValues,
                              final float[] obsCanonicalEventValues,
                              final long[] fcstT0s)
    {
        for(int i = 0; i < fcstCanonicalEventValues.length; i++)
        {
            if((!Float.isNaN(fcstCanonicalEventValues[i])) && (!Float.isNaN(obsCanonicalEventValues[i])))
            {
                if(!_enoughEventsFlags[i])
                {
                    _gatheredFcstEventValues[i].add(fcstCanonicalEventValues[i]);
                    if(fcstT0s != null)
                    {
                        _gatheredFcstT0s[i].add(fcstT0s[i]);
                    }
                    _gatheredObsEventValues[i].add(obsCanonicalEventValues[i]);
                }
            }
        }
    }

    /**
     * Gathers all of the event values for all days making them accessible to outside callers through the standard
     * {@link #getForecastGatheredEventValues(CanonicalEvent)} and
     * {@link #getObservedGatheredEventValues(CanonicalEvent)} methods.
     */
    public void gatherEventValuesForAllDays()
    {
        clearGatheredEventValues();
        appendValuesForAllDays(1, 365);
    }

    /**
     * Appends the canonical events for all of the specified days of the the year to the
     * {@link #_gatheredFcstEventValues} and {@link #_gatheredObsEventValues}. Either day must be between 1 and 365,
     * inclusive.
     * 
     * @param lowerBoundDay The first day of the year for which to add canonical event values. Jan 1 = 1.
     * @param upperBoundDay The last day of the year.
     */
    private void appendValuesForAllDays(final int lowerBoundDay, final int upperBoundDay)
    {
        //Values to add stores, for each day, an array of canonical event values the same length as _events.size().
        List<float[]> fcstValuesToAdd = null;
        List<float[]> obsValuesToAdd = null;
        List<long[]> fcstT0sToAdd = null;

        //Start with the lowerBoundDay, loop to the upper bound adding events to _gatheredFcstEvents and
        //_gatheredObsEvents.  Note that, conceptually, this is best done as a while loop, since the day 
        //of year must loop when it hits 366.
        int dayOfYear = lowerBoundDay;
        boolean afterUpperBound = false;
        while(!afterUpperBound)
        {
            //Get the values to add.  Then loop based on the fcstValuesToAdd, calling append for both.
            //The values must be appended simultaneously to keep all lists in sync (fcst with corresponding obs).
            fcstValuesToAdd = getFcstEventValues(dayOfYear);
            obsValuesToAdd = getObsEventValues(dayOfYear);
            fcstT0sToAdd = getFcstT0s(dayOfYear);
            for(int i = 0; i < fcstValuesToAdd.size(); i++)
            {
                if(fcstT0sToAdd != null)
                {
                    appendValues(fcstValuesToAdd.get(i), obsValuesToAdd.get(i), fcstT0sToAdd.get(i));
                }
                else
                {
                    appendValues(fcstValuesToAdd.get(i), obsValuesToAdd.get(i), null);
                }
            }

            if(dayOfYear == upperBoundDay)
            {
                //Stop the loop
                afterUpperBound = true;
            }
            else
            {
                //Increment day of year and loop it if necessary.  Note that day 366 is included in this loop,
                //even though the bounds are computed based on 365 days.
                dayOfYear++;
                if(dayOfYear == 367)
                {
                    dayOfYear = 1;
                }
            }
        }
    }

    public void clearGatheredEventValues()
    {
        for(int i = 0; i < _events.size(); i++)
        {
            _gatheredFcstEventValues[i].clear();
            _gatheredObsEventValues[i].clear();
        }
    }

    /**
     * @return The event number the most recent time that the {@link #_listAcceptanceCheck} apply method was called.
     */
    protected int getWorkingEventNumber()
    {
        return _workingEventNumber;
    }

    public int getMaximumWindowWidth()
    {
        return _maximumWindowWidth;
    }

    /**
     * @return The events for which event values are being gathered.
     */
    public CanonicalEventList getEvents()
    {
        return _events;
    }

    /**
     * Entry point for the data gathering algorithm. After calling this, the {@link #_gatheredFcstEventValues},
     * {@link #_gatheredObsEventValues} and {@link #_enoughEventsFlags} will be populated. Call the get methods to
     * acquire the gathered values and determine if enough values were gathered base on
     * {@link #_listAcceptanceCheck}.<br>
     * <br>
     * NOTE: The bounds are determined based on a 365 day year. This means that when the 366th day of the year is
     * between the bounds, it will be included, and the actual window width for that year will be one larger than it
     * should be (because it includes a 366th day).
     * 
     * @param dayOfYear Day of the year for which to gather values. Counting starts at 1 for Jan 1.
     */
    public void gatherEventValues(final int dayOfYear)
    {
        clearGatheredEventValues();

        int currentWindowWidth = _startingWindowWidth;
        int windowLB = determineWindowLowerBound(dayOfYear, currentWindowWidth);
        int windowUB = determineWindowUpperBound(dayOfYear, currentWindowWidth);
        if(windowUB == windowLB) //Only happens if currentWindowWidth is 0.
        {
            windowUB = -1; //Indicate it should not be used the first time to gather values.
        }
        int previousLB = dayOfYear;
        int previousUB = dayOfYear + 1; //Must never equal previousLB!

        Arrays.fill(_enoughEventsFlags, false);
        Arrays.fill(_windowWidthsUsed, -1);

        boolean allEventsDone = false;
        while(!allEventsDone)
        {
            //Gather the event values.  If windowUB is negative, it indicates that the ub and lb are equal, so, to
            //avoid gathering the same data twice, we do not append values for it.  It also indicates that it must be 
            //the first time through the loop.
            appendValuesForAllDays(windowLB, previousLB);
            if(windowUB > 0)
            {
                appendValuesForAllDays(previousUB, windowUB);
            }
            else
            {
                windowUB = dayOfYear + 1; //This is done so that the window expansion below works.
            }

            //Check if enough based on predicate
            allEventsDone = true;
            for(int eventNumber = 0; eventNumber < _events.size(); eventNumber++)
            {
                if(!_enoughEventsFlags[eventNumber])
                {
                    _workingEventNumber = eventNumber; //In case a subclass needs this!
                    _enoughEventsFlags[eventNumber] = _listAcceptanceCheck.apply(_gatheredFcstEventValues[eventNumber],
                                                                                 _gatheredObsEventValues[eventNumber]);
                    allEventsDone = allEventsDone && _enoughEventsFlags[eventNumber];
                    if(_enoughEventsFlags[eventNumber])
                    {
                        _windowWidthsUsed[eventNumber] = currentWindowWidth;
                    }
                }
            }

            //Exit the loop immediately if the starting and max window widths are equal; i.e., a fixed window width is 
            //used.  This will be the case for CFSv2 and some RFC testing
            if(_startingWindowWidth == _maximumWindowWidth)
            {
                break;
            }

            //Expand window.  If the new width is too large, break out: we are done gathering.
            currentWindowWidth += 5;
            if(currentWindowWidth > _maximumWindowWidth)
            {
                break;
            }

            //Set the previous values so that the append methods will pick the right days with no repeats.
            //Specifically, previousLB is one before windowLB, and previousUB is one after windowUB. 
            previousLB = windowLB - 1;
            if(previousLB == 0)
            {
                previousLB = 366;
            }
            previousUB = windowUB + 1;
            if(previousUB == 367)
            {
                previousUB = 1;
            }

            windowLB = determineWindowLowerBound(dayOfYear, currentWindowWidth);
            windowUB = determineWindowUpperBound(dayOfYear, currentWindowWidth);
        }

        //TEST OUTPUT!!!
//        final List<Float> outputFcstValues = _gatheredFcstEventValues[0];
//        final List<Float> outputObsValues = _gatheredObsEventValues[0];
//        for(int i = 0; i < outputFcstValues.size(); i++)
//        {
//            System.err.println("####>> " + outputFcstValues.get(i) + "  " + outputObsValues.get(i));
//        }
//        System.err.println("####>> COUNT ------------ " + outputFcstValues.size());
    }

    public List<Long> getForecastT0s(final CanonicalEvent evt)
    {
        return getForecastT0s(_events.indexOf(evt));
    }

    public List<Long> getForecastT0s(final int evtIndex)
    {
        return this._gatheredFcstT0s[evtIndex];
    }

    public List<Float> getForecastGatheredEventValues(final CanonicalEvent evt)
    {
        return getForecastGatheredEventValues(_events.indexOf(evt));
    }

    public List<Float> getForecastGatheredEventValues(final int evtIndex)
    {
        return this._gatheredFcstEventValues[evtIndex];
    }

    public List<Float> getObservedGatheredEventValues(final CanonicalEvent evt)
    {
        return getObservedGatheredEventValues(_events.indexOf(evt));
    }

    public List<Float> getObservedGatheredEventValues(final int evtIndex)
    {
        return this._gatheredObsEventValues[evtIndex];
    }

    public boolean wereEnoughValuesGathered(final CanonicalEvent evt)
    {
        return wereEnoughValuesGathered(_events.indexOf(evt));
    }

    public boolean wereEnoughValuesGathered(final int evtIndex)
    {
        return this._enoughEventsFlags[evtIndex];
    }

    public int getWindowWidthUsedForGatheredValues(final CanonicalEvent evt)
    {
        return getWindowWidthUsedForGatheredValues(_events.indexOf(evt));
    }

    public int getWindowWidthUsedForGatheredValues(final int evtIndex)
    {
        return this._windowWidthsUsed[evtIndex];
    }
}
