package ohd.hseb.hefs.mefp.tools.canonical;

import java.io.File;

import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.notice.CanonicalEventsCopiedNotice;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.notify.NoticeForwarder;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.xml.CompositeXMLReader;
import ohd.hseb.hefs.utils.xml.CompositeXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReadable;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLWritable;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.base.Objects;

/**
 * Manages the {@link CanonicalEvent}s, not the values, and is used for reading/writing, storing, and allowing users to
 * edit the events. It keeps lists of the precipitation and temperature events independently.
 * 
 * @author Hank.Herr
 */
public class CanonicalEventsManager extends NoticeForwarder implements XMLReadable, XMLWritable
{
    private static final Logger LOG = LogManager.getLogger(CanonicalEventsManager.class);

    /**
     * Necessary for event posting purposes.
     */
    private final MEFPParameterEstimatorRunInfo _runInfo;

    private final File _baseDirectory;
    @SuppressWarnings("unused")
    private final File _systemFilesDirectory;
    private CanonicalEventList _precipitationBaseCanonicalEventList = new CanonicalEventList(true);
    private CanonicalEventList _precipitationModulationCanonicalEventList = new CanonicalEventList(true);
    private CanonicalEventList _temperatureBaseCanonicalEventList = new CanonicalEventList(false);
    private CanonicalEventList _temperatureModulationCanonicalEventList = new CanonicalEventList(false);

    public CanonicalEventsManager(final File baseDirectory,
                                  final File systemFilesDirectory,
                                  final MEFPParameterEstimatorRunInfo runInfo) throws Exception
    {
        _runInfo = runInfo;
        _precipitationBaseCanonicalEventList.setXMLTagName("precipitatinBaseCanonicalEventList");
        _precipitationModulationCanonicalEventList.setXMLTagName("precipitationModulationCanonicalEventList");
        _temperatureBaseCanonicalEventList.setXMLTagName("temperatureBaseCanonicalEventList");
        _temperatureModulationCanonicalEventList.setXMLTagName("temperatureModulationCanonicalEventList");

        _baseDirectory = baseDirectory;
        _systemFilesDirectory = systemFilesDirectory;
    }

    /**
     * Constructor for testing purposes.
     * 
     * @param precipBase array of precip base events
     * @param precipMod array of precip mod events
     * @param tempBase array of temp base events
     * @param tempMod array of temp mod events
     */
    public CanonicalEventsManager(final CanonicalEvent[] precipBase,
                                  final CanonicalEvent[] precipMod,
                                  final CanonicalEvent[] tempBase,
                                  final CanonicalEvent[] tempMod)
    {
        _runInfo = null;
        _precipitationBaseCanonicalEventList.addAll(precipBase);
        _precipitationModulationCanonicalEventList.addAll(precipMod);
        _temperatureBaseCanonicalEventList.addAll(tempBase);
        _temperatureModulationCanonicalEventList.addAll(tempMod);
        _baseDirectory = null;
        _systemFilesDirectory = null;
    }

    /**
     * Populate the {@link CanonicalEventList} instances in this based on those provided in the argument. This will NOT
     * modify {@link #_baseDirectory} or {@link #_systemFilesDirectory}.
     * 
     * @param type The {@link Type} of data for which to copy the events. Pass in null to copy all.
     */
    public void copyEvents(final CanonicalEventsManager other, final ParameterId.Type type)
    {
        if(type == ParameterId.Type.PRECIPITATION || type == null)
        {
            _precipitationBaseCanonicalEventList.clear();
            _precipitationBaseCanonicalEventList.addAll(other.getPrecipitationBaseCanonicalEventList());

            _precipitationModulationCanonicalEventList.clear();
            _precipitationModulationCanonicalEventList.addAll(other.getPrecipitationModulationCanonicalEventList());

            //Post the copied event.
            if(_runInfo != null)
            {
                _runInfo.post(new CanonicalEventsCopiedNotice(this,
                                                              other.getPrecipitationBaseCanonicalEventList(),
                                                              other.getPrecipitationModulationCanonicalEventList()));
            }
        }

        if(type == ParameterId.Type.TEMPERATURE || type == null)
        {
            _temperatureBaseCanonicalEventList.clear();
            _temperatureBaseCanonicalEventList.addAll(other.getTemperatureBaseCanonicalEventList());

            _temperatureModulationCanonicalEventList.clear();
            _temperatureModulationCanonicalEventList.addAll(other.getTemperatureModulationCanonicalEventList());

            //Post the copied event.
            if(_runInfo != null)
            {
                _runInfo.post(new CanonicalEventsCopiedNotice(this,
                                                              other.getTemperatureBaseCanonicalEventList(),
                                                              other.getTemperatureModulationCanonicalEventList()));
            }
        }
    }

    /**
     * Given the provided events, it populates the appropriate attribute based on its
     * {@link CanonicalEventList#isBaseEventType()}, {@link CanonicalEventList#isModulationEventType()}, and
     * {@link CanonicalEventList#isPrecipitation()}.
     */
    public void copyEvents(final CanonicalEventList events)
    {
        if((events.isBaseEventType()) && (events.isPrecipitation()))
        {
            _precipitationBaseCanonicalEventList.clear();
            _precipitationBaseCanonicalEventList.addAll(events);
        }
        else if((events.isModulationEventType()) && (events.isPrecipitation()))
        {
            _precipitationModulationCanonicalEventList.clear();
            _precipitationModulationCanonicalEventList.addAll(events);
        }
        else if((events.isBaseEventType()) && (!events.isPrecipitation()))
        {
            _temperatureBaseCanonicalEventList.clear();
            _temperatureBaseCanonicalEventList.addAll(events);
        }
        else if((events.isModulationEventType()) && (!events.isPrecipitation()))
        {
            _temperatureModulationCanonicalEventList.clear();
            _temperatureModulationCanonicalEventList.addAll(events);
        }

        //Post the copied event.
        if(_runInfo != null)
        {
            _runInfo.post(new CanonicalEventsCopiedNotice(this, events));
        }
    }

    /**
     * This method will import files from the designated import directory under _baseDirectory. It will attempt to find
     * four canonical events files. Each one found will be imported. If a file is not found, then the current events for
     * that data type and canonical event type (base or modulation) are unchanged. If a file cannot be processed due to
     * an incorrect format, importing will stop and NO events will be imported; an {@link Exception} will be thrown, but
     * ALL four files will still be removed.
     * 
     * @throws Exception
     */
    public void loadFromImportFiles() throws Exception
    {
        final CanonicalEventFileHandler handler = new CanonicalEventFileHandler(new File(_baseDirectory.getAbsolutePath()
            + "/import"));

        if(handler.doAnyASCIICanonicalEventsFilesExist())
        {
            LOG.info("Import canonical events files found in " + handler.getBaseDirectory() + ".  Loading...");

            //Null returned lists are okay, and are ignored.  However, if any of the load calls results in an
            //exception, still call the remove method below (see finally) but do NOT use ANY of the events loaded.
            //Its okay to leave files out, but not okay to have an improper file.
            try
            {
                final CanonicalEventList precipitationBaseCanonicalEventList = handler.loadBaseEvents(true);
                final CanonicalEventList precipitationModulationCanonicalEventList = handler.loadModulationEvents(true);
                final CanonicalEventList temperatureBaseCanonicalEventList = handler.loadBaseEvents(false);
                final CanonicalEventList temperatureModulationCanonicalEventList = handler.loadModulationEvents(false);

                if(precipitationBaseCanonicalEventList != null)
                {
                    _precipitationBaseCanonicalEventList = precipitationBaseCanonicalEventList;
                }
                if(precipitationModulationCanonicalEventList != null)
                {
                    _precipitationModulationCanonicalEventList = precipitationModulationCanonicalEventList;
                }
                if(temperatureBaseCanonicalEventList != null)
                {
                    _temperatureBaseCanonicalEventList = temperatureBaseCanonicalEventList;
                }
                if(temperatureModulationCanonicalEventList != null)
                {
                    _temperatureModulationCanonicalEventList = temperatureModulationCanonicalEventList;
                }
            }
            //ALWAYS remove the four files.
            finally
            {
                handler.removeAllASCIICanonicalEventsFiles();
            }

            //Reset the tags because the handler does not set them correctly.
            _precipitationBaseCanonicalEventList.setXMLTagName("precipitatinBaseCanonicalEventList");
            _precipitationModulationCanonicalEventList.setXMLTagName("precipitationModulationCanonicalEventList");
            _temperatureBaseCanonicalEventList.setXMLTagName("temperatureBaseCanonicalEventList");
            _temperatureModulationCanonicalEventList.setXMLTagName("temperatureModulationCanonicalEventList");

            LOG.info("Import canonical events files were loaded successfully and have been removed.");
        }
        //TESTING!!!
        //XMLTools.createXMLFileFromXMLWriter(new File("testdata/canonicalEventsManager/benchmark.test1_output.xml"),
        //                                    this,
        //                                    true);
    }

    public CanonicalEventList getEventList(final CanonicalEventType eventType, final ParameterId.Type dataType)
    {
        if(eventType == CanonicalEventType.BASE)
        {
            if(dataType == ParameterId.Type.PRECIPITATION)
            {
                return getPrecipitationBaseCanonicalEventList();
            }
            else if(dataType == ParameterId.Type.TEMPERATURE)
            {
                return getTemperatureBaseCanonicalEventList();
            }
        }
        else if(eventType == CanonicalEventType.MODULATION)
        {
            if(dataType == ParameterId.Type.PRECIPITATION)
            {
                return getPrecipitationModulationCanonicalEventList();
            }
            else if(dataType == ParameterId.Type.TEMPERATURE)
            {
                return getTemperatureModulationCanonicalEventList();
            }
        }
        return null;
    }

    public CanonicalEventList getBaseCanonicalEventList(final LocationAndDataTypeIdentifier identifier)
    {
        if(identifier.isPrecipitationDataType())
        {
            return this.getPrecipitationBaseCanonicalEventList();
        }
        else if(identifier.isTemperatureDataType())
        {
            return this.getTemperatureBaseCanonicalEventList();
        }
        return null;
    }

    public CanonicalEventList getModulationCanonicalEventList(final LocationAndDataTypeIdentifier identifier)
    {
        if(identifier.isPrecipitationDataType())
        {
            return this.getPrecipitationModulationCanonicalEventList();
        }
        else if(identifier.isTemperatureDataType())
        {
            return this.getTemperatureModulationCanonicalEventList();
        }
        return null;
    }

    public CanonicalEventList getPrecipitationBaseCanonicalEventList()
    {
        return _precipitationBaseCanonicalEventList;
    }

    public CanonicalEventList getPrecipitationModulationCanonicalEventList()
    {
        return _precipitationModulationCanonicalEventList;
    }

    public CanonicalEventList getTemperatureBaseCanonicalEventList()
    {
        return _temperatureBaseCanonicalEventList;
    }

    public CanonicalEventList getTemperatureModulationCanonicalEventList()
    {
        return _temperatureModulationCanonicalEventList;
    }

    /**
     * @return A {@link CanonicalEventList} containing both base and modulation precipitation events in sorted order.
     */
    public CanonicalEventList buildFullPrecipitationCanonicalEventList()
    {
        return buildFullCanonicalEventList(true,
                                           getPrecipitationBaseCanonicalEventList(),
                                           getPrecipitationModulationCanonicalEventList());
    }

    /**
     * @return A {@link CanonicalEventList} containing both base and modulation temperature events in sorted order.
     */
    public CanonicalEventList buildFullTemperatureCanonicalEventList()
    {
        return buildFullCanonicalEventList(false,
                                           getTemperatureBaseCanonicalEventList(),
                                           getTemperatureModulationCanonicalEventList());
    }

    /**
     * @return A CanonicalEventList containing both the base and modulation events for the provided identifier. It calls
     *         either {@link #buildFullPrecipitationCanonicalEventList()} or
     *         {@link #buildFullTemperatureCanonicalEventList()}.
     */
    public CanonicalEventList buildFullCanonicalEventList(final LocationAndDataTypeIdentifier identifier)
    {
        if(identifier.isPrecipitationDataType())
        {
            return buildFullPrecipitationCanonicalEventList();
        }
        else if(identifier.isTemperatureDataType())
        {
            return buildFullTemperatureCanonicalEventList();
        }
        throw new IllegalArgumentException("Provided identifier must be either precip or temp data.");
    }

    private CanonicalEventList buildFullCanonicalEventList(final boolean precipitation,
                                                           final CanonicalEventList base,
                                                           final CanonicalEventList mod)
    {
        //NOTE: CanonicalEventList is a SortedCollection, so sorting should not be needed later.
        final CanonicalEventList fullSet = new CanonicalEventList(precipitation);
        for(final CanonicalEvent event: base)
        {
            fullSet.add(event);
        }
        for(final CanonicalEvent event: mod)
        {
            fullSet.add(event);
        }
        return fullSet;
    }

    @Override
    public String toString()
    {
        String results = "";
        results = "CanonicalEventsManager: _precipitationBaseCanonicalEventList = "
            + _precipitationBaseCanonicalEventList.toString() + "; ";
        results += "_precipitationModulationCanonicalEventList = "
            + _precipitationModulationCanonicalEventList.toString() + "; ";
        results += "_temperatureBaseCanonicalEventList = " + _temperatureBaseCanonicalEventList.toString() + "; ";
        results += "_temperatureModulationCanonicalEventList = " + _temperatureModulationCanonicalEventList.toString()
            + ".";
        return results;
    }

    public String getXMLTagName()
    {
        return "allCanonicalEvents";
    }

    /**
     * @return A {@link CompositeXMLWriter} that outputs only the precipitation event lists using the standard top level
     *         tag, {@link #getXMLTagName()}.
     */
    public CompositeXMLWriter getWriterForPrecipitationOnly()
    {
        return new CompositeXMLWriter(getXMLTagName(),
                                      _precipitationBaseCanonicalEventList,
                                      _precipitationModulationCanonicalEventList);

    }

    /**
     * @return A {@link CompositeXMLWriter} that outputs only the temperature event lists using the standard top level
     *         tag, {@link #getXMLTagName()}.
     */
    public CompositeXMLWriter getWriterForTemperatureOnly()
    {
        return new CompositeXMLWriter(getXMLTagName(),
                                      _temperatureBaseCanonicalEventList,
                                      _temperatureModulationCanonicalEventList);
    }

    /**
     * @return A {@link CompositeXMLReader} that goes with {@link #getWriterForPrecipitationOnly()} and populates only
     *         {@link #_precipitationBaseCanonicalEventList} and {@link #_precipitationModulationCanonicalEventList}.
     */
    public CompositeXMLReader getCompatibleReaderForSingleDataType(final CanonicalEventList baseEvents,
                                                                   final CanonicalEventList modEvents,
                                                                   final ParameterId.Type type)
    {
        baseEvents.setXMLTagName(getEventList(CanonicalEventType.BASE, type).getXMLTagName());
        modEvents.setXMLTagName(getEventList(CanonicalEventType.MODULATION, type).getXMLTagName());
        return new CompositeXMLReader(getXMLTagName(), baseEvents, modEvents)
        {
            @Override
            public void finalizeReading() throws XMLReaderException
            {
                baseEvents.setToBaseEventType();
                modEvents.setToModulationEventType();
            }
        };
    }

    @Override
    public CompositeXMLWriter getWriter()
    {
        return new CompositeXMLWriter(getXMLTagName(),
                                      _precipitationBaseCanonicalEventList,
                                      _precipitationModulationCanonicalEventList,
                                      _temperatureBaseCanonicalEventList,
                                      _temperatureModulationCanonicalEventList);
    }

    @Override
    public CompositeXMLReader getReader()
    {
        return new CompositeXMLReader(getXMLTagName(),
                                      _precipitationBaseCanonicalEventList,
                                      _precipitationModulationCanonicalEventList,
                                      _temperatureBaseCanonicalEventList,
                                      _temperatureModulationCanonicalEventList)
        {
            @Override
            public void finalizeReading() throws XMLReaderException
            {
                _precipitationBaseCanonicalEventList.setToBaseEventType();
                _precipitationModulationCanonicalEventList.setToModulationEventType();
                _temperatureBaseCanonicalEventList.setToBaseEventType();
                _temperatureModulationCanonicalEventList.setToModulationEventType();
            }
        };
    }

    /**
     * Only looks at contents, not assigned directory.
     */
    @Override
    public boolean equals(final Object other)
    {
        if(!(other instanceof CanonicalEventsManager))
        {
            return false;
        }
        final CanonicalEventsManager that = (CanonicalEventsManager)other;

        return this._precipitationBaseCanonicalEventList.equals(that._precipitationBaseCanonicalEventList)
            && this._precipitationModulationCanonicalEventList.equals(that._precipitationModulationCanonicalEventList)
            && this._temperatureBaseCanonicalEventList.equals(that._temperatureBaseCanonicalEventList)
            && this._temperatureModulationCanonicalEventList.equals(that._temperatureModulationCanonicalEventList);
    }

    @Override
    public int hashCode()
    {
        return Objects.hashCode(_precipitationBaseCanonicalEventList,
                                _precipitationModulationCanonicalEventList,
                                _temperatureBaseCanonicalEventList,
                                _temperatureModulationCanonicalEventList);
    }
}
