package ohd.hseb.hefs.mefp.tools.canonical;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.TimeZone;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.pe.sources.SourceDataHandler;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.datetime.DateTools;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLWriter;
import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.HStopWatch;

/**
 * Records observed and forecast {@link CanonicalEventValues} instances. Provides set/get methods for putting values
 * into the {@link CanonicalEventValues} instances and getting values out. Also provides methods to get
 * {@link XMLReader} and {@link XMLWriter} instances. Finally, three methods are provided for computing the event values
 * given different arguments.
 * 
 * @author Hank.Herr
 */
public class SourceCanonicalEventValues
{
    private final MEFPForecastSource _forecastSource;
    private final CanonicalEventValues _observedValues;
    private final CanonicalEventValues _forecastValues;

    public SourceCanonicalEventValues(final MEFPForecastSource forecastSource, final CanonicalEventList eventsToCompute)
    {
        _forecastSource = forecastSource;
        _observedValues = new CanonicalEventValues(eventsToCompute);
        _forecastValues = new CanonicalEventValues(eventsToCompute);
    }

    public void clear()
    {
        _observedValues.clearEventValues();
        _forecastValues.clearEventValues();
    }

    public CanonicalEventValues getObservedValues()
    {
        return _observedValues;
    }

    public CanonicalEventValues getForecastValues()
    {
        return _forecastValues;
    }

    public MEFPForecastSource getForecastSource()
    {
        return _forecastSource;
    }

    public void putObservationValues(final long forecastTime, final float[] values)
    {
        _observedValues.putEventValues(forecastTime, values);
    }

    public void putForecastValues(final long forecastTime, final float[] values)
    {
        _forecastValues.putEventValues(forecastTime, values);
    }

    public int getNumberOfForecastTimesWithValues()
    {
        return _forecastValues.getNumberOfComputationalTimesWithValues();
    }

    public float[] getObservationValues(final long forecastTime)
    {
        return _observedValues.getEventValues(forecastTime);
    }

    public float[] getForecastValues(final long forecastTime)
    {
        return _forecastValues.getEventValues(forecastTime);
    }

    public XMLWriter getObservationsWriter(final String tag)
    {
        _observedValues.setTopTagName(tag);
        return _observedValues.getWriter();
    }

    public XMLWriter getForecastsWriter(final String tag)
    {
        _forecastValues.setTopTagName(tag);
        return _forecastValues.getWriter();
    }

    public XMLReader getObservationsReader(final String tag)
    {
        _observedValues.setTopTagName(tag);
        return _observedValues.getReader();
    }

    public XMLReader getForecastsReader(final String tag)
    {
        _forecastValues.setTopTagName(tag);
        return _forecastValues.getReader();
    }

    /**
     * This method computes climatology-based events, which are computed from a historical time series, constructing
     * forecasts for each day at 12Z as if it were being used as a source of forecasts. OVERRIDE if the source must
     * provide its own {@link SourceDataHandler} specific computations for climatological canonical events.
     * 
     * @param observedTS The one long observed time series to use for computing observed events. Note that no parameter
     *            checking is done: the provided time series must be valid.
     * @param firstYear first year of forecast time series to build from observations. Any number smaller than 1900
     *            indicates that the firstYear is determined based on observedTS.
     * @param lastYear last year of forecast time series to build from observations. Any number smaller than 1900
     *            indicates that the lastYear is determined based on observedTS.
     */
    public void computeCanonicalEventValues(final TimeSeriesArray observedTS)
    {
        //Determine firstYear and lastYear from observedTS.
        final int firstYear = HCalendar.computeCalendarFromMilliseconds(observedTS.getStartTime()
            - observedTS.getHeader().getTimeStep().getStepMillis()).get(Calendar.YEAR);
        final int lastYear = HCalendar.computeCalendarFromMilliseconds(observedTS.getEndTime()).get(Calendar.YEAR);

        //Get the first and last forecast times for which to generate canonical events.
        //The first time is always a 1/1 date, while last is always 12/31.  So, we just need to find the one
        //that work.
        final Calendar firstCal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
        firstCal.set(firstYear, GregorianCalendar.JANUARY, 1, 12, 0, 0);
        firstCal.set(Calendar.MILLISECOND, 0);
        while(firstCal.before(observedTS.getStartTime() - observedTS.getHeader().getTimeStep().getStepMillis()))
        {
            firstCal.add(Calendar.YEAR, 1);
        }
        final Calendar lastCal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
        lastCal.set(lastYear, GregorianCalendar.DECEMBER, 31, 12, 0, 0);
        lastCal.set(Calendar.MILLISECOND, 0);
        while(lastCal.after(observedTS.getEndTime()))
        {
            lastCal.add(Calendar.YEAR, -1);
        }

        //Generate a list of forecast times.
        final List<Long> forecastTimes = new ArrayList<Long>(DateTools.countNumberOfDaysBetweenTwoTimes(firstCal.getTimeInMillis(),
                                                                                                        lastCal.getTimeInMillis()));
        for(long time = firstCal.getTimeInMillis(); time <= lastCal.getTimeInMillis(); time += 24 * HCalendar.MILLIS_IN_HR)
        {
            //Check to make sure that the times are not outside the observedTS.  An exception will be thrown if this ever occurs, so we need to 
            //restrict it now.
            if((time >= observedTS.getStartTime()) && (time < observedTS.getEndTime()))
            {
                forecastTimes.add(time);
            }
        }

        //Compute events only for _observedValues.
        _forecastValues.clearEventValues();
        _observedValues.computeEvents(observedTS, forecastTimes, 0);
        _forecastValues.copyValues(_observedValues);
    }

    /**
     * See
     * {@link #computeCanonicalEventValues(LocationAndDataTypeIdentifier, ParameterId, TimeSeriesArray, SourceDataHandler, int, int)}
     * . This passes in null for data type, so that all time series or ensembles are used in the computations.
     */
    public void computeCanonicalEventValues(final LocationAndDataTypeIdentifier identifier,
                                            final TimeSeriesArray observedTS,
                                            final SourceDataHandler handler)
    {
        computeCanonicalEventValues(identifier, null, observedTS, handler);
    }

    /**
     * OVERRIDE if the source must provide its own {@link SourceDataHandler} specific computations.
     * 
     * @param identifier The {@link LocationAndDataTypeIdentifier} identifying the location for which to load time
     *            series from the provided handler.
     * @param dataType The {@link ParameterId} specifying the data type of the data to use that is loaded by the
     *            handler, below. These should be {@link ParameterId#FMAP} for precipitation and
     *            {@link ParameterId#TFMN} and {@link ParameterId#TFMX} for temperature.
     * @param observedTS The one long observed time series to use for computing observed events. This allows for the obs
     *            data to come from a different source than the forecast time series. Note that no parameter checking is
     *            done: the provided time series must be valid.
     * @param handler The {@link SourceDataHandler} providing the forecast time series to read. The observed time series
     *            is provided via the argument above.
     * @param firstYear first year of time series to use in order to subset the forecast time series provided. Any
     *            number smaller than 1900 indicates that the forecast time is not checked against first year.
     * @param lastYear last year of time series to use. Any number smaller than 1900 indicates that the forecast time is
     *            not checked against last year.
     */
    public void computeCanonicalEventValues(final LocationAndDataTypeIdentifier identifier,
                                            final ParameterId dataType,
                                            final TimeSeriesArray observedTS,
                                            final SourceDataHandler handler)
    {
        //If no handler is provided, then this is an observed event only calculation.  Call it and return.
        if(handler == null)
        {
            computeCanonicalEventValues(observedTS);
            return;
        }

//        HStopWatch timer = new HStopWatch();
        //Pull out only those time series that are within the year limits.  Only use a time series if it 
        //matches the specified parameteId, unless it is null in which case use all.
        final List<TimeSeriesArray> timeSeriesUsed = new ArrayList<TimeSeriesArray>();
        for(final TimeSeriesArray ts: handler.getLoadedForecastTimeSeries(identifier))
        {
            if((dataType == null) || (ParameterId.of(ts.getHeader()).equals(dataType)))
            {
                timeSeriesUsed.add(ts);
            }
        }

        //Compute the events.
        final HStopWatch timer = new HStopWatch();
        _forecastValues.computeEvents(timeSeriesUsed,
                                      getForecastSource().getOperationalLagInHoursWhenEstimatingParameters(identifier));
//        System.err.println("####>> ------------------- fcst -- " + timer.getElapsedMillis());
        timer.reset();

        _observedValues.computeEvents(observedTS, _forecastValues.getComputationalTimes(), 0);
//        System.err.println("####>> ------------------- obs -- " + timer.getElapsedMillis());
//        System.err.println("####>> TIME -- " + timer.getElapsedMillis());
    }

}
