package ohd.hseb.hefs.mefp.tools.canonical;

import java.util.List;

import ohd.hseb.hefs.mefp.models.precipitation.IPTPrecipitationParameterCalculator;
import ohd.hseb.hefs.utils.BinaryPredicate;
import ohd.hseb.hefs.utils.tools.NumberTools;

/**
 * A subclass of {@link CanonicalEventValuesGatherer} that implements an MEFP-standard {@link BinaryPredicate} list
 * acceptance checker. It is capably of checking for acceptance based on the following<br>
 * <br>
 * - Number of observed (and therefore forecast) values.<br>
 * - Number of positive observed values.<br>
 * - Number of positive forecast values.<br>
 * - Number of obs and fcst pairs where both are positive. <br>
 * <br>
 * The first is always checked, but the others can be turned on or off as needed. Furthermore, the threshold used to
 * define 'zero' can be determined from calling
 * {@link IPTPrecipitationParameterCalculator#computePrecipitationThreshold(double[], double)}, can be fixed, or can be
 * ignored; see {@link #_thresholdOverride} for more details.<br>
 * <br>
 * If you want to do some positive checks but not all, then set the appropriate attribute to be {@link #DO_NOT_CHECK}.
 * Those attributes are {@link #_minimumRequiredBothPositive}, {@link #_minimumRequiredForecastPositive}, and
 * {@link #_minimumRequiredObservedPositive}. The check for number of observations is always done, so
 * {@link #_minimumRequiredObservations} should never be set to {@link #DO_NOT_CHECK}.
 * 
 * @author hank.herr
 */
public class StandardCanonicalEventValuesGatherer
extends
    CanonicalEventValuesGatherer
{
    /**
     * Constant forces the positive checks to always return true, basically ignoring them.
     */
    public final static int DO_NOT_CHECK = Integer.MIN_VALUE;

    /**
     * Constant defines the fraction of total precipitation values that should be used to compute 0. This value is
     * passed to {@link IPTPrecipitationParameterCalculator#computePrecipitationThreshold(double[], double)} (second
     * argument).
     */
    public double _fractionOfTotalPrecipGTZero = 0.0;

    /**
     * Use {@link #DO_NOT_CHECK} to indicate that the check is not to be performed.
     */
    public int _minimumRequiredBothPositive; //minposobs in the code

    /**
     * Use {@link #DO_NOT_CHECK} to indicate that the check is not to be performed.
     */
    public int _minimumRequiredObservedPositive; //minposobs in the code

    /**
     * Use {@link #DO_NOT_CHECK} to indicate that the check is not to be performed.
     */
    public int _minimumRequiredForecastPositive; //minposobs in the code

    /**
     * Must always be a reasonable number.
     */
    public int _minimumRequiredObservations;

    /**
     * Store the threshold computed the last time a comparison was done for observed values. On value is stored per
     * canonical event.
     */
    public double[] _lastComputedObservedThreshold;

    /**
     * Store the threshold computed the last time a comparison was done for forecast values. On value is stored per
     * canonical event.
     */
    public double[] _lastComputedForecastThreshold;

    /**
     * {@link Double#isNaN()} indicates the threshold must be computed. If {@link Double#isInfinite()} returns true,
     * then the threshold is ignored, so that all positive counts are ignored. Any other value is a fixed threshold and
     * defines what 0 is. The threshold is only useful for precipitation data, so for temperature it is set to
     * {@link Double#POSITIVE_INFINITY}, indicating it is not used.
     */
    public double _thresholdOverride = Double.NaN;

    /**
     * This is the constructor used for temperature and a fixed window width.<br>
     * <br>
     * The following variable settings are used: {@link #DO_NOT_CHECK} is used for {@link #_minimumRequiredBothPositive}
     * , {@link #_minimumRequiredForecastPositive}, and {@link #_minimumRequiredObservedPositive}, and the
     * {@link #_thresholdOverride} is set to {@link Double#POSITIVE_INFINITY} so that it is not used. The value of
     * {@link #_fractionOfTotalPrecipGTZero} is left empty, since it is not used.
     */
    public StandardCanonicalEventValuesGatherer(final CanonicalEventValues forecastValues,
                                                final CanonicalEventValues observedValues,
                                                final int fixedWindowWidth,
                                                final int firstYear,
                                                final int lastYear,
                                                final int minimumRequiredObservations)
    {
        this(forecastValues,
             observedValues,
             fixedWindowWidth,
             fixedWindowWidth,
             firstYear,
             lastYear,
             minimumRequiredObservations);
    }

    /**
     * This is the constructor used for temperature and a variable width. The following variable settings are used:
     * {@link #DO_NOT_CHECK} is used for {@link #_minimumRequiredBothPositive},
     * {@link #_minimumRequiredForecastPositive}, and {@link #_minimumRequiredObservedPositive}, and the
     * {@link #_thresholdOverride} is set to {@link Double#POSITIVE_INFINITY} so that it is not used. The value of
     * {@link #_fractionOfTotalPrecipGTZero} is left unset, since it is not used.
     */
    public StandardCanonicalEventValuesGatherer(final CanonicalEventValues forecastValues,
                                                final CanonicalEventValues observedValues,
                                                final int startingWindowWidth,
                                                final int maximumWindowWidth,
                                                final int firstYear,
                                                final int lastYear,
                                                final int minimumRequiredObservations)
    {
        this(forecastValues,
             observedValues,
             startingWindowWidth,
             maximumWindowWidth,
             firstYear,
             lastYear,
             0.0D,
             minimumRequiredObservations,
             DO_NOT_CHECK,
             DO_NOT_CHECK,
             DO_NOT_CHECK);
        this.setThresholdOverride(Double.POSITIVE_INFINITY);
    }

    /**
     * This is the constructor used for precipitation with a fixed window width. Pass in {@link #DO_NOT_CHECK} for any
     * minimum argument that is not to be used, except minimumRequiredObservations, which must be reasonable.
     */
    public StandardCanonicalEventValuesGatherer(final CanonicalEventValues forecastValues,
                                                final CanonicalEventValues observedValues,
                                                final int fixedWindowWidth,
                                                final int firstYear,
                                                final int lastYear,
                                                final double fractionOfTotalPrecipGTZero,
                                                final int minimumRequiredObservations,
                                                final int minimumRequiredObservedPositive,
                                                final int minimumRequiredForecastPositive,
                                                final int minimumRequiredBothPositive)
    {
        this(forecastValues,
             observedValues,
             fixedWindowWidth,
             fixedWindowWidth,
             firstYear,
             lastYear,
             fractionOfTotalPrecipGTZero,
             minimumRequiredObservations,
             minimumRequiredObservedPositive,
             minimumRequiredForecastPositive,
             minimumRequiredBothPositive);
    }

    /**
     * This is the constructor to use for precipitation with a variable window width.
     * 
     * @param forecastValues The forecast {@link CanonicalEventValues}.
     * @param observedValues The observed {@link CanonicalEventValues} constructed using the forecast times for the
     *            forecast values.
     * @param startingWindowWidth The smallest and initial window width to use in the gatherer.
     * @param maximumWindowWidth The largest allowed window width.
     * @param firstYear The first year for which to gather events. Pass in -1 to provide an unlimited lower bound.
     * @param lastYear The last year for which to gather events. Pass in -1 to provide an unlimited upper bound.
     * @param fractionOfTotalPrecipGTZero The fraction used in the computation of 0 via
     *            {@link IPTPrecipitationParameterCalculator#computePrecipitationThreshold(double[], double)}
     * @param minimumRequiredObservations The number of observations required for the gathered events to be considered
     *            enough.
     * @param minimumRequiredObservedPositive The number of observations that must be positive for the gathered events
     *            to be considered enough. Pass in {@link #DO_NOT_CHECK} to not use (i.e., always pass) this check.
     * @param minimumRequiredForecastPositive The number of forecasts that must be positive for the gathered events to
     *            be considered enough. Pass in {@link #DO_NOT_CHECK} to not use (i.e., always pass) this check.
     * @param minimumRequiredBothPositive The number of obs and fcst pairs for which bothmust be positive for the
     *            gathered events to be considered enough. Typically, this value will be set to
     *            minimumRequiredObservedPositive. Pass in {@link #DO_NOT_CHECK} to not use (i.e., always pass) this
     *            check.
     */
    public StandardCanonicalEventValuesGatherer(final CanonicalEventValues forecastValues,
                                                final CanonicalEventValues observedValues,
                                                final int startingWindowWidth,
                                                final int maximumWindowWidth,
                                                final int firstYear,
                                                final int lastYear,
                                                final double fractionOfTotalPrecipGTZero,
                                                final int minimumRequiredObservations,
                                                final int minimumRequiredObservedPositive,
                                                final int minimumRequiredForecastPositive,
                                                final int minimumRequiredBothPositive)
    {
        super(forecastValues,
              observedValues,
              null,
              startingWindowWidth,
              maximumWindowWidth,
              firstYear,
              lastYear);
        setListAcceptanceCheck(new PrecipitationAcceptanceCheck());
        _fractionOfTotalPrecipGTZero = fractionOfTotalPrecipGTZero;
        _minimumRequiredObservations = minimumRequiredObservations;
        _minimumRequiredObservedPositive = minimumRequiredObservedPositive;
        _minimumRequiredForecastPositive = minimumRequiredForecastPositive;
        _minimumRequiredBothPositive = minimumRequiredBothPositive;
        _lastComputedObservedThreshold = new double[getEvents().size()];
        _lastComputedForecastThreshold = new double[getEvents().size()];
    }

    /**
     * XXX Sensitivity analysis version is the same as
     * {@link #StandardCanonicalEventValuesGatherer(CanonicalEventValues, CanonicalEventValues, int, int, int, int, int, double, int, int, int, int)}
     * , but includes the argument documented below.
     * 
     * @param daysBetweenT0s The number of days between successive T0s to use.
     */
    public StandardCanonicalEventValuesGatherer(final CanonicalEventValues forecastValues,
                                                final CanonicalEventValues observedValues,
                                                final int startingWindowWidth,
                                                final int maximumWindowWidth,
                                                final int firstYear,
                                                final int lastYear,
                                                final int daysBetweenT0s,
                                                final double fractionOfTotalPrecipGTZero,
                                                final int minimumRequiredObservations,
                                                final int minimumRequiredObservedPositive,
                                                final int minimumRequiredForecastPositive,
                                                final int minimumRequiredBothPositive)
    {
        super(forecastValues,
              observedValues,
              null,
              startingWindowWidth,
              maximumWindowWidth,
              firstYear,
              lastYear,
              daysBetweenT0s);
        setListAcceptanceCheck(new PrecipitationAcceptanceCheck());
        _fractionOfTotalPrecipGTZero = fractionOfTotalPrecipGTZero;
        _minimumRequiredObservations = minimumRequiredObservations;
        _minimumRequiredObservedPositive = minimumRequiredObservedPositive;
        _minimumRequiredForecastPositive = minimumRequiredForecastPositive;
        _minimumRequiredBothPositive = minimumRequiredBothPositive;
        _lastComputedObservedThreshold = new double[getEvents().size()];
        _lastComputedForecastThreshold = new double[getEvents().size()];
    }

    /**
     * XXX Sensitivity analysis version is the same as
     * {@link #StandardCanonicalEventValuesGatherer(CanonicalEventValues, CanonicalEventValues, int, int, int, int, int)}
     * , but includes the argument documented below.
     * 
     * @param daysBetweenT0s The number of days between successive T0s to use.
     */
    public StandardCanonicalEventValuesGatherer(final CanonicalEventValues forecastValues,
                                                final CanonicalEventValues observedValues,
                                                final int startingWindowWidth,
                                                final int maximumWindowWidth,
                                                final int firstYear,
                                                final int lastYear,
                                                final int daysBetweenT0s,
                                                final int minimumRequiredObservations)
    {
        this(forecastValues,
             observedValues,
             startingWindowWidth,
             maximumWindowWidth,
             firstYear,
             lastYear,
             daysBetweenT0s,
             0.0D,
             minimumRequiredObservations,
             DO_NOT_CHECK,
             DO_NOT_CHECK,
             DO_NOT_CHECK);
        this.setThresholdOverride(Double.POSITIVE_INFINITY);
    }

    /**
     * @param thresholdOverride Pass in {@link Double#NaN} to use default computation, {@link Double#POSITIVE_INFINITY}
     *            to ignore the threshold checks, or a valid number for a fixed threshold.
     */
    public void setThresholdOverride(final double thresholdOverride)
    {
        _thresholdOverride = thresholdOverride;
    }

    /**
     * @param count The minimum required for which both forecast and observed are positive. This is exposed because the
     *            EPT requires that this not be checked, while the IPT requires it is checked.
     */
    public void setMinimumRequiredBothPositive(final int count)
    {
        this._minimumRequiredBothPositive = count;
    }

    /**
     * @param canonicalEventIndex
     * @return The last forecast threshold computed for that event, meaning that it is the appropriate threshold to use
     *         for the gathered forecast event values.
     */
    public double getLastComputedForecastThreshold(final int canonicalEventIndex)
    {
        return _lastComputedForecastThreshold[canonicalEventIndex];
    }

    /**
     * @param canonicalEventIndex
     * @return The last observed threshold computed for that event, meaning that it is the appropriate threshold to use
     *         for the gathered observed event values.
     */
    public double getLastComputedObservedThreshold(final int canonicalEventIndex)
    {
        return _lastComputedObservedThreshold[canonicalEventIndex];
    }

    public double getFractionOfTotalPrecipGTZero()
    {
        return _fractionOfTotalPrecipGTZero;
    }

    /**
     * @return The last forecast threshold computed for that event, meaning that it is the appropriate threshold to use
     *         for the gathered forecast event values.
     */
    public double getLastComputedForecastThreshold(final CanonicalEvent event)
    {
        return _lastComputedForecastThreshold[getEvents().indexOf(event)];
    }

    /**
     * @return The last observed threshold computed for that event, meaning that it is the appropriate threshold to use
     *         for the gathered observed event values.
     */
    public double getLastComputedObservedThreshold(final CanonicalEvent event)
    {
        return _lastComputedObservedThreshold[getEvents().indexOf(event)];
    }

    /**
     * The list acceptance checker used in {@link CanonicalEventValuesGatherer}.
     * 
     * @author hank.herr
     */
    //TODO This predicate is actually used for both temperature and precip. For temp, many of the values are set to +/- infinity to make it work.
    public class PrecipitationAcceptanceCheck
    extends
        BinaryPredicate<List<Float>, List<Float>>
    {

        @Override
        public boolean apply(final List<Float> forecastEventValues,
                             final List<Float> observedEventValues)
        {
            //No need to do anything if the minimum number of observations is not exceeded.
            if(observedEventValues.size() < _minimumRequiredObservations)
            {
                return false;
            }

            final double[] fcsts =
                                 NumberTools.convertNumbersToDoublesArray(forecastEventValues);
            final double[] obs =
                               NumberTools.convertNumbersToDoublesArray(observedEventValues);

            //pthresh1 in fortran code is obsthresh, 2 is fcstThresh.  The threshold is only computed
            //when _thresholdOverride is set to a valid (non infinite) value.  
            double obsThresh = _thresholdOverride;
            double fcstThresh = _thresholdOverride;
            if(Double.isNaN(obsThresh))
            {
                obsThresh =
                          IPTPrecipitationParameterCalculator.computePrecipitationThreshold(obs,
                                                                                            _fractionOfTotalPrecipGTZero);
                fcstThresh =
                           IPTPrecipitationParameterCalculator.computePrecipitationThreshold(fcsts,
                                                                                             _fractionOfTotalPrecipGTZero);
            }

            //Record the last computed thresholds.
            _lastComputedForecastThreshold[getWorkingEventNumber()] =
                                                                    fcstThresh;
            _lastComputedObservedThreshold[getWorkingEventNumber()] = obsThresh;

            //Count the number of positives.
            int countPosObs = 0;
            int countPosFcst = 0;
            int countPosBoth = 0;
            if(!Double.isInfinite(_thresholdOverride))
            {
                for(int i = 0; i < fcsts.length; i++)
                {
                    if(fcsts[i] > fcstThresh)
                    {
                        countPosFcst++;
                    }
                    if(obs[i] > obsThresh)
                    {
                        countPosObs++;
                    }
                    if((obs[i] > obsThresh) && (fcsts[i] > fcstThresh))
                    {
                        countPosBoth++;
                    }
                }
            }

            /**
             * Do the check.
             */
            return (fcsts.length >= _minimumRequiredObservations)
                && (countPosBoth >= _minimumRequiredBothPositive)
                && (countPosFcst >= _minimumRequiredForecastPositive)
                && (countPosObs >= _minimumRequiredObservedPositive);
        }

    }
}
