package ohd.hseb.hefs.pe.acceptance;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.FlowLayout;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import ohd.hseb.hefs.pe.acceptance.group.GroupEditorPanel;
import ohd.hseb.hefs.pe.acceptance.group.ZipGroup;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepOptionsPanel;
import ohd.hseb.hefs.pe.core.StepUnit;
import ohd.hseb.hefs.pe.notice.SelectedIdentifiersChangedNotice;
import ohd.hseb.hefs.pe.notice.StepStatusRefreshAllNotice;
import ohd.hseb.hefs.pe.notice.StepUnitsSelectedNotice;
import ohd.hseb.hefs.pe.notice.StepUpdatedNotice;
import ohd.hseb.hefs.pe.tools.GenericSummaryTablePanel;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.gui.help.HelpFile;
import ohd.hseb.hefs.utils.gui.jtable.GenericTable;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.notify.collect.CollectionModifiedNotice;
import ohd.hseb.hefs.utils.tools.ListTools;

import com.google.common.eventbus.Subscribe;

/**
 * This may need to be overridden to specify a component-specific help file.
 * 
 * @author hank.herr
 */
@SuppressWarnings("serial")
@HelpFile("helpManual.htm#AcceptanceSubpanelRef")
public class AcceptancePEStepOptionsPanel extends ParameterEstimatorStepOptionsPanel implements
SelectedIdentifiersChangedNotice.Subscriber, CollectionModifiedNotice.Subscriber, StepUpdatedNotice.Subscriber,
StepStatusRefreshAllNotice.Subscriber
{

    private final GenericTable<ZipGroup> _groupsTable;

    private JTabbedPane _acceptanceTabs;
    private GenericSummaryTablePanel _identifiersPanel;

    public AcceptancePEStepOptionsPanel(ParameterEstimatorRunInfo runInfo, AcceptancePEStepProcessor stepProc)
    {
        super(runInfo, stepProc);

        _groupsTable = new GroupAcceptanceTableModel(getRunInfo()).createDefaultTable();

        initializeDisplay();
    }

    private void refreshTable()
    {
        getRunInfo().getAcceptedZipFileHandler().resetMemory();
        _groupsTable.refresh();
        _identifiersPanel.refreshTable();
    }

    @Override
    protected void initializeDisplay()
    {
        JPanel editGroupPanel = new GroupEditorPanel(getRunInfo());

        //Identifier panel.
        JPanel identifiersWrapperPanel = new JPanel(new BorderLayout());
        JPanel labelLocPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
        labelLocPanel.setBorder(new EmptyBorder(1, 2, 3, 2));
        labelLocPanel.add(new JLabel("Fully update all zip files associated with selected locations."));
        identifiersWrapperPanel.add(labelLocPanel, BorderLayout.NORTH);
        _identifiersPanel = new GenericSummaryTablePanel(getRunInfo(),
                                                         "Summary of Already Accepted Parameters",
                                                         new IdentifierAcceptanceTableModel(getRunInfo()),
                                                         true,
                                                         Collections.<Component>emptyList());
        identifiersWrapperPanel.add(_identifiersPanel, BorderLayout.CENTER);

        //Group panel.
        JPanel groupWrapperPanel = new JPanel(new BorderLayout());
        JPanel groupLabelPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
        groupLabelPanel.setBorder(new EmptyBorder(1, 2, 3, 2));
        groupLabelPanel.add(new JLabel("Fully update zip files for selected groups."));
        groupWrapperPanel.add(groupLabelPanel, BorderLayout.NORTH);
        JPanel groupPanel = _groupsTable.createDisplayPanel(HSwingFactory.createTitledBorder(BorderFactory.createEtchedBorder(1),
                                                                                             "Summary of Group Zip Files",
                                                                                             null));
        groupWrapperPanel.add(groupPanel, BorderLayout.CENTER);

        _acceptanceTabs = new JTabbedPane();
        _acceptanceTabs.add("Edit Groups", editGroupPanel);
        _acceptanceTabs.add("Prepare by Location", identifiersWrapperPanel);
        _acceptanceTabs.setToolTipTextAt(1, "Fully update zip files for all groups containing selected locations.");
        _acceptanceTabs.add("Prepare by Group", groupWrapperPanel);
        _acceptanceTabs.setToolTipTextAt(2, "Fully update the zip files for selected groups.");

        _acceptanceTabs.addChangeListener(new ChangeListener()
        {
            @Override
            public void stateChanged(ChangeEvent e)
            {
                post(new StepUnitsSelectedNotice(AcceptancePEStepOptionsPanel.this,
                                                 getStepProcessor().getClass(),
                                                 getStepUnitsToPerform()));
            }
        });

        this.setLayout(new BorderLayout());
        this.add(_acceptanceTabs, BorderLayout.CENTER);
    }

    @Override
    public List<? extends StepUnit> getStepUnitsToPerform()
    {
        if(_acceptanceTabs.getSelectedIndex() == 1)
        {
            return _identifiersPanel.getSelectedIdentifiers();
        }
        else if(_acceptanceTabs.getSelectedIndex() == 2)
        {
            return _groupsTable.getSelectedWholeRows();
        }
        return new ArrayList<StepUnit>();
    }

    @Override
    public void reactToSelectedIdentifiersChanged(SelectedIdentifiersChangedNotice evt)
    {
        _identifiersPanel.getModel().setIdentifiers(evt.getIdentifiers());
        _identifiersPanel.refreshTable();
    }

    @Override
    public void gotoUnit(Collection<StepUnit> units)
    {
        if(units.isEmpty())
        {
            _identifiersPanel.clearSelection();
            return;
        }
        if(units.iterator().next() instanceof LocationAndDataTypeIdentifier)
        {
            _acceptanceTabs.setSelectedIndex(1);
            _identifiersPanel.selectIdentifiers(ListTools.convertCollection(units, (LocationAndDataTypeIdentifier)null));
        }
        else
        {
            throw new IllegalArgumentException(String.format("%s not of class %s",
                                                             units.iterator().next(),
                                                             LocationAndDataTypeIdentifier.class.getCanonicalName()));
        }
    }

    @Override
    public void reactToCollectionModified(CollectionModifiedNotice evt)
    {
        if(evt.getTarget() == getRunInfo().getZipGroupInfo())
        {
            _identifiersPanel.getModel().fireTableDataChanged();
        }
    }

    @Override
    @Subscribe
    public void reactToStepUpdated(StepUpdatedNotice evt)
    {
        //The acceptance tables cannot update while run-all is being done, because it may cause concurrency issues.
        //This will react to the final StepUpdatedNotice event posted to the bus, which is marked by its class being the
        //interface ParameterEstimatorStepProcessor.
        if(evt.getStep() == null)
        {
            refreshTable();
        }
    }

    @Override
    @Subscribe
    public void reactToStepStatusRefreshAll(StepStatusRefreshAllNotice notice)
    {
        //If the location summary panel is refreshing all, then reset the memory of the zip file handler
        //in order to ensure a complete refresh.
        this.refreshTable();
    }

}
