package ohd.hseb.hefs.pe.acceptance;

import java.io.File;
import java.util.List;

import ohd.hseb.hefs.pe.acceptance.group.GroupZipFileHandler;
import ohd.hseb.hefs.pe.acceptance.group.ZipGroup;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.StepUnit;
import ohd.hseb.hefs.pe.estimation.GenericEstimationPEStepProcessor;
import ohd.hseb.hefs.pe.notice.StepUpdatedNotice;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.jobs.JobMessenger;
import ohd.hseb.hefs.utils.notify.collect.CollectionAddAllNotice;
import ohd.hseb.hefs.utils.notify.collect.CollectionAddNotice;
import ohd.hseb.hefs.utils.notify.collect.CollectionModifiedNotice;
import ohd.hseb.hefs.utils.notify.collect.CollectionRemoveNotice;
import ohd.hseb.hefs.utils.status.StatusIndicator;

import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.eventbus.Subscribe;

public class AcceptedParameterFileHandler implements StepUpdatedNotice.Subscriber, CollectionModifiedNotice.Subscriber
{
    private final ParameterEstimatorRunInfo _runInfo;
    private final LoadingCache<LocationAndDataTypeIdentifier, IdentifierZipFileHandler> _identifierHandlers;
    private final LoadingCache<ZipGroup, GroupZipFileHandler> _groupHandlers;

    public AcceptedParameterFileHandler(ParameterEstimatorRunInfo runInfo)
    {

        this(runInfo, IdentifierZipFileHandler.makeCache(runInfo));
    }

    public AcceptedParameterFileHandler(ParameterEstimatorRunInfo runInfo,
                                        LoadingCache<LocationAndDataTypeIdentifier, IdentifierZipFileHandler> identifierHandlers)
    {
        _runInfo = runInfo;
        _runInfo.register(this);
        _identifierHandlers = identifierHandlers;
        _groupHandlers = GroupZipFileHandler.makeCache(_runInfo);

        //I need to ensure that all group handlers are loaded when they are added to the zip group info so that they
        //can be edited properly within the GroupEditorPanel.  
        _runInfo.getZipGroupInfo().register(this);
    }

    public IdentifierZipFileHandler getZipFileHandler(LocationAndDataTypeIdentifier identifier)
    {
        return _identifierHandlers.getUnchecked(identifier);
    }

    public GroupZipFileHandler getHandler(ZipGroup group)
    {
        return _groupHandlers.getUnchecked(group);
    }

    public LoadingCache<LocationAndDataTypeIdentifier, IdentifierZipFileHandler> getIdentifierCache()
    {
        return _identifierHandlers;
    }

    public StatusIndicator getZipStatus(StepUnit unit)
    {
        if(unit instanceof LocationAndDataTypeIdentifier)
        {
            return _identifierHandlers.getUnchecked((LocationAndDataTypeIdentifier)unit).getZipStatus();
        }
        else if(unit instanceof ZipGroup)
        {
            return _groupHandlers.getUnchecked((ZipGroup)unit).getStatus();
        }
        return null;
    }

    public List<File> prepareZipFiles(StepUnit unit) throws Exception
    {
        if(unit instanceof LocationAndDataTypeIdentifier)
        {
            JobMessenger.updateNote("Updating all zip files for location "
                + ((LocationAndDataTypeIdentifier)unit).buildStringToDisplayInTree() + "...");
            return getZipFileHandler((LocationAndDataTypeIdentifier)unit).prepare();
        }
        else if(unit instanceof ZipGroup)
        {
            JobMessenger.updateNote("Zipping files for group " + ((ZipGroup)unit).getName() + "...");
            return Lists.newArrayList(getHandler((ZipGroup)unit).prepare());
        }
        return null;
    }

    public void resetMemory()
    {
        for(GroupZipFileHandler handler: _groupHandlers.asMap().values())
        {
            handler.resetMemory();
        }
        for(IdentifierZipFileHandler handler: _identifierHandlers.asMap().values())
        {
            handler.resetMemory();
        }
    }

    @Override
    @Subscribe
    public void reactToStepUpdated(StepUpdatedNotice evt)
    {
        if((evt.getStep() != null) && (GenericEstimationPEStepProcessor.class.isAssignableFrom(evt.getStep())))
        {
            resetMemory();
        }
    }

    @Override
    @Subscribe
    public void reactToCollectionModified(CollectionModifiedNotice notice)
    {
        //Changes to ZipGroupInfo.  Update the cache as needed.
        if(notice.getSource() == _runInfo.getZipGroupInfo())
        {
            if(notice instanceof CollectionAddAllNotice)
            {
                for(Object item: ((CollectionAddAllNotice)notice).getAddedItems())
                {
                    _groupHandlers.getUnchecked((ZipGroup)item);
                }
            }
            else if(notice instanceof CollectionAddNotice)
            {
                _groupHandlers.getUnchecked((ZipGroup)((CollectionAddNotice)notice).getAddedItem());
            }
            else if(notice instanceof CollectionRemoveNotice)
            {
                _groupHandlers.invalidate(((CollectionRemoveNotice)notice).getRemovedItem());
            }
        }
    }
}
