package ohd.hseb.hefs.pe.acceptance;

import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import ohd.hseb.hefs.pe.acceptance.group.GroupZipFileHandler;
import ohd.hseb.hefs.pe.acceptance.group.ZipGroup;
import ohd.hseb.hefs.pe.acceptance.group.ZipGroupInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorTable;
import ohd.hseb.hefs.pe.core.ParameterEstimatorTableModel;
import ohd.hseb.hefs.pe.notice.StepUnitsSelectedNotice;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifierList;
import ohd.hseb.hefs.utils.gui.jtable.models.RefreshTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.SelectionMaintainingTableModel;
import ohd.hseb.hefs.utils.notify.collect.CollectionModifiedNotice;
import ohd.hseb.hefs.utils.status.StatusLabel;
import ohd.hseb.hefs.utils.tools.StringTools;

import com.google.common.eventbus.Subscribe;

/**
 * Make sure to set the event bus, so this reacts to changes in group info.
 * 
 * @author alexander.garbarino
 */
@SuppressWarnings("serial")
public class GroupAcceptanceTableModel extends ParameterEstimatorTableModel implements
SelectionMaintainingTableModel<ZipGroup>, RefreshTableModel
{
    private static final String[] COLUMN_NAMES = {"Group", "Required #", "Present #", "Up-to-date #", "Prepared?"};
    private static final String[] COLUMN_TIPS = {"The name of this group.",
        "The number of files to be put in this group's zip file.",
        "The number of files present in this group's zip file.",
        "The number of up-to-date files in this group's zip file.", "The status of this zip group."};

    private final AcceptedParameterFileHandler _handler;
    private final ZipGroupInfo _groupInfo;

    public GroupAcceptanceTableModel(ParameterEstimatorRunInfo runInfo)
    {
        super(runInfo);
        _handler = runInfo.getAcceptedZipFileHandler();
        _groupInfo = runInfo.getZipGroupInfo();
        _groupInfo.register(this);
    }

    @Subscribe
    public void reactToZipGroupsChanged(CollectionModifiedNotice evt)
    {
        refresh();
        this.fireTableDataChanged();
    }

    @Override
    public int getRowCount()
    {
        return _groupInfo.size();
    }

    @Override
    public int getColumnCount()
    {
        return COLUMN_NAMES.length;
    }

    @Override
    public String getColumnName(int column)
    {
        if(0 <= column && column < COLUMN_NAMES.length)
        {
            return COLUMN_NAMES[column];
        }
        return "";
    }

    @Override
    public Class<?> getColumnClass(int column)
    {
        if(column == 4)
        {
            return StatusLabel.class;
        }
        else
        {
            return String.class;
        }
    }

    @Override
    public Object getValueAt(int rowIndex, int columnIndex)
    {
        ZipGroup group = _groupInfo.get(rowIndex);
        GroupZipFileHandler handler = _handler.getHandler(group);
        if(columnIndex == 0)
        {
            return group.getName();
        }
        else if(columnIndex == 1)
        {
            return handler.getRequiredFiles().size();
        }
        else if(columnIndex == 2)
        {
            return handler.getPresentEntries().size();
        }
        else if(columnIndex == 3)
        {
            return handler.getUpdatedEntries().size();
        }
        else if(columnIndex == 4)
        {
            return StatusLabel.make(handler.getStatus());
        }

        return null;
    }

    @Override
    public String getColumnHeaderToolTip(int col)
    {
        if(0 <= col && col < COLUMN_TIPS.length)
        {
            return COLUMN_TIPS[col];
        }
        return null;
    }

    @Override
    public String getCellToolTip(int rowIndex, int visibleColIndex)
    {
        if((visibleColIndex == 0) && (rowIndex >= 0) && (rowIndex <= this.getRowCount()))
        {
            LocationAndDataTypeIdentifierList idList = new LocationAndDataTypeIdentifierList(_groupInfo.get(rowIndex));
            String result = "Identifiers in group: ";
            if(idList.isEmpty())
            {
                result += "-none- ";
            }
            else
            {
                result += idList.buildListOfIdentifierStrings();
            }
            return StringTools.htmlizeString(StringTools.wordWrap(result, 100));
        }
        else if((visibleColIndex == 4) && (rowIndex >= 0) && (rowIndex <= this.getRowCount()))
        {
            return ((StatusLabel)getValueAt(rowIndex, visibleColIndex)).getToolTipText();
        }
        return null;
    }

    @Override
    public ZipGroup getRow(int index)
    {
        return _groupInfo.get(index);
    }

    @Override
    public int getRowIndex(ZipGroup group)
    {
        return _groupInfo.indexOf(group);
    }

    @Override
    public void refresh()
    {
        _handler.resetMemory();
        this.fireTableDataChanged();
    }

    @Override
    public GroupAcceptanceTable instantiateDefaultTable()
    {
        super.instantiateDefaultTable();
        return new GroupAcceptanceTable();
    }

    private class GroupAcceptanceTable extends ParameterEstimatorTable
    {
        private GroupAcceptanceTable()
        {
            super(GroupAcceptanceTableModel.this);
            getSelectionModel().addListSelectionListener(new ListSelectionListener()
            {
                @Override
                public void valueChanged(ListSelectionEvent e)
                {
                    if(e.getValueIsAdjusting())
                    {
                        return;
                    }

                    post(new StepUnitsSelectedNotice(this, AcceptancePEStepProcessor.class, getSelectedWholeRows()));
                }
            });

            getSelectionModel().setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
        }
    }
}
