package ohd.hseb.hefs.pe.acceptance;

import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.StepTableModel;
import ohd.hseb.hefs.pe.notice.StepUpdatedNotice;
import ohd.hseb.hefs.pe.tools.GenericSummaryTableStatusProvider;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.gui.jtable.models.RefreshTableModel;
import ohd.hseb.hefs.utils.notify.collect.CollectionModifiedNotice;
import ohd.hseb.hefs.utils.status.StatusLabel;

import com.google.common.cache.LoadingCache;
import com.google.common.eventbus.Subscribe;

@SuppressWarnings("serial")
public class IdentifierAcceptanceTableModel extends StepTableModel implements RefreshTableModel,
CollectionModifiedNotice.Subscriber
{
    private final AcceptedParameterFileHandler _handler;

    public IdentifierAcceptanceTableModel(final ParameterEstimatorRunInfo runInfo)
    {
        super(runInfo,
              AcceptancePEStepProcessor.class,
              new FileAvailableTableStatusProvider(runInfo.getAcceptedZipFileHandler()),
              new IdentifierZipStatusProvider(runInfo.getAcceptedZipFileHandler()));
        _handler = runInfo.getAcceptedZipFileHandler();
    }

    @Override
    @Subscribe
    public void reactToCollectionModified(final CollectionModifiedNotice evt)
    {
        if(evt.getTarget() == getRunInfo().getZipGroupInfo())
        {
            refresh();
        }
    }

    @Override
    public void reactToStepUpdated(final StepUpdatedNotice evt)
    {
        // Do nothing, as the GroupsChangedEvent provides better notification.
    }

    @Override
    public void refresh()
    {
        _handler.resetMemory();
        fireTableDataChanged();
    }

    /**
     * Provider for the Est? status column.
     * 
     * @author hankherr
     */
    private static class FileAvailableTableStatusProvider implements GenericSummaryTableStatusProvider
    {
        private final AcceptedParameterFileHandler _handler;

        private FileAvailableTableStatusProvider(final AcceptedParameterFileHandler handler)
        {
            _handler = handler;
        }

        @Override
        public String getToolTipForColumnHeader()
        {
            return "Indicates if all parameter files required for acceptance are available.";
        }

        @Override
        public String getStatusColumnName()
        {
            return "Est?";
        }

        @Override
        public StatusLabel getStatus(final LocationAndDataTypeIdentifier identifier)
        {
            //Returns the yes/no status label indicating if all required parameter files are present for zipping.
            if(_handler.getZipStatus(identifier).canBePrepared())
            {
                return StatusLabel.make(true, "<html>All required estimated parameter files are available.<br>"
                    + "Check the Estimation Panel to see if they are up-to-date.</html>");
            }
            else
            {
                return StatusLabel.make(false, "<html>The following required estimated parameter file is missing:<br>"
                    + _handler.getZipFileHandler(identifier).getMissingFiles() + "<br>"
                    + "Regenerate the parameters via the Estimation Panel in order to build the missing file.</html>");
            }
        }
    }

    /**
     * Provider for the Status column, displaying if the parameter files have been zipped.
     * 
     * @author hankherr
     */
    private static class IdentifierZipStatusProvider implements GenericSummaryTableStatusProvider
    {
        private final LoadingCache<LocationAndDataTypeIdentifier, IdentifierZipFileHandler> _handlers;

        private IdentifierZipStatusProvider(final AcceptedParameterFileHandler handler)
        {
            _handlers = handler.getIdentifierCache();
        }

        @Override
        public String getStatusColumnName()
        {
            return "Status";
        }

        @Override
        public String getToolTipForColumnHeader()
        {
            return "If the given identifier has been placed in all specified group zips.";
        }

        @Override
        public StatusLabel getStatus(final LocationAndDataTypeIdentifier identifier)
        {
            return StatusLabel.make(_handlers.getUnchecked(identifier).getZipStatus());
        }

    }

}
