package ohd.hseb.hefs.pe.acceptance;

import java.io.File;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.zip.ZipEntry;

import ohd.hseb.hefs.pe.acceptance.group.GroupZipFileHandler;
import ohd.hseb.hefs.pe.acceptance.group.ZipGroup;
import ohd.hseb.hefs.pe.acceptance.group.ZipGroupInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.notify.collect.ElementsChangedNotice;
import ohd.hseb.hefs.utils.status.BooleanStatus;
import ohd.hseb.hefs.utils.tools.FileTools;

import com.google.common.base.Predicates;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.eventbus.Subscribe;

public class IdentifierZipFileHandler implements ElementsChangedNotice.Subscriber<ZipGroup, ZipGroupInfo>
{
    private final LocationAndDataTypeIdentifier _identifier;
    private final ParameterEstimatorRunInfo _runInfo;
    private final AcceptedParameterFileHandler _handler;
    private final File _preparedDir;

    protected Set<File> _requiredFiles;
    private Set<File> _missingFiles;
    private Boolean _isPresentInZips;
    private Boolean _isUpdatedInZips;
    private IdentifierZipStatus _status;

    protected IdentifierZipFileHandler(final LocationAndDataTypeIdentifier identifier,
                                       final ParameterEstimatorRunInfo runInfo)
    {
        _identifier = identifier;
        _runInfo = runInfo;
        _handler = _runInfo.getAcceptedZipFileHandler();
        _preparedDir = new File(runInfo.getBaseDirectory().getAbsolutePath() + "/parameters");

        _runInfo.getZipGroupInfo().register(this);

        resetMemory();
    }

    public LocationAndDataTypeIdentifier getIdentifier()
    {
        return _identifier;
    }

    public ParameterEstimatorRunInfo getRunInfo()
    {
        return _runInfo;
    }

    /**
     * Override this to add additional required parameter files. Be sure to add any additional files to _requiredFiles
     * within the overridden method so that the list need not be built more than once.
     * 
     * @return Set of required files for zipping given the identifier specified in the constructor.
     */
    public synchronized Set<File> getRequiredFiles()
    {
        if(_requiredFiles == null)
        {
            _requiredFiles = Sets.newHashSet();
            if(_runInfo.getEstimatedParametersFileHandler() != null)
            {
                _requiredFiles.addAll(_runInfo.getEstimatedParametersFileHandler().getTrackedFiles(_identifier));
            }
        }
        return _requiredFiles;
    }

    public synchronized Set<File> getMissingFiles()
    {
        if(_missingFiles == null)
        {
            _missingFiles = Sets.filter(getRequiredFiles(), Predicates.not(FileTools.DOES_FILE_EXIST));
        }
        return _missingFiles;
    }

    public BooleanStatus getCanPrepareStatus()
    {
        return BooleanStatus.get(getMissingFiles().isEmpty());
    }

    public synchronized IdentifierZipStatus getZipStatus()
    {
        if(_status == null)
        {
            if(!getMissingFiles().isEmpty())
            {
                _status = IdentifierZipStatus.MISSING_COMPONENTS_ON_FILE_SYSTEM;
            }
            else if(_runInfo.getZipGroupInfo().forIdentifier(_identifier).isEmpty())
            {
                _status = IdentifierZipStatus.NO_ASSIGNED_GROUPS;
            }
            else if(!isPresentInZips())
            {
                _status = IdentifierZipStatus.MISSING_FILES_IN_ZIPS;
            }
            else if(!isUpdatedInZips())
            {
                _status = IdentifierZipStatus.NEEDS_UPDATE;
            }
            else
            {
                _status = IdentifierZipStatus.COMPLETE;
            }
        }
        return _status;
    }

    public synchronized boolean isPresentInZips()
    {
        if(_isPresentInZips == null)
        {
            _isPresentInZips = true;
            outer: for(final ZipGroup group: _runInfo.getZipGroupInfo().forIdentifier(_identifier))
            {
                final GroupZipFileHandler handler = _handler.getHandler(group);
                final Set<ZipEntry> entries = Sets.newHashSet(handler.getPresentEntries());
                for(final File file: getRequiredFiles())
                {
                    boolean found = false;
                    for(final ZipEntry entry: entries)
                    {
                        if(FileTools.getRelativeFile(_preparedDir, file).getPath().equals(entry.getName()))
                        {
                            found = true;
                            entries.remove(entry);
                            break;
                        }
                    }
                    if(!found)
                    {
                        _isPresentInZips = false;
                        break outer;
                    }
                }
            }
        }
        return _isPresentInZips;
    }

    /**
     * @return True if all of the required files for the identifier are older than the entries in all containing group
     *         zip files.
     */
    public synchronized boolean isUpdatedInZips()
    {
        if(_isUpdatedInZips == null)
        {
            _isUpdatedInZips = true;
            for(final ZipGroup group: _runInfo.getZipGroupInfo().forIdentifier(_identifier))
            {
                final GroupZipFileHandler handler = _handler.getHandler(group);
                Set<ZipEntry> entries = new HashSet<ZipEntry>();
                if(!handler.getUpdatedEntries().isEmpty())
                {
                    entries = Sets.newHashSet(handler.getUpdatedEntries());
                }

                for(final File file: getRequiredFiles())
                {
                    boolean found = false;
                    for(final ZipEntry entry: entries)
                    {
                        if(FileTools.getRelativeFile(_preparedDir, file).getPath().equals(entry.getName()))
                        {
                            found = true;
                            entries.remove(entry);
                            break;
                        }
                    }
                    if(!found)
                    {
                        _isUpdatedInZips = false;
                        return _isUpdatedInZips;
                    }
                }
            }
        }
        return _isUpdatedInZips;
    }

    /**
     * Prepare all groups that <code>_identifier</code> is in.
     * 
     * @return all prepared files
     */
    public List<File> prepare() throws Exception
    {
        final List<File> files = Lists.newArrayList();
        for(final ZipGroup group: _runInfo.getZipGroupInfo().forIdentifier(_identifier))
        {
            files.add(_handler.getHandler(group).prepare()); //used to pass in false for list notifications
        }
        resetMemory();

//XXX If I include these notify prepared things, it may cause multithreading issues because the
//zip file handler resets are called when the table updates... not sure of details.
//_runInfo.getZipGroupInfo().notifyPrepared(this, _identifier);

        return files;
    }

    public synchronized void resetMemory()
    {
        _requiredFiles = null;
        _missingFiles = null;
        _isPresentInZips = null;
        _isUpdatedInZips = null;
        _status = null;
    }

    @Override
    @Subscribe
    public void reactToElementsChanged(final ElementsChangedNotice<ZipGroup, ZipGroupInfo> evt)
    {
        resetMemory();
    }

    /**
     * Subclasses may need to 'override' this static method. For example,
     * {@link MEFPIdentifierZipFileHandler#makeCache(ParameterEstimatorRunInfo)}.
     * 
     * @return By default, an Cache of IdentifierZipFileHandler objects.
     */
    public static LoadingCache<LocationAndDataTypeIdentifier, IdentifierZipFileHandler> makeCache(final ParameterEstimatorRunInfo runInfo)
    {
        final CacheBuilder builder = CacheBuilder.newBuilder();
        return builder.build(new CacheLoader<LocationAndDataTypeIdentifier, IdentifierZipFileHandler>()
        {
            @Override
            public IdentifierZipFileHandler load(final LocationAndDataTypeIdentifier key) throws Exception
            {
                return new IdentifierZipFileHandler(key, runInfo);
            }
        });
    }
}
