package ohd.hseb.hefs.pe.acceptance.group;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorSubPanel;
import ohd.hseb.hefs.pe.notice.AvailableIdentifiersChangedNotice;
import ohd.hseb.hefs.pe.notice.SelectedIdentifiersChangedNotice;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifierList;
import ohd.hseb.hefs.utils.gui.components.ItemSelectionPanel;
import ohd.hseb.hefs.utils.gui.components.SelectAllOnFocusTextField;
import ohd.hseb.hefs.utils.gui.jtable.GenericTable;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.JToolBarBuilder;
import ohd.hseb.hefs.utils.gui.tools.MultiItemButton;
import ohd.hseb.hefs.utils.gui.tools.SelfListeningButton;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.tools.ListTools;

import com.google.common.eventbus.Subscribe;

@SuppressWarnings("serial")
public class GroupEditorPanel extends ParameterEstimatorSubPanel implements
AvailableIdentifiersChangedNotice.Subscriber, SelectedIdentifiersChangedNotice.Subscriber
{
    private final JSplitPane _mainPane;
    private final GenericTable _groupTable;
    private ItemSelectionPanel _selector;
    private final JTextField _groupFileNameField;

    private final ZipGroupInfo _groups;

    private ZipGroup _selectedGroup;

    // TESTING
    @SuppressWarnings("unused")
    private GroupEditorPanel(ParameterEstimatorRunInfo runInfo, LocationAndDataTypeIdentifierList identifiers)
    {
        this(runInfo);
        createSelector(identifiers);
    }

    public GroupEditorPanel(ParameterEstimatorRunInfo runInfo)
    {
        super(runInfo);
        ZipGroupInfo groups = runInfo.getZipGroupInfo();
        _groups = groups;
        _selectedGroup = null;

        _mainPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
        _groupFileNameField = new SelectAllOnFocusTextField();
        _groupTable = new GroupListTableModel(groups).makeTable();
        _groupTable.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);

        initializeDisplay();
        createSelector();
    }

    private void activateGroupEditorPanelForSelectedGroup(int selectedGroupIndex)
    {
        if(_selectedGroup != null)
        {
            _selectedGroup.unregister(this);
        }
        _selectedGroup = _groups.get(selectedGroupIndex);
        _selectedGroup.register(this);
        _groupFileNameField.setText(_selectedGroup.getName());
        _selector.setOutputCollection(_selectedGroup);
        _selector.setTitle("Select Group Locations for " + _selectedGroup.getName() + ":");
        _selector.setEnabled(true);
        _groupFileNameField.setEnabled(true);
    }

    private void deactivateGroupEditorPanel()
    {
        _selector.setTitle("Select Group Locations");
        _selector.setOutputCollection(null);
        _selector.setEnabled(false);
        _groupFileNameField.setEnabled(false);
        _groupFileNameField.setText("");
    }

    public void createSelector()
    {
        createSelector(getRunInfo().getCurrentlyWorkingIdentifiers());
    }

    /**
     * Reconstructs the selector as needed using the given identifiers.
     * 
     * @param identifiers
     */
    private void createSelector(LocationAndDataTypeIdentifierList identifiers)
    {
        //record current table selection
        int selectedRow = _groupTable.getSelectedRow();

        _selector = new ItemSelectionPanel<LocationAndDataTypeIdentifier>(identifiers,
                                                                          null,
                                                                          LocationAndDataTypeIdentifier.TO_DISPLAY_STRING);
        _selector.setTitle("Select Group Locations");
        _selector.setEnabled(false);
        _selector.setSelectionTitles("Available Identifiers", "Identifiers in Group");

        //file name text field.
        JPanel fieldPanel = new JPanel(new BorderLayout());
        fieldPanel.add(new JLabel("Group File Name: "), BorderLayout.WEST);
        _groupFileNameField.setText("");
        fieldPanel.add(_groupFileNameField, BorderLayout.CENTER);
        _groupFileNameField.setToolTipText("<html>The name of the zip file to be created, without .zip<br>"
            + "Should match the module instance descriptor of module to use it.</html>");
        fieldPanel.setBorder(new EmptyBorder(2, 2, 2, 2));
        _groupFileNameField.setEnabled(false);

        JPanel panel = new JPanel(new BorderLayout());
        panel.add(fieldPanel, BorderLayout.NORTH);
        panel.add(_selector, BorderLayout.CENTER);

        _mainPane.setRightComponent(panel);
        this.validate();

        if(selectedRow > 0)
        {
            _groupTable.getSelectionModel().setSelectionInterval(selectedRow, selectedRow);
        }
    }

    /**
     * @param prefix
     * @return True if any group was added, false if no groups were added.
     */
    private boolean addGroupForEachIdentifier(String prefix)
    {
        List<LocationAndDataTypeIdentifier> identifiers = getRunInfo().getCurrentlyWorkingIdentifiers();
        boolean identifierWasAdded = false;

        for(LocationAndDataTypeIdentifier identifier: identifiers)
        {
            ZipGroup group = new ZipGroup(prefix + identifier.getLocationId() + "_" + identifier.getParameterId());
            group.add(identifier);
            identifierWasAdded = _groups.addGroupIfNotPresent(group) || identifierWasAdded;
        }

        return identifierWasAdded;
    }

    @Override
    protected void initializeDisplay()
    {
        _groupTable.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);

        //The table selection listener.
        _groupTable.getSelectionModel().addListSelectionListener(new ListSelectionListener()
        {
            @Override
            public void valueChanged(ListSelectionEvent e)
            {
                int[] indices = _groupTable.getSelectedRows();
                if(indices.length == 1 && indices[0] < _groupTable.getModel().getRowCount() - 1)
                {
                    activateGroupEditorPanelForSelectedGroup(indices[0]);
                }
                else
                {
                    deactivateGroupEditorPanel();
                }
            }
        });

        //Text field listener: updates table with each key stroke
        _groupFileNameField.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent e)
            {
                //Check the text field value to see if something else exists with that name.
                String newName = _groupFileNameField.getText().trim();
                if(_groups.getGroup(newName) != null)
                {
                    if(_groups.getGroup(newName) != _selectedGroup)
                    {
                        _groupFileNameField.setText(_selectedGroup.getName());
                        JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(GroupEditorPanel.this),
                                                      "A group with that name already exists.",
                                                      "Duplicate Name Not Allowed!",
                                                      JOptionPane.ERROR_MESSAGE);
                    }
                    else
                    {
                        //DO NOTHING!  The name is already set to new name, so do nothing.
                    }
                }
                //This is a valid changed name...
                else
                {
                    //Maintain the selection only if groups are renamed.
                    int selectedIndex = _groupTable.getSelectedRow();
                    _selectedGroup.setName(_groupFileNameField.getText().trim());
                    _selector.setTitle("Select Group Locations for " + _selectedGroup.getName() + ":");
                    _groupTable.getSelectionModel().setSelectionInterval(selectedIndex, selectedIndex);
                }
            }
        });
        _groupFileNameField.addFocusListener(new FocusAdapter()
        {
            @Override
            public void focusLost(FocusEvent e)
            {
                if(!_groupFileNameField.getText().trim().equals(_selectedGroup.getName()))
                {
                    _groupFileNameField.postActionEvent();
                }
            }
        });

        JButton addOnePerLocationButton = new SelfListeningButton("addMany20x20", "Create one group for each location")
        {
            @Override
            public void actionPerformed(ActionEvent e)
            {
                String prefix = (String)JOptionPane.showInputDialog(SwingTools.getGlobalDialogParent(GroupEditorPanel.this),
                                                                    "Provide prefix for all group zip file names:",
                                                                    "Group Name Prefix",
                                                                    JOptionPane.QUESTION_MESSAGE,
                                                                    null,
                                                                    null,
                                                                    "");
                if(prefix != null)
                {
                    if(!addGroupForEachIdentifier(prefix))
                    {
                        JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(GroupEditorPanel.this),
                                                      "No groups were added because groups by those names already exist.",
                                                      "No Groups Added",
                                                      JOptionPane.WARNING_MESSAGE);
                    }
                }
            }
        };
        JButton removeSelectedLocationsButton = new MultiItemButton("selectedRowDelete20x20", "Remove selected groups")
        {
            @Override
            public void actionPerformed(ActionEvent e)
            {
                int option = JOptionPane.showConfirmDialog(SwingTools.getGlobalDialogParent(GroupEditorPanel.this),
                                                           "Are you sure you want to remove all selected groups?",
                                                           "Confirm Removal",
                                                           JOptionPane.YES_NO_OPTION);
                if(option == JOptionPane.YES_OPTION)
                {
                    int[] indices = _groupTable.getSelectedRows();
                    _groups.removeAll(ListTools.retrieveAll(_groups, indices));
                }
            }
        };
        JButton removeAllButton = new SelfListeningButton("delete20x20", "Remove all groups")
        {
            @Override
            public void actionPerformed(ActionEvent e)
            {
                int option = JOptionPane.showConfirmDialog(SwingTools.getGlobalDialogParent(GroupEditorPanel.this),
                                                           "Are you sure you want to remove all groups?",
                                                           "Confirm Removal",
                                                           JOptionPane.YES_NO_OPTION);
                if(option == JOptionPane.YES_OPTION)
                {
                    _groups.clear();
                }
            }
        };
        _groupTable.getRowSelectionBus().register(removeSelectedLocationsButton);

        JToolBarBuilder toolBarBuilder = new JToolBarBuilder();
        toolBarBuilder.setFloatable(false);
        toolBarBuilder.add(addOnePerLocationButton);
        toolBarBuilder.add(removeSelectedLocationsButton);
        toolBarBuilder.add(removeAllButton);

        JScrollPane tableSP = new JScrollPane(_groupTable);
        tableSP.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);

        //Table panel
        JPanel leftPanel = new JPanel();
        leftPanel.setLayout(new BorderLayout());
        leftPanel.add(tableSP, BorderLayout.CENTER);
        leftPanel.add(toolBarBuilder.build(), BorderLayout.SOUTH);
        leftPanel.setMinimumSize(new Dimension(125, 10));
        leftPanel.setPreferredSize(new Dimension(125, 10));

        //Split pane
        _mainPane.setContinuousLayout(true);
        _mainPane.setLeftComponent(leftPanel);
        _mainPane.setDividerSize(5);
        _mainPane.setResizeWeight(0);

        this.setLayout(new BorderLayout());
        this.add(_mainPane, BorderLayout.CENTER);

        this.setBorder(HSwingFactory.createTitledBorder(BorderFactory.createEtchedBorder(), "Edit Zip Groups", null));
    }

    @Override
    @Subscribe
    public void reactToAvailableIdentifiersChanged(AvailableIdentifiersChangedNotice evt)
    {
        createSelector();
    }

    @Override
    @Subscribe
    public void reactToSelectedIdentifiersChanged(SelectedIdentifiersChangedNotice evt)
    {
        createSelector();
    }

}
