package ohd.hseb.hefs.pe.acceptance.group;

import java.io.File;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import ohd.hseb.hefs.pe.acceptance.AcceptedParameterFileHandler;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.notice.ZipGroupRenamedGroupNotice;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.ZipWriter;
import ohd.hseb.hefs.utils.jobs.JobMessenger;
import ohd.hseb.hefs.utils.notify.collect.CollectionModifiedNotice;
import ohd.hseb.hefs.utils.tools.FileTools;

import com.google.common.base.Predicates;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Sets;
import com.google.common.eventbus.Subscribe;
import com.google.common.io.Files;

/**
 * Handles the zip file for a single {@link ZipGroup}.
 * 
 * @author alexander.garbarino
 */
public class GroupZipFileHandler implements
CollectionModifiedNotice.Subscriber<LocationAndDataTypeIdentifier, ZipGroup>
{
    private final ParameterEstimatorRunInfo _runInfo;
    private final AcceptedParameterFileHandler _handler;

    private final ZipGroup _group;
    private File _zipFile;
    private final File _preparedDir;

    private Set<File> _requiredFiles;
    private Set<File> _missingRequiredFiles;
    private Set<ZipEntry> _presentEntries;
    private Set<ZipEntry> _updatedEntries;
    private Boolean _isMissingFiles;
    private Boolean _hasExtraFiles;
    private GroupZipStatus _status;

    public GroupZipFileHandler(ZipGroup group, ParameterEstimatorRunInfo runInfo)
    {
        _group = group;
        _group.register(this);
        _runInfo = runInfo;
        _handler = _runInfo.getAcceptedZipFileHandler();

        setZipFile();

        _preparedDir = new File(runInfo.getBaseDirectory().getAbsolutePath() + "/parameters");
    }

    private void setZipFile()
    {
        _zipFile = FileTools.newFile(_runInfo.getConfigDirectory().getAbsolutePath(),
                                     "ModuleDataSetFiles",
                                     "hefs",
                                     _group.getName() + ".zip");
    }

    public GroupZipStatus getStatus()
    {
        if(_status == null)
        {
            try
            {
                if(_group.isEmpty())
                {
                    _status = GroupZipStatus.NO_IDENTIFIERS;
                }
                else if(!exists())
                {
                    _status = GroupZipStatus.MISSING;
                }
                else if(isMissingFiles())
                {
                    _status = GroupZipStatus.INCOMPLETE;
                }
                else if(getUpdatedEntries().size() < getRequiredFiles().size())
                {
                    _status = GroupZipStatus.NEEDS_UPDATE;
                }
                else if(hasExtraFiles())
                {
                    _status = GroupZipStatus.EXTRA_FILES;
                }
                else
                {
                    _status = GroupZipStatus.COMPLETE;
                }
            }
            catch(IOException e)
            {
                _status = GroupZipStatus.ERROR;
            }
        }
        return _status;
    }

    public boolean exists()
    {
        return _zipFile.exists();
    }

    public Set<ZipEntry> getPresentEntries()
    {
        if(_presentEntries == null)
        {
            ZipFile zip = null;
            try
            {
                _presentEntries = Sets.newHashSet();
                zip = new ZipFile(_zipFile);

                Enumeration<? extends ZipEntry> iter = zip.entries();
                while(iter.hasMoreElements())
                {
                    _presentEntries.add(iter.nextElement());
                }
            }
            catch(IOException e)
            {
                _presentEntries = Sets.newHashSet();
            }
            finally
            {
                //CLose it!
                if(zip != null)
                {
                    try
                    {
                        zip.close();
                    }
                    catch(IOException e)
                    {
                    }
                }
            }
        }
        return _presentEntries;
    }

    public Set<File> getRequiredFiles()
    {
        if(_requiredFiles == null)
        {
            _requiredFiles = Sets.newHashSet();
            for(LocationAndDataTypeIdentifier identifier: _group)
            {
                _requiredFiles.addAll(_handler.getZipFileHandler(identifier).getRequiredFiles());
            }
        }
        return _requiredFiles;
    }

    public Set<File> getMissingRequiredFiles()
    {
        if(_missingRequiredFiles == null)
        {
            _missingRequiredFiles = Sets.filter(getRequiredFiles(), Predicates.not(FileTools.DOES_FILE_EXIST));
        }
        return _missingRequiredFiles;
    }

    public boolean doRequiredFilesExist()
    {
        return getMissingRequiredFiles().isEmpty();
    }

    /**
     * @return A {@link Set} of {@link ZipEntry} instances, each of which is not older than the parameter file that
     *         corresponds to it. In otherwords, a list of the entries that are fully up to date. Any parameter file not
     *         in this list is newer than what is in the zip file.
     */
    public Set<ZipEntry> getUpdatedEntries()
    {
        if(_updatedEntries == null)
        {
            _updatedEntries = Sets.newHashSet();
            try
            {
                Set<File> required = Sets.newHashSet(getRequiredFiles());

                // Test timestamps.
                for(ZipEntry entry: getPresentEntries())
                {
                    File matchingFile = null;
                    for(File reqFile: required)
                    {
                        if(FileTools.getRelativeFile(_preparedDir, reqFile).getPath().equals(entry.getName()))
                        {
                            matchingFile = reqFile;
                            break;
                        }
                    }
                    if(matchingFile == null)
                    {
                        continue;
                    }
                    required.remove(matchingFile);

                    //The time stored in the zip file is the time of the file on the file system when zipped.
                    //Sometimes, however, the file time will be minutely larger than the entry time due to system
                    //approximations used.  So, use the 1000 (1 second) factor to provide a bit of buffer room.
                    if((entry.getTime() != -1) && (entry.getTime() >= matchingFile.lastModified() - 2000))
                    {
                        _updatedEntries.add(entry);
                    }
                }
            }
            catch(Exception e)
            {
            }
        }
        return _updatedEntries;
    }

    private void computeFileStatus() throws IOException
    {
        Set<String> present = Sets.newHashSet();
        for(ZipEntry entry: getPresentEntries())
        {
            present.add(entry.getName());
        }

        Set<String> required = Sets.newHashSet();
        for(File file: getRequiredFiles())
        {
            required.add(FileTools.getRelativeFile(_preparedDir, file).getPath());
        }

        Iterator<String> reqIterator = required.iterator();
        while(reqIterator.hasNext())
        {
            String req = reqIterator.next();
            if(present.contains(req))
            {
                reqIterator.remove();
                present.remove(req);
            }
//            else
//            {
//                System.err.println("####>> FILE MISSING: " + req);
//            }
        }

        _isMissingFiles = !required.isEmpty();
        _hasExtraFiles = !present.isEmpty();
    }

    public boolean isMissingFiles() throws IOException
    {
        if(_isMissingFiles == null)
        {
            computeFileStatus();
        }
        return _isMissingFiles;
    }

    public boolean hasExtraFiles() throws IOException
    {
        if(_hasExtraFiles == null)
        {
            computeFileStatus();
        }
        return _hasExtraFiles;
    }

    /**
     * Forces this file handler to recheck all files.
     */
    public synchronized void resetMemory()
    {
        _requiredFiles = null;
        _missingRequiredFiles = null;
        _presentEntries = null;
        _updatedEntries = null;
        _isMissingFiles = null;
        _hasExtraFiles = null;
        _status = null;
    }

    public synchronized void resetIdentifiers()
    {
        for(LocationAndDataTypeIdentifier identifier: _group)
        {
            _handler.getZipFileHandler(identifier).resetMemory();
        }
    }

    public File getFile()
    {
        return _zipFile;
    }

    //This used to generate notifications and included a parameter indicating if the notifications should be part of a 
    //list or individual.  That notification stuff has been removed.

    public File prepare() throws Exception
    {
        //Zip file is already prepared.
        if(this.getStatus().isReady())
        {
            return _zipFile;
        }

        Files.createParentDirs(_zipFile);

        ZipWriter writer = null;
        boolean filesHaveBeenZipped = false;
        try
        {
            writer = new ZipWriter(_zipFile);
            File[] inputFiles = getRequiredFiles().toArray(new File[0]);

            JobMessenger.newMonitorSubJob();
            JobMessenger.setMaximumNumberOfSteps(inputFiles.length);

            for(int i = 0; i < inputFiles.length; i++)
            {
                try
                {
                    JobMessenger.updateNote("Zipping file: " + inputFiles[i].getAbsolutePath());
                    if(inputFiles[i].exists())
                    {
                        filesHaveBeenZipped = true;
                        writer.writeFile(inputFiles[i], FileTools.getRelativeFile(_preparedDir, inputFiles[i]));
                    }
                    JobMessenger.madeProgress();
                }
                catch(IOException e)
                {
                    // Do nothing.
                }
            }

            if(!filesHaveBeenZipped)
            {
                throw new Exception("Group '"
                    + _group.getName()
                    + "' parameters cannot be zipped because no required files can be found for any included locations.");
            }

            JobMessenger.clearMonitorSubJob();
        }
        finally
        {
            if(writer != null)
            {
                try
                {
                    writer.close();
                }
                catch(Exception e)
                {
                    //Will be thrown if nothing was added to the zip file.
                    //I want to handle exception due to no entries in zip file with the if check and throw above.
                }
            }
        }

        resetMemory();
        resetIdentifiers();

//If I include these notify prepared things, it may cause multithreading issues because the
//zip file handler resetMemory methods are called when the table updates, cause the getValueAt methods
//in the table model to null pointer except becase an underlying object is cleared while checking it.  
//
//NOTE: I've changed the notify prepared notices so that they do not extend the collection notices.
//I'm not sure if that will resolve this problem or not, as I have not had time to test.
//        if(makeListNotification)
//        {
//            _runInfo.getZipGroupInfo().notifyPrepared(this, _group);
//        }
//        _group.notifyPrepared(this);

        return _zipFile;
    }

    @Override
    @Subscribe
    public void reactToCollectionModified(CollectionModifiedNotice<LocationAndDataTypeIdentifier, ZipGroup> evt)
    {
        if(evt instanceof ZipGroupRenamedGroupNotice)
        {
            reactToRename((ZipGroupRenamedGroupNotice)evt);
        }
        else
        {
            resetMemory();
        }
    }

    /**
     * Renames the existing zip file to match the new file name.
     * 
     * @param evt
     */
    public void reactToRename(ZipGroupRenamedGroupNotice evt)
    {
        File oldFile = _zipFile;
        setZipFile();
        try
        {
            if(oldFile.exists())
            {
                FileTools.renameFileReplaceExisting(oldFile, _zipFile);
            }
        }
        catch(IOException e)
        {
            // No big problem if the move fails - just have to prepare again.
            e.printStackTrace();
        }
        resetMemory();
    }

    public static LoadingCache<ZipGroup, GroupZipFileHandler> makeCache(final ParameterEstimatorRunInfo runInfo)
    {
        return CacheBuilder.newBuilder().build(new CacheLoader<ZipGroup, GroupZipFileHandler>()
        {
            @Override
            public GroupZipFileHandler load(ZipGroup key) throws Exception
            {
                return new GroupZipFileHandler(key, runInfo);
            }
        });
    }
}
