package ohd.hseb.hefs.pe.acceptance.group;

import java.util.ArrayList;
import java.util.List;

import ohd.hseb.hefs.pe.acceptance.AcceptancePEStepProcessor;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.notice.StepUnitsUpdatedNotice;
import ohd.hseb.hefs.pe.notice.ZipGroupRenamedListNotice;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.collect.ElementListeningList;
import ohd.hseb.hefs.utils.notify.collect.CollectionModifiedNotice;
import ohd.hseb.hefs.utils.notify.collect.ElementsChangedNotice;
import ohd.hseb.hefs.utils.notify.collect.NotifyingListWrapper;
import ohd.hseb.hefs.utils.xml.CollectionXMLReader;
import ohd.hseb.hefs.utils.xml.CollectionXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReadable;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderFactory;
import ohd.hseb.hefs.utils.xml.XMLWritable;
import ohd.hseb.hefs.utils.xml.XMLWriter;

import com.google.common.collect.Lists;
import com.google.common.eventbus.Subscribe;

public class ZipGroupInfo extends NotifyingListWrapper<ZipGroup, List<ZipGroup>> implements XMLReadable, XMLWritable,
CollectionModifiedNotice.Subscriber<LocationAndDataTypeIdentifier, ZipGroup>
{
    public static final String XML_TAG = "groupInfo";
    public static final XMLReaderFactory<ZipGroupInfo> DEFAULT_FACTORY = new DefaultFactory();

    public ZipGroupInfo(ParameterEstimatorRunInfo runInfo)
    {
        super(ElementListeningList.create(new ArrayList<ZipGroup>()));
        ((ElementListeningList)delegate()).register(this);
        this.register(runInfo);
    }

    private ZipGroupInfo()
    {
        super(ElementListeningList.create(new ArrayList<ZipGroup>()));
        ((ElementListeningList)delegate()).register(this);
    }

    /**
     * Return a new <code>GroupInfo</code> containing all the groups in this which include <code>identifier</code>. The
     * new <code>GroupInfo</code> will not notify a bus of any changes (notify is disabled).
     * 
     * @param identifier the identifier to grab the groups of
     * @return all groups containing <code>identifier</code>
     */
    public ZipGroupInfo forIdentifier(LocationAndDataTypeIdentifier identifier)
    {
        ZipGroupInfo info = new ZipGroupInfo();
        info.setNotifyEnabled(false);
        for(ZipGroup group: this)
        {
            if(group.contains(identifier))
            {
                info.add(group);
            }
        }
        return info;
    }

    public void renameGroup(ZipGroup group, String newName)
    {
        renameGroup(group, newName, this);
    }

    public void renameGroup(ZipGroup group, String newName, Object source)
    {
        String oldName = group.getName();
        group.setName(newName, source);
        post(new ZipGroupRenamedListNotice(source, this, group, oldName));
    }

    public ZipGroup getGroup(String groupName)
    {
        for(ZipGroup group: this)
        {
            if(group.getName().equals(groupName))
            {
                return group;
            }
        }
        return null;
    }

    /**
     * Adds a group only if a group with its name does not yet exist.
     * 
     * @param group Group to add.
     * @return True if the group was added; false if not.
     */
    public boolean addGroupIfNotPresent(ZipGroup group)
    {
        if(getGroup(group.getName()) == null)
        {
            this.add(group);
            return true;
        }
        return false;
    }

//XXX These are not being called for now, but the code is left here possibly for future testing.
//These calls apparently led to multithreading/sync issue when a table was redrawn while 
//resetMemory was being called for zip group instances.  
//    public void notifyPrepared(Object source, ZipGroup group)
//    {
//        post(new ZipGroupPreparedListNotice(source, this, Lists.<ZipGroup>newArrayList(group)));
//    }
//
//    public void notifyPrepared(Object source, LocationAndDataTypeIdentifier identifier)
//    {
//        post(new ZipGroupPreparedListNotice(source, this, this.forIdentifier(identifier)));
//    }

    @Override
    @Subscribe
    public void reactToCollectionModified(CollectionModifiedNotice<LocationAndDataTypeIdentifier, ZipGroup> evt)
    {
        post(new ElementsChangedNotice(evt.getSource(), this, Lists.<ZipGroup>newArrayList(evt.getCollection())));
        post(new StepUnitsUpdatedNotice(evt.getSource(), AcceptancePEStepProcessor.class, evt.getTarget()));
    }

    public String getXMLTagName()
    {
        return XML_TAG;
    }

    @Override
    public XMLWriter getWriter()
    {
        return new CollectionXMLWriter(getXMLTagName(), this);
    }

    @Override
    public XMLReader getReader()
    {
        return new CollectionXMLReader<ZipGroup>(getXMLTagName(), this, ZipGroup.DEFAULT_FACTORY);
    }

    private static class DefaultFactory implements XMLReaderFactory<ZipGroupInfo>
    {
        @Override
        public ZipGroupInfo get()
        {
            return new ZipGroupInfo();
        }
    }
}
