package ohd.hseb.hefs.pe.core;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;

import ohd.hseb.hefs.pe.acceptance.AcceptedParameterFileHandler;
import ohd.hseb.hefs.pe.acceptance.group.ZipGroupInfo;
import ohd.hseb.hefs.pe.estimation.EstimatedParametersFileHandler;
import ohd.hseb.hefs.pe.estimation.EstimationLogFileHandler;
import ohd.hseb.hefs.pe.estimation.GenericEstimationPEStepProcessor;
import ohd.hseb.hefs.pe.estimation.options.ControlOption;
import ohd.hseb.hefs.pe.estimation.options.EstimationControlOptions;
import ohd.hseb.hefs.pe.model.ParameterEstimationModel;
import ohd.hseb.hefs.pe.notice.AvailableIdentifiersChangedNotice;
import ohd.hseb.hefs.pe.notice.SelectedIdentifiersChangedNotice;
import ohd.hseb.hefs.pe.notice.StepUpdatedNotice;
import ohd.hseb.hefs.pe.sources.ForecastSource;
import ohd.hseb.hefs.pe.sources.SourceDataHandler;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifierList;
import ohd.hseb.hefs.utils.VariableSetNotice;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.notify.NoticeForwarder;
import ohd.hseb.hefs.utils.notify.NoticePoster;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tools.ParameterId.Type;
import ohd.hseb.hefs.utils.xml.CompositeXMLReader;
import ohd.hseb.hefs.utils.xml.CompositeXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReadable;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLWritable;
import ohd.hseb.hefs.utils.xml.XMLWriter;

import org.apache.commons.io.FileUtils;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.eventbus.Subscribe;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;

/**
 * Objects required for the ParameterEstimator to run. Two types of information should be stored: objects to be used to
 * run the parameter estimator, including models, data handlers, and general file handlers. The second type of
 * information are settings for the program, including: working identifiers, working control parameters, and others as
 * needed.<br>
 * <br>
 * By default the following storage objects are provided: base directory, configuration directory, system files
 * directory, source data handlers list, working identifiers list, and working control parameters. The XML reading
 * portion of this class outputs the working identifiers and control parameters, in order.<br>
 * <br>
 * 
 * @author hank.herr
 */
public abstract class ParameterEstimatorRunInfo extends NoticeForwarder implements XMLReadable, XMLWritable,
NoticePoster
{
    private static final Logger LOG = LogManager.getLogger(ParameterEstimatorRunInfo.class);

    private final ListeningExecutorService _executorService;

    private final File _baseDirectory;
    private final File _configDirectory;
    private final File _systemFilesDirectory;

    /**
     * The forecast sources used for this parameter estimator.
     */
    private final LinkedHashMap<String, ForecastSource> _forecastSources = new LinkedHashMap<>();

    /**
     * This list must always be ordered according to how the model parameters are ordered in the parameters file.
     */
//    private final List<SourceDataHandler> _sourceDataHandlers = Lists.newArrayList();

    /**
     * Records the LocationAndDataTypeIdentifier instances that summarize the working identifiers, including mappings
     * created by users. This is the master list for the list displayed in the locations summary table.
     */
    private final LocationAndDataTypeIdentifierList _availableIdentifiers = new LocationAndDataTypeIdentifierList();

    /**
     * Records the current {@link EstimationControlOptions}, including changes made by users, by parameter type:
     * precipitation or temperature.
     */
    private final Map<ParameterId.Type, EstimationControlOptions> _estimationControlOptions;

    /**
     * Map stores the default options found in the jar file default run-time info.
     */
    private final Map<ParameterId.Type, EstimationControlOptions> _defaultOptionsMap = Maps.newHashMap();

    /**
     * Handler for reading/writing with estimated parameters.
     */
    private EstimatedParametersFileHandler _estimatedParametersFileHandler;

    /**
     * A handler for the estimation log files.
     */
    private EstimationLogFileHandler _estimationLogFileHandler;

    /**
     * Handler for creating zipped parameter files.
     */
    private AcceptedParameterFileHandler _acceptedZipFileHandler;

    /**
     * Keeps track of output zip group specifications.
     */
    private final ZipGroupInfo _zipGroupInfo;

    /**
     * The currently selected data type for execution.
     */
    private ParameterId.Type _selectedDataType;

    /**
     * The saving thread.
     */
    private RunInfoSaveThread _saveThread;

    /**
     * Do nothing constructor must ONLY be called directly but subclasses when reading XML. Do NOT call this for any
     * other purpose.
     */
    protected ParameterEstimatorRunInfo()
    {
        _zipGroupInfo = new ZipGroupInfo(this);
        _systemFilesDirectory = null;
        _executorService = null;
        _estimationControlOptions = Maps.newLinkedHashMap();
        _configDirectory = null;
        _baseDirectory = null;

        initializeXMLReaders();
    }

    /**
     * Standard constructor to call for operationally usable run-time info.
     * 
     * @param executor {@link ExecutorService} used to initialize info from system files.
     * @param baseDirectory The base directory for the parameter estimator.
     * @param configDirectory The config directory corresponding to the stand-alone running the paramter estimator.
     * @throws Exception
     */
    public ParameterEstimatorRunInfo(final ExecutorService es, final File baseDirectory, final File configDirectory) throws Exception
    {
        _executorService = MoreExecutors.listeningDecorator(es);

        _baseDirectory = baseDirectory;
        _systemFilesDirectory = new File(baseDirectory.getAbsolutePath() + File.separator + ".systemFiles");
        _configDirectory = configDirectory;

        _zipGroupInfo = new ZipGroupInfo(this);

        // Transform a change in available identifiers to a change in active identifiers.
        this.register(new AvailableIdentifiersChangedNotice.Subscriber()
        {
            @Override
            @Subscribe
            public void reactToAvailableIdentifiersChanged(final AvailableIdentifiersChangedNotice evt)
            {
                post(new SelectedIdentifiersChangedNotice(evt, _selectedDataType, getSelectedIdentifiers()));
            }
        });

        initializeSourcesHandlersControlOptionsFileAndAvailableIdentifiers();

        // After handlers are initialized.

        // Changed control parameters -> Estimation step updated.
        final VariableSetNotice.Subscriber subscriber = new VariableSetNotice.Subscriber()
        {
            @Override
            @Subscribe
            public void reactToVariableSet(final VariableSetNotice notice)
            {
                post(new StepUpdatedNotice(notice, GenericEstimationPEStepProcessor.class));
            }
        };
        _estimationControlOptions = Maps.newLinkedHashMap();
        for(final ParameterId.Type type: getSupportedDataTypes())
        {
            final EstimationControlOptions ecp = constructEstimationControlOptions(type);
            ecp.register(subscriber);
            _estimationControlOptions.put(type, ecp);

        }

        initializeXMLReaders();
        loadWorkingRunInfoFromStandardFile();

        setSelectedDataType(getSupportedDataTypes().iterator().next());
    }

    public ListeningExecutorService getExecutor()
    {
        return _executorService;
    }

    /**
     * @return the set of data types that are supported.
     */
    public abstract EnumSet<ParameterId.Type> getSupportedDataTypes();

    /**
     * Create a new set of estimation control parameters for {@code type}.
     * 
     * @param type the type of control parameters to create
     * @return the new control parameters
     */
    public abstract EstimationControlOptions constructEstimationControlOptions(ParameterId.Type type);

    public abstract String getDefaultRunInformationXMLFile();

    /**
     * This loads the default run time options, first. It then records the estimation control options and loads the
     * working (system) run time file. It then uses the default estimation control options to setup default options for
     * the estimation control options read from the run time file. This feature has not been thoroughly tested; i.e., I
     * have not changed the default options and made sure it gets into the GUI via the default button.
     * 
     * @throws Exception For various reasons.
     */
    private void loadWorkingRunInfoFromStandardFile() throws Exception
    {
        //Load the default run time information into this and record the estimation options.  Those options
        //will be used to set the default values for the working estimation options.
        InputStream stream = ClassLoader.getSystemResource(getDefaultRunInformationXMLFile()).openStream();
        if(stream == null)
        {
            throw new FileNotFoundException("No default runtime information found as system resource: "
                + getDefaultRunInformationXMLFile());
        }
        XMLTools.readXMLFromStreamAndClose(stream, false, this); //Closes the stream for me!
        for(final ParameterId.Type type: _estimationControlOptions.keySet())
        {
            _defaultOptionsMap.put(type, _estimationControlOptions.get(type).clone());
        }
        LOG.info("Reading default run-time information for setting defaults...");

        //Now load the working control options from either the system file or use the defaults, again.
        final File toLoad = new File(_systemFilesDirectory, "runTimeInformation.xml");
        if(toLoad.exists())
        {
            XMLTools.readXMLFromFile(toLoad, this);
            LOG.info("Reading saved run-time information ...");
        }
        else
        {
//This code failed on Linux because the URI was not hierarchical.
//            toLoad = new File(ClassLoader.getSystemResource(getDefaultRunInformationXMLFile()).toURI());
//            if(!toLoad.exists())
//            {
//                throw new FileNotFoundException("No default runtime information found.");
//            }

            stream = ClassLoader.getSystemResource(getDefaultRunInformationXMLFile()).openStream();
            if(stream == null)
            {
                throw new FileNotFoundException("No default runtime information found as system resource: "
                    + getDefaultRunInformationXMLFile());
            }
            XMLTools.readXMLFromStreamAndClose(stream, false, this); //Closes the stream for me!
            LOG.info("Reading default run-time information since no saved information was found ...");
        }

        //From the initial reading of the default options, set the default values for the working control options.
        copyDefaultEstimationOptionsIntoWorkingOptions();

        LOG.info("Done reading run-time information.");
    }

    /**
     * The attribute {@link #_defaultOptionsMap} is populated at start-up with default values loaded from the jar file.
     * This map stores those values so that, if the {@link #_estimationControlOptions} are loaded from a parameter file,
     * the defaults can be properly recovered afterwards.
     */
    public void copyDefaultEstimationOptionsIntoWorkingOptions()
    {
        //From the initial reading of the default options, set the default values for the working control options.
        for(final ParameterId.Type type: _estimationControlOptions.keySet())
        {
            _estimationControlOptions.get(type).setDefaultValue(_defaultOptionsMap.get(type).get());
        }
    }

    /**
     * Write the run info XML file to a temporary location and copy over top of the permanent location once done
     * writing.
     */
    public synchronized void saveWorkingRunInfoToStandardFile() throws Exception
    {
        final File tmpFile = new File(_baseDirectory + "/.systemFiles/tmp.runTimeInformation.xml");
        final File bakFile = new File(_baseDirectory + "/.systemFiles/runTimeInformation.xml.bak");
        final File standardFile = new File(_baseDirectory + "/.systemFiles/runTimeInformation.xml");
        if((tmpFile.exists()) && (!tmpFile.canWrite()))
        {
            throw new IOException("Temporary run-time information file exists, but cannot be written to: "
                + tmpFile.getAbsolutePath());
        }
        XMLTools.writeXMLFileFromXMLWriter(tmpFile, this, true);
        if(standardFile.exists())
        {
            if(!standardFile.delete())
            {
                throw new IOException("Standard file, " + standardFile.getAbsolutePath()
                    + ", cannot be deleted; leaving temporary file in place.");
            }
        }
        if(!tmpFile.renameTo(standardFile))
        {
            throw new IOException("Temporary run-time information file was written, but cannot be renamed to: "
                + standardFile.getAbsolutePath() + "; leaving temporary file in place.");
        }
        try
        {
            FileUtils.copyFile(standardFile, bakFile);
        }
        catch(final IOException e)
        {
            throw new IOException("Cannot create a backup file " + tmpFile.getAbsolutePath() + " for: "
                + standardFile.getAbsolutePath() + ".");
        }
    }

    protected abstract void initializeSourcesHandlersControlOptionsFileAndAvailableIdentifiers() throws Exception;

    public LocationAndDataTypeIdentifierList getSelectedIdentifiers()
    {
        return getCurrentlyWorkingIdentifiers().ofType(_selectedDataType);
    }

    public ParameterId.Type getSelectedDataType()
    {
        return _selectedDataType;
    }

    /**
     * Sets the currently selected data type, and sends out a {@link SelectedIdentifiersChangedNotice} to all listeners.
     * 
     * @param type the new data type
     */
    public void setSelectedDataType(final ParameterId.Type type)
    {
        setSelectedDataType(type, this);
    }

    /**
     * Sets the currently selected data type, and sends out a {@link SelectedIdentifiersChangedNotice} to all listeners.
     * 
     * @param type the new data type
     * @param source the object triggering this change
     */
    public void setSelectedDataType(final ParameterId.Type type, final Object source)
    {
        _selectedDataType = type;
        post(generateDataTypeSelectedEvent(source));
    }

    @Override
    public void post(final Object event)
    {
        super.post(event);
    }

    /**
     * Creates a {@link SelectedIdentifiersChangedNotice} to match the current selected data type.
     * 
     * @param source the source of the event
     * @return a new event matching this object's current state
     */
    public SelectedIdentifiersChangedNotice generateDataTypeSelectedEvent(final Object source)
    {
        return new SelectedIdentifiersChangedNotice(source, _selectedDataType, getSelectedIdentifiers());
    }

    /**
     * @return An {@link ArrayList} of the {@link SourceDataHandler} instances returned by
     *         {@link ForecastSource#getSourceDataHandler()} for each forecast source in {@link #_forecastSources}, in
     *         order.
     */
    public List<? extends SourceDataHandler> getSourceDataHandlers()
    {
        final List<SourceDataHandler> handlers = new ArrayList<>();
        for(final ForecastSource source: _forecastSources.values())
        {
            handlers.add(source.getSourceDataHandler());
        }
        return handlers;
    }

    /**
     * @return The list of {@link ForecastSource} instances associated with this PE.
     */
    public List<? extends ForecastSource> getForecastSources()
    {
        return Lists.newArrayList(_forecastSources.values());
    }

    /**
     * @return The {@link ForecastSource} whose {@link ForecastSource#getSourceId()} matches the provided id.
     */
    public ForecastSource getForecastSource(final String id)
    {
        return _forecastSources.get(id);
    }

    /**
     * Useful for test purposes primarily, it allows for the run information to be loaded normally, but then the sources
     * are set to something non-standard. Use in conjunction with {@link #addForecastSource(ForecastSource)}.
     */
    public void clearForecastSources()
    {
        _forecastSources.clear();
    }

    /**
     * Adds the source to {@link #_forecastSources}, initializes a source data handler, and adds it to
     * {@link #_sourceDataHandlers}. It also calls {@link SourceDataHandler#register(Object)} with this so that this can
     * react to events from the handler.
     * 
     * @param source The source to add.
     * @throws Exception If the data handler cannot be initialized for some reason.
     */
    public void addForecastSource(final ForecastSource source) throws Exception
    {
        _forecastSources.put(source.getSourceId(), source);
        source.initializeSourceDataHandler(_baseDirectory, this);
        source.getSourceDataHandler().register(this);
    }

    public EstimationControlOptions getEstimationControlOptions(final ParameterId.Type type)
    {
        return _estimationControlOptions.get(type);
    }

    public EstimationControlOptions getEstimationControlOptions(final LocationAndDataTypeIdentifier identifier)
    {
        return getEstimationControlOptions(identifier.getParameterIdType());
    }

    /**
     * @return Pulls the {@link EstimationControlOptions} from {@link #_estimationControlOptions} for the provided
     *         {@link Type} and then calls
     *         {@link EstimationControlOptions#getControlOptions(ohd.hseb.hefs.pe.estimation.options.ControlOptionSupplier)}
     *         for the provided source. The results are returned.
     */
    public ControlOption getSourceControlOptions(final ParameterId.Type type, final ForecastSource source)
    {
        return getEstimationControlOptions(type).getControlOptions(source);
    }

    public EstimationLogFileHandler getEstimationLogFileHandler()
    {
        return _estimationLogFileHandler;
    }

    public void setEstimationLogFileHandler(final EstimationLogFileHandler estimationLogFileHandler)
    {
        _estimationLogFileHandler = estimationLogFileHandler;
    }

    public abstract ParameterEstimationModel getModel(LocationAndDataTypeIdentifier identifier);

    public File getBaseDirectory()
    {
        return _baseDirectory;
    }

    public File getConfigDirectory()
    {
        return _configDirectory;
    }

    public File getSystemFilesDirectory()
    {
        return _systemFilesDirectory;
    }

    public File getHelpDirectory()
    {
        return new File(_systemFilesDirectory, "help");
    }

    public LocationAndDataTypeIdentifierList getAvailableIdentifiers()
    {
        return _availableIdentifiers;
    }

    public EstimatedParametersFileHandler getEstimatedParametersFileHandler()
    {
        return _estimatedParametersFileHandler;
    }

    public void setEstimatedParametersFileHandler(final EstimatedParametersFileHandler estimatedParametersFileHandler)
    {
        _estimatedParametersFileHandler = estimatedParametersFileHandler;
    }

    public EstimatedParametersFileHandler getEstimatedParametersBackupFileHandler()
    {
        return _estimatedParametersFileHandler.getBackupHandler();
    }

    public ZipGroupInfo getZipGroupInfo()
    {
        return _zipGroupInfo;
    }

    public AcceptedParameterFileHandler getAcceptedZipFileHandler()
    {
        return _acceptedZipFileHandler;
    }

    public void setAcceptedZipFileHandler(final AcceptedParameterFileHandler acceptedZipFileHandler)
    {
        this._acceptedZipFileHandler = acceptedZipFileHandler;
    }

    /**
     * @return List of identifiers currently allowed to be worked on based on data availability. The returned list
     *         should be all possibly locations for all possible data types for which data (typically, historical data)
     *         is available for parameter estimation.
     */
    public abstract LocationAndDataTypeIdentifierList getCurrentlyWorkingIdentifiers();

    /**
     * @return The name of the program to place in the about dialog.
     */
    public abstract String getProgramNameForAboutDialog();

    public String getXMLTagName()
    {
        return "parameterEstimationRunInformation";
    }

    /**
     * Call to notify the run info that available identifiers have changed.
     */
    public void notifyAvailableIdentifiersChanged(final Object source)
    {
        updateAvailableIdentifiers();
        post(new AvailableIdentifiersChangedNotice(source, getAvailableIdentifiers()));
    }

    /**
     * Should ensure that the available identifiers returned by {@link #getAvailableIdentifiers()} match whatever data
     * is available.
     */
    public abstract void updateAvailableIdentifiers();

    /**
     * Called just before the reading is done.
     */
    protected abstract void initializeXMLReaders();

    @Override
    public CompositeXMLReader getReader()
    {
        final List<XMLReadable> components = Lists.newArrayList();
        components.add(_zipGroupInfo);
        components.add(_availableIdentifiers);
        components.addAll(_estimationControlOptions.values());
        return new CompositeXMLReader(getXMLTagName(), components);
    }

    @Override
    public XMLWriter getWriter()
    {
        final List<XMLWritable> components = Lists.newArrayList();
        components.add(_availableIdentifiers);
        components.addAll(_estimationControlOptions.values());
        components.add(_zipGroupInfo);
        return new CompositeXMLWriter(getXMLTagName(), components);
    }

    /**
     * Starts a thread to save the run-time information to the standard location.
     */
    public void startSaveThread()
    {
        _saveThread = new RunInfoSaveThread();
        _saveThread.startJob();
    }

    public void killSaveThread()
    {
        if(_saveThread != null)
        {
            _saveThread.setDone(true);
        }
    }

    public class RunInfoSaveThread extends GenericJob
    {
        @Override
        public void processJob()
        {
            while(true)
            {
                try
                {
                    Thread.sleep(60000);
                    if(isCanceled() || isDone())
                    {
                        break;
                    }
                    saveWorkingRunInfoToStandardFile();
                    LOG.info(this.getClass().getSimpleName() + ": run-time information saved.");
                }
                catch(final Exception e)
                {
                    LOG.error(this.getClass().getSimpleName() + ": failed to save run-time information: "
                        + e.getMessage());
                }
            }
        }

    }
}
