package ohd.hseb.hefs.pe.core;

import java.awt.Component;
import java.util.List;

import ohd.hseb.hefs.pe.estimation.GenericEstimationPEStepProcessor;
import ohd.hseb.hefs.pe.gui.ParameterEstimatorStepsPanel;
import ohd.hseb.hefs.utils.status.StatusIndicator;

/**
 * Interface implemented by classes devoted to performing a single step in a parameter estimation process. Generally
 * speaking this includes forecast data source gathering/validating steps, as well as setup, estimation, and acceptance
 * steps.<br>
 * <br>
 * Note that for data-source based steps, implementors of this interface often works closely with a SourceDataHandler.
 * 
 * @author hank.herr
 */
public interface ParameterEstimatorStepProcessor
{
    /**
     * @return An instance of ParameterEstimatorOptionsPanel that can be used to validate data and/or specify
     *         input/options for the step to be performed.
     */
    public ParameterEstimatorStepOptionsPanel constructOptionsPanel();

    /**
     * CAlled by {@link ParameterEstimatorStepsPanel}.
     * 
     * @param units The units for which estimation will be performed.
     * @return The {@link Component} to use to confirm if a run should be done. This is overridden in
     *         {@link GenericParameterEstimatorStepProcessor} to simply list the units. However, it can be used to
     *         specify run-options, as is done in {@link GenericEstimationPEStepProcessor}, in which it asks if the
     *         parameters file will be backed up.
     */
    public Component constructConfirmRunPanel(List<? extends StepUnit> units);

    /**
     * Get the status for {@code unit}.
     * 
     * @param unit the unit to get the status for
     * @return the status for the given unit
     */
    public StatusIndicator getStatus(StepUnit unit);

    /**
     * Prepare the step for running. This is called before the step is executed only one time for all units to be
     * processed.
     * 
     * @param skipUserInput Indicates if user input for the step preparation should be skipped.
     * @return if the preparation succeeded or not
     * @throws Exception if the preparation fails
     */
    public void prepareStep(boolean skipUserInput) throws Exception;

    /**
     * Perform the step, either data preparation or parameter estimation, for the given identifier. The step is
     * performed one location at a time, but this routine is called within a loop that passes through all locations
     * selected for estimation.
     * 
     * @param unit Unit of location and data type for which step is to be performed.
     * @throws Exception
     */
    public void performStep(StepUnit unit) throws Exception;

    /**
     * Called after a job is completed. This may mean after running for a single location or after running for many
     * locations. The cleanup should not be location specific and will always be called, even if an exception occurred
     * while running the step.
     * 
     * @param jobRunningStep The job that ran the step, if that's useful.
     */
    public void cleanupAfterStep();

    /**
     * @return Text describing the step which will be displayed as a tool tip over the tab and as part of the run
     *         button. If this is null AND {@link #getStepNameForRunButton()}, then the step will not include any run
     *         button at all.
     */
    public String getToolTipTextDescribingStep();

    /**
     * @return The name to put in a tab.
     */
    public String getTabNameForStep();

    /**
     * @return The name to use for the run button for this step. If null is returned, no run button will be displayed
     *         and the performStep method will never be called.
     */
    public String getStepNameForRunButton();

    /**
     * @return A short string (a few letters) to display at the top of the column in the LocationSummaryTablePanel.
     */
    public String getShortNameForIdentifierTableColumn();

    /**
     * @return A tool tip to apply to the column header.
     */
    public String getToolTipTextForIdentifierTableColumnHeader();

    /**
     * @return HTML instructions to be displayed in a help dialog.
     */
    public String getInstructions();

    /**
     * @return A string to be printed at the beginning of performing a step for each location. It appears in the perform
     *         step dialog and is usually followed by the word "for" and either the number of locations or location
     *         info.
     */
    public String getPerformStepPrefix();

    /**
     * @return String to display by default, usually getStepNameForRunButton().
     */
    @Override
    public String toString();

    /**
     * @return if this step can be run for several identifiers at once.
     */
    public boolean concurrencySupported();

}
