package ohd.hseb.hefs.pe.estimation;

import java.io.InputStream;
import java.util.Calendar;
import java.util.List;

import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.ParameterType;
import nl.wldelft.util.timeseries.SimpleEquidistantTimeStep;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.charter.ChartEngine;
import ohd.hseb.charter.datasource.DefaultXYChartDataSource;
import ohd.hseb.charter.parameters.SeriesDrawingParameters;
import ohd.hseb.graphgen.GraphGenSettingsController;
import ohd.hseb.graphgen.arguments.GraphGenArgumentsProcessor;
import ohd.hseb.graphgen.core.GraphicsGeneratorEngine;
import ohd.hseb.graphgen.core.GraphicsGeneratorProductParameters;
import ohd.hseb.graphgen.references.ReferencedTemplateLoader;
import ohd.hseb.graphgen.references.ReferencedTemplateParameters;
import ohd.hseb.hefs.pe.gui.ModelParameterDiagnosticDisplay;
import ohd.hseb.hefs.pe.model.FullModelParameters;
import ohd.hseb.hefs.pe.model.ModelParameterType;
import ohd.hseb.hefs.pe.model.OneTypeParameterValues;
import ohd.hseb.hefs.pe.sources.SourceModelParameters;
import ohd.hseb.hefs.utils.xml.GenericXMLReadingHandler;
import ohd.hseb.util.misc.HCalendar;

/**
 * This diagnostic display is built based on the system resource acceptanceParameterDisplayDiagnostics.xml under
 * mefppe/diagnostics. It is a Graphics Generator product which displays a single parameter. For all parameters to
 * display, that one product is built, and then they are put together to form a single overall chart, with each
 * parameter displayed in a subplot.<br>
 * <br>
 * To modify the base product, run the GraphGenForDiagnosticsEditingRunner under ohd.hseb.mefp.tools as a java app.
 * Select the product you want to edit under nonsrc/mefppe/diagnostics, and modify it. When done, copy the parameters
 * created and viewed via the Output panel preview button (at the bottom) into the product by hand. Add any needed
 * required arguments to the execute parameter section of the XML.<br>
 * <br>
 * Instructions for overriding the settings based on model parameter type can be provided within subclasses of
 * ModelParameterType. When multiple parameters are displayed on one chart, the chart modification methods are called in
 * order of parameter type.
 * 
 * @author hank.herr
 */
public class DefaultModelParameterDiagnosticDisplay implements ModelParameterDiagnosticDisplay
{

    private final SourceModelParameters _selectedSourceParameters;

    public DefaultModelParameterDiagnosticDisplay(final FullModelParameters fullParameters,
                                                  final SourceModelParameters selectedSourceParameters)
    {
        _selectedSourceParameters = selectedSourceParameters;
    }

    /**
     * @param selectedParameterIndex
     * @return A TimeSeriesArrays object containing the parameter values for a selected parameter (specified by index)
     *         as an ensemble with each member corresponding to a canonical event.
     */
    private TimeSeriesArrays buildTimeSeriesOfSelectedParameter(final ModelParameterType parameterType)
    {
        final int numberOfEnsembleMembers = _selectedSourceParameters.getNumberOfEnsembleMembersForDiagnosticDisplay();

        //Build the time series: one source with on time series per canonical event, each spanning one year (1900).
        final TimeSeriesArrays displayedTimeSeries = new TimeSeriesArrays(DefaultTimeSeriesHeader.class,
                                                                          numberOfEnsembleMembers);
        for(int memIndex = 0; memIndex < numberOfEnsembleMembers; memIndex++)
        {
            final TimeSeriesArray outputForEvent = new TimeSeriesArray(new DefaultTimeSeriesHeader());
            prepareParameterDiagnosticTimeSeriesHeader(outputForEvent, memIndex);

            final Calendar cal = HCalendar.convertStringToCalendar("1999-01-01 00:00:00", HCalendar.DEFAULT_DATE_FORMAT);
            while(cal.before(HCalendar.convertStringToCalendar("2000-01-01 00:00:00", HCalendar.DEFAULT_DATE_FORMAT)))
            {
                final OneTypeParameterValues values = _selectedSourceParameters.getParameterValues(parameterType);
                outputForEvent.putValue(cal.getTimeInMillis(),
                                        (float)values.getValue(cal.get(Calendar.DAY_OF_YEAR), memIndex));
                cal.add(Calendar.DAY_OF_YEAR, 1);
            }
            displayedTimeSeries.add(outputForEvent);
        }
        return displayedTimeSeries;
    }

    private void prepareParameterDiagnosticTimeSeriesHeader(final TimeSeriesArray output, final int ensembleIndex)
    {
        final DefaultTimeSeriesHeader editableHeader = (DefaultTimeSeriesHeader)output.getHeader();

        editableHeader.setForecastTime(0L);
        editableHeader.setLocationDescription("");
        editableHeader.setLocationId("DIAGNOSTIC");
        editableHeader.setLocationName("DIAGNOSTIC");

        editableHeader.setParameterId("STAT");
        editableHeader.setParameterName("STAT");
        editableHeader.setParameterType(ParameterType.INSTANTANEOUS);

        editableHeader.setEnsembleId("EVENTS");
        editableHeader.setEnsembleMemberIndex(ensembleIndex);

        editableHeader.setTimeStep(SimpleEquidistantTimeStep.getInstance(HCalendar.MILLIS_IN_HR * 24));
    }

    private ReferencedTemplateLoader constructTemplateLoaderForSelectedParameter(final GraphGenArgumentsProcessor arguments,
                                                                                 final ModelParameterType parameterType,
                                                                                 final int subPlotIndex) throws Exception
    {
        final int numberOfEnsembleMembers = _selectedSourceParameters.getNumberOfEnsembleMembersForDiagnosticDisplay();

        //Build the time series: one source with on time series per canonical event, each spanning one year (1900).
        final TimeSeriesArrays displayedTimeSeries = buildTimeSeriesOfSelectedParameter(parameterType);

        //Load GraphGen product parameters from a resource
        final GraphicsGeneratorProductParameters parameters = new GraphicsGeneratorProductParameters();
        final GenericXMLReadingHandler reader = new GenericXMLReadingHandler(parameters);
        final String resourceName = "mefppe/diagnostics/defaultParameterDisplayDiagnostics.xml";
        final InputStream stream = ClassLoader.getSystemResourceAsStream(resourceName);
        if(stream == null)
        {
            throw new Exception("Unable to find diagnostic display resource with name " + resourceName);
        }
        reader.readXMLFromStreamAndClose(stream, false);

        //Set the subplot index to the desired index.
        parameters.getTemplateParameters()
                  .getChartDrawingParameters()
                  .getDataSourceParameters(0)
                  .setSubPlotIndex(subPlotIndex);
        parameters.getTemplateParameters()
                  .getChartDrawingParameters()
                  .moveSubPlotAndAllSeriesDrawnOnIt(0, subPlotIndex);

        //Construct series parameters.  It is assumed that there is one instance of SeriesDrawingParameters defined
        //for the product which we use to construct series parameters for all series, copying all overridden parameters
        //and setting the legend entry manually.
        final SeriesDrawingParameters seriesParms = parameters.getTemplateParameters()
                                                              .getChartDrawingParameters()
                                                              .getDataSourceParameters()
                                                              .get(0)
                                                              .getSeriesDrawingParametersForSeriesIndex(0);
        parameters.getTemplateParameters()
                  .getChartDrawingParameters()
                  .getDataSourceParameters()
                  .get(0)
                  .removeAllSeriesDrawingParameters();
        for(int evtIndex = 0; evtIndex < numberOfEnsembleMembers; evtIndex++)
        {
            final SeriesDrawingParameters newSeriesParms = new SeriesDrawingParameters(evtIndex);
            newSeriesParms.copyOverriddenParameters(seriesParms);

            newSeriesParms.setNameInLegend(_selectedSourceParameters.getLegendEntryForEnsembleMember(evtIndex));
            if(subPlotIndex > 0)
            {
                newSeriesParms.setShowInLegend(false);
            }
            parameters.getTemplateParameters()
                      .getChartDrawingParameters()
                      .getDataSourceParameters()
                      .get(0)
                      .addSeriesDrawingParameters(newSeriesParms);
        }

        //If there are no ensemble members (i.e., only one ts)...
        if(numberOfEnsembleMembers == 1)
        {
            parameters.getTemplateParameters().getChartDrawingParameters().getLegend().setVisible(false);
        }

        //Override based on model parameter type.
        parameterType.modifyDiagnosticDisplayParameters(parameters.getTemplateParameters().getChartDrawingParameters());

        //Remove input series providers.  I'm going to provide time series manually.
        parameters.getTemplateParameters().getInputSeriesProviderParameters().clear();

        //Determine parameter name displayed on y-axis.
        String yAxisLabel = parameterType.getName();
        if(parameterType.getUnit() != null)
        {
            yAxisLabel += " [" + parameterType.getUnit() + "]";
        }

        //Build ReferencedTemplateParameters.
        final ReferencedTemplateParameters refParms = new ReferencedTemplateParameters();
        refParms.setTemplateId("diag" + subPlotIndex);
        refParms.addParameter("displayedParameterName", yAxisLabel);
        refParms.addParameter("displayedLocationId", _selectedSourceParameters.getIdentifier().getLocationId());
        refParms.addParameter("displayedParameterId", _selectedSourceParameters.getIdentifier().getParameterId());
        refParms.addParameter("displayedForecastSource", _selectedSourceParameters.getForecastSource().getName());

        //Build the loader and specify the time series it will be using.
        final ReferencedTemplateLoader loader = new ReferencedTemplateLoader(arguments, refParms);
        loader.setLoadedReferenceParameters(parameters.getTemplateParameters());
        loader.useExternallyLoadedTimeSeries(displayedTimeSeries);
        return loader;
    }

    /**
     * Constructs a single chart displaying all of the parameters using GraphicsGenerator.
     * 
     * @param selectedParameters The parameters to display, each as a subplot.
     * @return ChartEngine displaying the chart.
     * @throws Exception If GraphicsGenerator has any problems.
     */
    @Override
    public ChartEngine constructChart(final List<ModelParameterType> selectedParameters) throws Exception
    {
        //Diagnostics are constructed by building full GraphGen plots and then turning each into a template
        //visible within a single over-arching GraphGen plot.

        final GraphicsGeneratorProductParameters productParms = new GraphicsGeneratorProductParameters();
        final GraphicsGeneratorEngine overallEngine = new GraphicsGeneratorEngine(GraphGenSettingsController.getGlobalGraphGenSettings(),
                                                                                  productParms);

        int subPlotIndex = 0;

        //Each parameter is a template.
        for(final ModelParameterType type: selectedParameters)
        {
            final ReferencedTemplateLoader loader = constructTemplateLoaderForSelectedParameter(overallEngine.getArgumentsProcessor(),
                                                                                                type,
                                                                                                subPlotIndex);
            overallEngine.getProductParameters()
                         .getTemplateParameters()
                         .getReferencedTemplateParameters()
                         .add(loader.getParameters());
            overallEngine.getReferenceHandler().setLoader(loader.getParameters(), loader);

            subPlotIndex++;
        }

        //Update the legend title.
        if(_selectedSourceParameters.getLegendTitle() != null)
        {
            overallEngine.getParameters()
                         .getChartDrawingParameters()
                         .getLegend()
                         .getLegendTitle()
                         .setText(this._selectedSourceParameters.getLegendTitle());
        }

        //Prepare the chart engine.
        overallEngine.prepareChart();

        //Get the chart engine and update the source names in the table choice box based on the selected parameter names.
        //Sources are created in order of the referenced templates, which is in order of selected parameters.
        final ChartEngine chartEngine = overallEngine.getChartEngine();
        for(int index = 0; index < selectedParameters.size(); index++)
        {
            ((DefaultXYChartDataSource)chartEngine.getDataSources().get(index)).setSourceNameInTable(selectedParameters.get(index)
                                                                                                                       .getName());
        }

        return chartEngine;
    }
}
