package ohd.hseb.hefs.pe.estimation;

import java.util.List;

import ohd.hseb.hefs.pe.core.ParameterEstimatorDiagnosticPanel;
import ohd.hseb.hefs.pe.gui.ModelParameterDiagnosticDisplay;
import ohd.hseb.hefs.pe.model.FullModelParameters;
import ohd.hseb.hefs.pe.model.ModelParameterType;
import ohd.hseb.hefs.pe.sources.SourceModelParameters;
import ohd.hseb.hefs.pe.tools.DefaultParameterDisplayChartDiagnosticPanel;
import ohd.hseb.hefs.utils.jobs.GenericJob;

/**
 * A default implementation of {@link ParameterDiagnosticPanelGenerator} that displays the parameters in a
 * {@link DefaultParameterDisplayChartDiagnosticPanel}, which is simply a scatter plot of the parameter values against time.
 * 
 * @author hankherr
 */
public class DefaultParameterDiagnosticPanelGenerator implements ParameterDiagnosticPanelGenerator
{

    /**
     * @param selectedRows The rows to check.
     * @return True if all of the parameter values for the provided selected rows are missing (NaN).
     */
    public boolean areAllSelectedParameterValuesMissing(final SourceModelParameters sourceParameters,
                                                        final List<ModelParameterType> types)
    {
        for(final ModelParameterType type: types)
        {
            if(!sourceParameters.getParameterValues(type).areAllParameterValuesMissing())
            {
                return false;
            }
        }
        return true;
    }

    /**
     * @param fullParameters A complete set of parameters.
     * @param sourceParameters The parameters for the currently chosen source.
     * @return A {@link ModelParameterDiagnosticDisplay} that can be used to construct a chart which displays
     *         parameters.
     */
    public ModelParameterDiagnosticDisplay buildModelParameterDiagnosticDisplay(final FullModelParameters fullParameters,
                                                                                final SourceModelParameters sourceParameters)
    {
        return new DefaultModelParameterDiagnosticDisplay(fullParameters, sourceParameters);
    }

    @Override
    public ParameterEstimatorDiagnosticPanel buildDiagnosticPanel(final GenericJob job,
                                                                  final FullModelParameters fullParameters,
                                                                  final SourceModelParameters sourceParameters,
                                                                  final List<ModelParameterType> displayedParameterTypes) throws Exception
    {
        job.updateNote("Initializing display...");
        if(areAllSelectedParameterValuesMissing(sourceParameters, displayedParameterTypes))
        {
            throw new Exception("All parameter values are missing for the selected parameters.");
        }

        final ModelParameterDiagnosticDisplay diag = buildModelParameterDiagnosticDisplay(fullParameters,
                                                                                          sourceParameters);

        job.updateNote("Constructing chart...");
        return new DefaultParameterDisplayChartDiagnosticPanel(diag.constructChart(displayedParameterTypes));
    }

}
