package ohd.hseb.hefs.pe.estimation;

import static com.google.common.base.Preconditions.checkState;
import static org.apache.commons.io.FileUtils.forceDelete;

import java.io.File;
import java.io.IOException;
import java.util.List;

import net.jcip.annotations.Immutable;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.estimation.options.EstimationControlOptions;
import ohd.hseb.hefs.pe.model.FullModelParameters;
import ohd.hseb.hefs.pe.sources.ForecastSource;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.xml.GenericXMLReadingHandlerException;
import ohd.hseb.hefs.utils.xml.XMLTools;

import com.google.common.collect.ImmutableList;

/**
 * General superclass identifies if parameter files exist and are up-to-date and provides methods for reading and
 * writing files. This class stores as attributes a List of source DataHander instances, since that list is independent
 * of the location and data type identifier. The methods in general require specifying an identifier.
 * 
 * @author hank.herr
 */
@Immutable
public abstract class EstimatedParametersFileHandler
{
    private final File _parametersDirectory;
    private final ImmutableList<? extends ForecastSource> _forecastSources;
    private final EstimatedParametersFileHandler _backupHandler;

    public EstimatedParametersFileHandler(final ParameterEstimatorRunInfo runInfo)
    {
        //NOTE: This class cannot store and use runInfo, because it is also used in model adapters, which will not be able 
        //to provide a runInfo.
        this(runInfo.getBaseDirectory(), ImmutableList.copyOf(runInfo.getForecastSources()));
    }

    public EstimatedParametersFileHandler(final File baseDirectory,
                                          final ImmutableList<? extends ForecastSource> forecastSources)
    {
        this(FileTools.newAbsoluteFile(baseDirectory, "parameters"), ImmutableList.copyOf(forecastSources), true);
    }

    public EstimatedParametersFileHandler(final ParameterEstimatorRunInfo runInfo,
                                          final File forcedParameterDirectory,
                                          final boolean createBackup)
    {
        //NOTE: This class cannot store and use runInfo, because it is also used in model adapters, which will not be able 
        //to provide a runInfo.
        this(forcedParameterDirectory, ImmutableList.copyOf(runInfo.getForecastSources()), createBackup);
    }

    protected EstimatedParametersFileHandler(final File forcedParameterDirectory,
                                             final ImmutableList<? extends ForecastSource> forecastSources,
                                             final boolean createBackup)
    {
        _parametersDirectory = forcedParameterDirectory;
        _forecastSources = forecastSources;
        if(createBackup)
        {
            _backupHandler = createBackupHandler();
        }
        else
        {
            _backupHandler = null;
        }
    }

    /**
     * Method can be called by subclasses to read the control parameters XML file using
     * {@link XMLTools#readXMLFromFile(File, ohd.hseb.hefs.utils.xml.XMLReadable)}.
     * 
     * @param parameters {@link FullModelParameters} with an initialized identifier and constructed
     *            {@link EstimationControlOptions}.
     * @throws GenericXMLReadingHandlerException
     */
    protected void readControlOptionsXMLFile(final FullModelParameters parameters) throws GenericXMLReadingHandlerException
    {
        XMLTools.readXMLFromFile(getControlOptionsXmlFile(parameters.getIdentifier()),
                                 parameters.getEstimationControlOptions());
    }

    /**
     * Writes the control parameters XML file
     * 
     * @param parameters The {@link FullModelParameters} containing the control parameters to write.
     * @throws Exception If a problem occurs.
     */
    public void writeControlOptionsXMLFile(final FullModelParameters parameters) throws Exception
    {
        writeControlOptionsXMLFile(parameters.getIdentifier(), parameters.getEstimationControlOptions());
    }

    /**
     * Writes the control parameters XML file to be stored with estimated parameters.
     * 
     * @param identifier
     * @param comtrolOptions
     * @throws Exception
     */
    public void writeControlOptionsXMLFile(final LocationAndDataTypeIdentifier identifier,
                                              final EstimationControlOptions comtrolOptions) throws Exception
    {
        final File xmlFile = getControlOptionsXmlFile(identifier);
        XMLTools.writeXMLFileFromXMLWriter(xmlFile, comtrolOptions, true);
    }

    /**
     * @param identifier Identifier for which to check if parameters exist.
     * @return True if the primary parameter file exists. No other tracked files are checked.
     */
    public boolean haveParametersBeenCreatedAlready(final LocationAndDataTypeIdentifier identifier)
    {
        return getPrimaryParameterFile(identifier).exists();
//        return Iterables.all(getTrackedFiles(identifier), DOES_FILE_EXIST); <-- XXX To check all files!!!
    }

    /**
     * Move parameter files into an alternate dir. The directory structure under alteranteDir must mimic that under the
     * parameter directory; i.e., it must contain precip_parms and temp_parms subdirectories.
     * 
     * @param alternateDir The alternate directory, which has the same dir structure as parameters.
     * @param identifier Identifier for which to move parameter files.
     * @param swapFiles If true, the parameter files will be swapped with any existing files by the same name. This is
     *            used for recovering backup files.
     * @throws IOException
     */
    public void moveOrSwapParameterFilesToAlternateDir(final File alternateParametersDir,
                                                       final LocationAndDataTypeIdentifier identifier,
                                                       final boolean swapFiles) throws IOException
    {
        for(final File base: getTrackedFiles(identifier))
        {
            final File target = FileTools.constructFileWithNewAncestor(getParametersDirectory(), alternateParametersDir, base);
            FileTools.moveOrSwapFiles(base, target, swapFiles);
        }
    }

    /**
     * Remove all parameter files for the identifier.
     * 
     * @param identifier
     * @throws IOException If a file cannot be removed.
     */
    public void removeParameterFiles(final LocationAndDataTypeIdentifier identifier) throws IOException
    {
        for(final File file: getTrackedFiles(identifier))
        {
            if(file.exists())
            {
                forceDelete(file);
            }
        }
    }

    /**
     * Gets the primary parameter file for {@code identifier}. Defaults to the first file returned from
     * {@link #getTrackedFiles(LocationAndDataTypeIdentifier)}, but can be overridden.
     * 
     * @param identifier the identifier to get the primary parameter file for
     * @return the primary parameter file for {@code identifier}
     */
    public File getPrimaryParameterFile(final LocationAndDataTypeIdentifier identifier)
    {
        return getTrackedFiles(identifier).get(0);
    }

    /**
     * Gets the saved parameter xml file for {@code identifier}. Defaults to the second file returned from
     * {@link #getTrackedFiles(LocationAndDataTypeIdentifier)}, but can be overridden.
     * 
     * @param identifier the identifier to get the xml file for
     * @return the parameter xml file for {@code identifier}
     */
    public File getControlOptionsXmlFile(final LocationAndDataTypeIdentifier identifier)
    {
        return getTrackedFiles(identifier).get(1);
    }

    /**
     * Returns the backup handler associated with this handler.
     * 
     * @return this handler's backup file handler
     * @throws IllegalStateException if this object is, itself, a backup handler.
     */
    public EstimatedParametersFileHandler getBackupHandler()
    {
        checkState(_backupHandler != null, "This is the backup handler.");
        return _backupHandler;
    }

    protected File getBaseDirectory()
    {
        return _parametersDirectory.getParentFile();
    }

    public File getParametersDirectory()
    {
        return _parametersDirectory;
    }

    public ImmutableList<? extends ForecastSource> getForecastSources()
    {
        return _forecastSources;
    }

    /**
     * Called in the public constructor to create the handler for backup parameters.
     * 
     * @return a backup handler
     */
    protected abstract EstimatedParametersFileHandler createBackupHandler();

    /**
     * Gets the directory in which the parameter files for {@code identifier} will be placed.
     * 
     * @param identifier the identifier to get the directory for
     * @return the directory for {@code identifier}'s parameter files
     */
    public abstract File getParametersDirectory(LocationAndDataTypeIdentifier identifier);

    /**
     * Returns a {@link List} of all {@link File}s which are tracked by this handler for {@code identifier}. The file at
     * index 0 is assumed to be the primary parameter file and index 1 is the control options file.
     * 
     * @param identifier Identifier for which to generate a list of tracked files.
     * @return A list of File instances, one per parameter file.
     */
    public abstract List<File> getTrackedFiles(LocationAndDataTypeIdentifier identifier);

    /**
     * Be sure to read in the control options file when overriding this.
     * 
     * @param identifier Identifier for which to read parameters.
     * @return FullModelParameters file read in from the default location.
     * @throws Exception
     */
    public abstract FullModelParameters readModelParameters(LocationAndDataTypeIdentifier identifier) throws Exception;

    /**
     * Writes parameter file(s) to default location given the parameters specified. Be sure to write the control options
     * file as well when overriding this.
     * 
     * @param parameters FullModelParameters instance which contains a specified identifier.
     */
    public abstract void writeParameterFile(FullModelParameters parameters) throws Exception;

    /**
     * @param identifier Identifier for which to build the FullModelParameters instance.
     * @return An instance of {@link FullModelParameters} that can be used to read the parameters for the given
     *         identifier.
     */
    public abstract FullModelParameters constructFullModelParameters(LocationAndDataTypeIdentifier identifier);
}
