package ohd.hseb.hefs.pe.estimation;

import static ohd.hseb.hefs.utils.tools.FileTools.deleteFiles;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;
import javax.swing.JToolBar;
import javax.swing.undo.UndoManager;
import javax.swing.undo.UndoableEdit;

import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepOptionsPanel;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.core.StepTableModel;
import ohd.hseb.hefs.pe.core.StepUnit;
import ohd.hseb.hefs.pe.estimation.options.ControlOptionUndoableEdit;
import ohd.hseb.hefs.pe.estimation.options.EstimationControlOptions;
import ohd.hseb.hefs.pe.model.FullModelParameters;
import ohd.hseb.hefs.pe.notice.SelectedIdentifiersChangedNotice;
import ohd.hseb.hefs.pe.notice.StepUnitsUpdatedNotice;
import ohd.hseb.hefs.pe.sources.SourceModelParameters;
import ohd.hseb.hefs.pe.tools.DefaultStepTableStatusProvider;
import ohd.hseb.hefs.pe.tools.GenericSummaryTablePanel;
import ohd.hseb.hefs.pe.tools.GenericSummaryTableStatusProvider;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.Dyad;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SelfListeningButton;
import ohd.hseb.hefs.utils.gui.tools.SingleItemButton;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.notify.ObjectModifiedNotice;
import ohd.hseb.hefs.utils.status.StatusLabel;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tools.StringTools;
import ohd.hseb.util.misc.HString;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.eventbus.Subscribe;

/**
 * Generic estimation panel displays a location summary panel with an estimation status, a ParameterSummaryPanel
 * displaying parameters loaded from the file, and a second tab which holds the control options for estimation. You may
 * need to override methods if your backup parameters include more than just what is specified by a
 * EstimatedParametersFileHandler (e.g., index files for MEFP) or if some parameters are stored outside of the standard
 * OneTypeParameterValues construct (e.g., CDFs for EnsPost).
 * 
 * @author hank.herr
 */
@SuppressWarnings("serial")
public class EstimationPEStepOptionsPanel extends ParameterEstimatorStepOptionsPanel implements
SelectedIdentifiersChangedNotice.Subscriber
{
    private JTabbedPane _tabbedPane;
    private EstimationControlOptions _workingControlOptions;
    private JScrollPane _parametersPanelScrollPane;
    private JPanel _currentlyEditedControlOptionsPanel;
    private final EstimatedParametersFileHandler _parameterHandler;
    private final EstimatedParametersFileHandler _backupParmHandler;
    private final ParameterSummaryPanel _parameterPanel;
    private final ViewLogButton _viewLogButton;

    /**
     * Holds one {@link UndoManager} per type specified in {@link #getRunInfo()}. The maps manage undo/redos for the
     * type-specific control options. The map also stores the previous options for use in adding edits to the undo
     * manager.
     */
    private final HashMap<ParameterId.Type, Dyad<EstimationOptionsUndoManager, EstimationControlOptions>> _undoMgrMap = Maps.newHashMap();

    /**
     * This button, and all other {@link SelfListeningButton}s, have their enabled stated controled via
     * {@link #reactToRowSelection(Collection)}, which receives events from {@link #_locationPanel} when selections are
     * made.
     */
    private final SelfListeningButton _restoreBackupButton = new SelfListeningButton("replaceData20x20",
                                                                                     "<html>Restore the backed up parameter and log files for the selected locations;<br>"
                                                                                         + "the current parameter and log files will become the backups</html>")
    {
        @Override
        public void actionPerformed(final ActionEvent e)
        {
            restoreBackupFiles();
        }
    };

    /**
     * This button, and all other {@link SelfListeningButton}s, have their enabled stated controled via
     * {@link #reactToRowSelection(Collection)}, which receives events from {@link #_locationPanel} when selections are
     * made.
     */
    private final SelfListeningButton _loadParametersButton = new SelfListeningButton("loadData20x20",
                                                                                      "Load the selected parameter(s)")
    {
        @Override
        public void actionPerformed(final ActionEvent e)
        {
            _parameterPanel.loadParametersForIdentifier(_locationPanel.getSelectedIdentifier());
        }
    };

    /**
     * This button, and all other {@link SelfListeningButton}s, have their enabled stated controled via
     * {@link #reactToRowSelection(Collection)}, which receives events from {@link #_locationPanel} when selections are
     * made.
     */
    private final SelfListeningButton _removeParametersButton = new SelfListeningButton("deleteData20x20",
                                                                                        "Remove the selected parameters to prevent zipping")
    {
        @Override
        public void actionPerformed(final ActionEvent e)
        {
            removeSelectedFiles();
        }
    };

    /**
     * This button, and all other {@link SelfListeningButton}s, have their enabled stated controled via
     * {@link #reactToRowSelection(Collection)}, which receives events from {@link #_locationPanel} when selections are
     * made.
     */
    private final SelfListeningButton _loadOptionsFromParmsButton = new SingleItemButton("gearLoad20x20",
                                                                                         "Set estimation options to those used for selected location's parameters.")
    {
        @Override
        public void actionPerformed(final ActionEvent e)
        {
            loadOptionsFromParameterFile();
        }
    };

    /**
     * Undoes a change.
     */
    private SelfListeningButton _undoButton = new SelfListeningButton("undo",
                                                                      "Undoes the most recent modification to the estimation options for the currently selected data type.")
    {
        @Override
        public void actionPerformed(final ActionEvent e)
        {
            //Undo the event.  Then copy the new modified options into the previous options stored in the map.
            final int scrollPosition = _undoMgrMap.get(getRunInfo().getSelectedDataType())
                                                  .getFirst()
                                                  .getEditToBeUndoneScrollPosition();
            _undoMgrMap.get(getRunInfo().getSelectedDataType()).getFirst().undo();
            final EstimationControlOptions modifiedOptions = getRunInfo().getEstimationControlOptions(getRunInfo().getSelectedDataType());
            _undoMgrMap.get(getRunInfo().getSelectedDataType()).getSecond().copyFrom(modifiedOptions);

            //Update the panel.
            updateCurrentlyEditedControlOptionsPanel(getRunInfo().getSelectedIdentifiers());
            _parametersPanelScrollPane.getVerticalScrollBar().setValue(scrollPosition);
        }
    };

    /**
     * Redoes a change.
     */
    private SelfListeningButton _redoButton = new SelfListeningButton("redo",
                                                                      "Redoes the most recent modification to the estimation options for the currently selected data type.")
    {
        @Override
        public void actionPerformed(final ActionEvent e)
        {
            //Undo the event.  Then copy the new modified options into the previous options stored in the map.
            final int scrollPosition = _undoMgrMap.get(getRunInfo().getSelectedDataType())
                                                  .getFirst()
                                                  .getEditToBeRedoneScrollPosition();
            _undoMgrMap.get(getRunInfo().getSelectedDataType()).getFirst().redo();
            final EstimationControlOptions modifiedOptions = getRunInfo().getEstimationControlOptions(getRunInfo().getSelectedDataType());
            _undoMgrMap.get(getRunInfo().getSelectedDataType()).getSecond().copyFrom(modifiedOptions);

            //Update the panel.
            updateCurrentlyEditedControlOptionsPanel(getRunInfo().getSelectedIdentifiers());
            _parametersPanelScrollPane.getVerticalScrollBar().setValue(scrollPosition);
        }
    };

    /**
     * Sets the enabled status of the undo button based on if there is anything to undo.
     */
    private void updateUndoButtonStatus()
    {
        if(_undoButton != null)
        {
            _undoButton.setEnabled(_undoMgrMap.get(getRunInfo().getSelectedDataType()).getFirst().canUndo());
            _redoButton.setEnabled(_undoMgrMap.get(getRunInfo().getSelectedDataType()).getFirst().canRedo());
        }
    }

    private GenericSummaryTablePanel _locationPanel;

    /**
     * Calls
     * {@link #EstimationPEStepOptionsPanel(ParameterEstimatorRunInfo, ParameterEstimatorStepProcessor, ParameterDiagnosticPanelGenerator)}
     * passing in an instance of {@link DefaultParameterDiagnosticPanelGenerator} as the diagnostic generator.
     */
    public EstimationPEStepOptionsPanel(final ParameterEstimatorRunInfo runInfo,
                                        final ParameterEstimatorStepProcessor stepProc,
                                        final boolean includeUndo)
    {
        this(runInfo, stepProc, new DefaultParameterDiagnosticPanelGenerator(), includeUndo);
    }

    /**
     * @param runInfo The run information for this PE.
     * @param stepProc The step processor for this options panel.
     * @param diagnosticGenerator A diagnostic panel generator to be used to create displays of the parameters.
     */
    private EstimationPEStepOptionsPanel(final ParameterEstimatorRunInfo runInfo,
                                         final ParameterEstimatorStepProcessor stepProc,
                                         final ParameterDiagnosticPanelGenerator diagnosticGenerator,
                                         final boolean includeUndo)
    {
        super(runInfo, stepProc);

        //Remove the buttons if undo is not performed.
        if(!includeUndo)
        {
            _undoButton = null;
            _redoButton = null;
        }

        _parameterHandler = runInfo.getEstimatedParametersFileHandler();
        _parameterPanel = new ParameterSummaryPanel(getRunInfo(),
                                                    _parameterHandler,
                                                    getAdditionalComponentsToDisplayInParameterSummaryPanel(),
                                                    diagnosticGenerator);

        //Setup the backup parameter file handler
        _backupParmHandler = runInfo.getEstimatedParametersBackupFileHandler();

        _viewLogButton = new ViewLogButton(runInfo.getEstimationLogFileHandler());

        //GUI stuff
        initializeDisplay();
        if(includeUndo)
        {
            initializeUndoManagers();
        }

        // Keep buttons updated.
        reactToSelectedIdentifiersChanged(new SelectedIdentifiersChangedNotice(this,
                                                                               ParameterId.Type.NONE,
                                                                               Lists.<LocationAndDataTypeIdentifier>newArrayList()));

    }

    @Override
    protected void initializeDisplay()
    {
        _locationPanel = new GenericSummaryTablePanel(getRunInfo(),
                                                      "Summary of Estimated Parameters Availability",
                                                      new StepTableModel(getRunInfo(),
                                                                         GenericEstimationPEStepProcessor.class,
                                                                         buildListOfStatusProviders()),
                                                      true,
                                                      Lists.<Component>newArrayList(_viewLogButton,
                                                                                    _loadParametersButton,
                                                                                    _restoreBackupButton,
                                                                                    _removeParametersButton,
                                                                                    _loadOptionsFromParmsButton));
        _locationPanel.getTable().getRowSelectionBus().register(_viewLogButton);
        _locationPanel.getTable().getRowSelectionBus().register(this); //See reactToRowSelection, below

        _parametersPanelScrollPane = new JScrollPane();
        _parametersPanelScrollPane.getVerticalScrollBar().setUnitIncrement(16);
        final JPanel optionsPanel = new JPanel(new BorderLayout());
        optionsPanel.add(_parametersPanelScrollPane, BorderLayout.CENTER);

        //Toolbar under the options panel.  
        if(_undoButton != null)
        {
            final JToolBar toolbar = new JToolBar();
            toolbar.setFloatable(false);
            toolbar.add(HSwingFactory.createFillerJPanel());
            toolbar.add(_undoButton);
            toolbar.add(_redoButton);
            optionsPanel.add(toolbar, BorderLayout.SOUTH);
        }

        this.setLayout(new BorderLayout());

        //Split pane
        final JSplitPane splitPane = HSwingFactory.createJSPlitPane(JSplitPane.VERTICAL_SPLIT,
                                                                    _locationPanel,
                                                                    _parameterPanel,
                                                                    true);
        splitPane.setDividerSize(5);
        splitPane.setResizeWeight(0.5);

        _tabbedPane = new JTabbedPane();
        _tabbedPane.add("Locations Summary", splitPane);
        _tabbedPane.add("Estimation Options", optionsPanel);

        add(_tabbedPane, BorderLayout.CENTER);
    }

    protected void refreshLocationSummaryTable()
    {
        _locationPanel.refreshTable();
    }

    protected List<GenericSummaryTableStatusProvider> buildListOfStatusProviders()
    {
        return Lists.newArrayList(new DefaultStepTableStatusProvider(getStepProcessor()),
                                  new LogFileStatusProvider(),
                                  new BackupFileStatusProvider());
    }

    /**
     * Sets up {@link #_undoMgrMap}.
     */
    private void initializeUndoManagers()
    {
        //For each supported type...
        for(final ParameterId.Type type: getRunInfo().getSupportedDataTypes())
        {
            //Add an entry to the map to store an undo manager and a clone of the current control options.
            _undoMgrMap.put(type,
                            new Dyad<EstimationOptionsUndoManager, EstimationControlOptions>(new EstimationOptionsUndoManager(),
                                                                                             getRunInfo().getEstimationControlOptions(type)
                                                                                                         .clone()));

            //Add a subscriber to the estimation options being tracked by the newly created undo manager.
            getRunInfo().getEstimationControlOptions(type).register(new ObjectModifiedNotice.Subscriber()
            {

                @Override
                @Subscribe
                public void reactToObjectModified(final ObjectModifiedNotice notice)
                {
                    addToUndoIfNeeded(type);
                }
            });
        }
    }

    /**
     * If a change has been made in the options for the provided type, then an edit will be added to the appropriate
     * {@link EstimationOptionsUndoManager} pulled from the {@link #_undoMgrMap}.
     */
    protected void addToUndoIfNeeded(final ParameterId.Type type)
    {
        //Whenever the estimation options change, call the addEdit method for the applicable UndoManager.
        final EstimationControlOptions modifiedOptions = getRunInfo().getEstimationControlOptions(type);
        final Dyad<EstimationOptionsUndoManager, EstimationControlOptions> dyad = _undoMgrMap.get(type);

        //Try to add the edit, but only record new previous and update button status if something was actually done!
        if(dyad.getFirst().addEdit(new ControlOptionUndoableEdit(modifiedOptions,
                                                                 dyad.getSecond(),
                                                                 modifiedOptions,
                                                                 _parametersPanelScrollPane.getVerticalScrollBar()
                                                                                           .getValue())))
        {
            //Update the previous options value.
            dyad.getSecond().copyFrom(modifiedOptions);

            //Update the undo button enabledness.
            updateUndoButtonStatus();
        }
    }

    /**
     * Call to load the estimation options from the parameter file. Override if the options cannot be loaded only from
     * the recorded estimation options file stored with each parameter file. For example, MEFP may need to load
     * canonical events. It can override this as needed.
     */
    protected void loadOptionsFromParameterFile()
    {
        final LocationAndDataTypeIdentifier identifier = _locationPanel.getSelectedIdentifier();
        if((identifier != null) && (getStepProcessor().doFilesExist(identifier)))
        {
            final EstimationControlOptions parmOptions = getStepProcessor().loadEstimationControlOptions(identifier);
            getRunInfo().getEstimationControlOptions(identifier).copyFrom(parmOptions);
            getRunInfo().copyDefaultEstimationOptionsIntoWorkingOptions();
            addToUndoIfNeeded(identifier.getParameterIdType());
            updateCurrentlyEditedControlOptionsPanel(getRunInfo().getSelectedIdentifiers());
            _locationPanel.refreshTable();
        }
    }

    protected LocationAndDataTypeIdentifier getSelectedIdentifier()
    {
        return _locationPanel.getSelectedIdentifier();
    }

    /**
     * @param identifier Identifier for which to check for backup files.
     * @return By default, true if the parameter files implied by the _backupParmHandler are all present. This may be
     *         overridden if other parameters must be checked for, such as index files for MEFP.
     */
    protected boolean areAllRequiredBackupFilesPresent(final LocationAndDataTypeIdentifier identifier)
    {
        //Parameter files.
        return _backupParmHandler.haveParametersBeenCreatedAlready(identifier);
    }

    protected EstimatedParametersFileHandler getBackupParmHandler()
    {
        return this._backupParmHandler;
    }

    protected void moveParameterFiles(final File backupDir, final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        moveParameterFiles(backupDir, identifier, true);
    }

    protected void moveParameterFiles(final File backupDir,
                                      final LocationAndDataTypeIdentifier identifier,
                                      final boolean swap) throws Exception
    {
        getRunInfo().getEstimatedParametersFileHandler().moveOrSwapParameterFilesToAlternateDir(backupDir,
                                                                                                identifier,
                                                                                                swap);
        getRunInfo().getEstimationLogFileHandler().moveOrSwapLogFileToAlternateDir(backupDir, identifier, swap);
    }

    protected void removeParameterFiles(final LocationAndDataTypeIdentifier identifier) throws IOException
    {
        deleteFiles(getRunInfo().getEstimatedParametersFileHandler().getTrackedFiles(identifier));
        getRunInfo().getEstimationLogFileHandler().determineFinalLogFile(identifier).delete();
        post(new StepUnitsUpdatedNotice(this, GenericEstimationPEStepProcessor.class, identifier));
    }

    /**
     * Override in order to, for example, add other buttons to the button panel for displaying other diagnostics.
     * 
     * @return List of Components to add to the button toolbar below the parameter summary table.
     */
    protected List<Component> getAdditionalComponentsToDisplayInParameterSummaryPanel()
    {
        return new ArrayList<Component>();
    }

    /**
     * Restores the backup files, swapping them with existing parameter file. So, to undo a restore, just click restore
     * again.
     */
    private void restoreBackupFiles()
    {
        //Verify action...
        final List<LocationAndDataTypeIdentifier> identifiers = _locationPanel.getSelectedIdentifiers();
        final String allLocationsStr = HString.buildStringFromList(LocationAndDataTypeIdentifier.createListOfIdentifierStrings(identifiers),
                                                                   ",");
        final int result = JOptionPane.showConfirmDialog(SwingTools.getGlobalDialogParent(this),
                                                         StringTools.wordWrap("Backup parameters will be restored for these locations: "
                                                                                  + allLocationsStr + ". Continue?",
                                                                              100),
                                                         "Continue?",
                                                         JOptionPane.YES_NO_OPTION);
        if(result != JOptionPane.YES_OPTION)
        {
            return;
        }

        //For all selected identifiers...
        final File backupDir = _backupParmHandler.getParametersDirectory();
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            //Check if all of the required backup files are present and, if so, swap them with existing files.
            if(areAllRequiredBackupFilesPresent(identifier))
            {
                try
                {
                    moveParameterFiles(backupDir, identifier);
                }
                catch(final Exception e)
                {
                    final int option = JOptionPane.showConfirmDialog(SwingTools.getGlobalDialogParent(this),
                                                                     "Unable to restore backup files for "
                                                                         + identifier.buildStringToDisplayInTree()
                                                                         + ":\n"
                                                                         + e.getMessage()
                                                                         + "\nContinue to next location? (Click No to cancel the restore backup.)",
                                                                     "Error Restoring Backup Files!",
                                                                     JOptionPane.YES_NO_OPTION,
                                                                     JOptionPane.ERROR_MESSAGE);
                    if(option == JOptionPane.NO_OPTION)
                    {
                        return;
                    }
                }
            }
            //If not, let the user know the backup is not complete.
            else
            {
                final int option = JOptionPane.showConfirmDialog(SwingTools.getGlobalDialogParent(this),
                                                                 "Backup files are not complete for "
                                                                     + identifier.buildStringToDisplayInTree()
                                                                     + ". Continue to next location? (Click No to cancel the restore backup.)",
                                                                 "Error Restoring Backup Files!",
                                                                 JOptionPane.YES_NO_OPTION,
                                                                 JOptionPane.ERROR_MESSAGE);
                if(option == JOptionPane.NO_OPTION)
                {
                    return;
                }
            }
            post(new StepUnitsUpdatedNotice(this, GenericEstimationPEStepProcessor.class, identifier));
        }

        _locationPanel.refreshTable();
    }

    private void removeSelectedFiles()
    {
        //Verify action.
        final List<LocationAndDataTypeIdentifier> identifiers = _locationPanel.getSelectedIdentifiers();
        final String allLocationsStr = HString.buildStringFromList(LocationAndDataTypeIdentifier.createListOfIdentifierStrings(identifiers),
                                                                   ",");
        final int result = JOptionPane.showConfirmDialog(SwingTools.getGlobalDialogParent(this),
                                                         StringTools.wordWrap("Parameters will be removed for these locations: "
                                                                                  + allLocationsStr
                                                                                  + ".\n\nDo you want to store the parameters as backups?\n"
                                                                                  + "Click No to remove them without backing up.",
                                                                              100),
                                                         "Backup Parameters?",
                                                         JOptionPane.YES_NO_CANCEL_OPTION);
        boolean backup;
        if(result == JOptionPane.YES_OPTION)
        {
            backup = true;
        }
        else if(result == JOptionPane.NO_OPTION)
        {
            backup = false;
        }
        else
        {
            return;
        }

        //Loop through the identifiers
        final File backupDir = _backupParmHandler.getParametersDirectory();
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            if(backup)
            {
                try
                {
                    moveParameterFiles(backupDir, identifier, false);
                }
                catch(final Exception e)
                {
                    final int option = JOptionPane.showConfirmDialog(SwingTools.getGlobalDialogParent(this),
                                                                     "Unable to backup parameter files for "
                                                                         + identifier.buildStringToDisplayInTree()
                                                                         + ":\n"
                                                                         + e.getMessage()
                                                                         + "\nFiles will not be removed. Continue to next location? (Click No to cancel removal)",
                                                                     "Error Backing Up Parameter Files!",
                                                                     JOptionPane.YES_NO_OPTION,
                                                                     JOptionPane.ERROR_MESSAGE);
                    if(option == JOptionPane.NO_OPTION)
                    {
                        return;
                    }
                }
            }
            else
            {
                try
                {
                    removeParameterFiles(identifier);
                }
                catch(final Exception e)
                {
                    final int option = JOptionPane.showConfirmDialog(SwingTools.getGlobalDialogParent(this),
                                                                     "Unable to remove parameter files for "
                                                                         + identifier.buildStringToDisplayInTree()
                                                                         + ":\n"
                                                                         + e.getMessage()
                                                                         + "\nContinue to next location? (Click No to cancel removal)",
                                                                     "Error Backing Up Parameter Files!",
                                                                     JOptionPane.YES_NO_OPTION,
                                                                     JOptionPane.ERROR_MESSAGE);
                    if(option == JOptionPane.NO_OPTION)
                    {
                        return;
                    }
                }
            }
            post(new StepUnitsUpdatedNotice(this, GenericEstimationPEStepProcessor.class, identifier));
        }

        _locationPanel.refreshTable();
    }

    protected void updateCurrentlyEditedControlOptionsPanel(final Collection<LocationAndDataTypeIdentifier> identifiers)
    {
        if(identifiers.isEmpty())
        {
            _currentlyEditedControlOptionsPanel = new JPanel();
            _currentlyEditedControlOptionsPanel.add(new JLabel("No locations selected for estimation."));
        }
        else
        {
            _workingControlOptions = getRunInfo().getEstimationControlOptions(ListTools.first(identifiers)
                                                                                       .getParameterIdType());
            _currentlyEditedControlOptionsPanel = _workingControlOptions.makeEditor();
        }

        _parametersPanelScrollPane.setViewportView(_currentlyEditedControlOptionsPanel);

        //Make sure the undo button is setup correctly.
        updateUndoButtonStatus();
    }

    /**
     * Listens to {@link #_locationPanel} events posted to its row selection bus. If a selection happens, this will
     * create a {@link SelectedIdentifiersChangedNotice} indicating that the selected identifiers have changed.
     */
    @Subscribe
    public void reactToRowSelection(final Collection collection)
    {
        //Get selected identifiers and check for existence of parameter files.
        final List<LocationAndDataTypeIdentifier> identifiers = _locationPanel.getSelectedIdentifiers();
        boolean doAnyBackupsExist = false;
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            if(getStepProcessor().doBackupFilesExist(identifier))
            {
                doAnyBackupsExist = true;
                break;
            }
        }
        boolean doAnyParmsExist = false;
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            if(getStepProcessor().doFilesExist(identifier))
            {
                doAnyParmsExist = true;
                break;
            }
        }

        _loadOptionsFromParmsButton.setEnabled((identifiers.size() == 1) && doAnyParmsExist);
        _loadParametersButton.setEnabled((identifiers.size() == 1) && doAnyParmsExist);
        _restoreBackupButton.setEnabled(doAnyBackupsExist);
        _loadParametersButton.setEnabled(doAnyParmsExist);
        _removeParametersButton.setEnabled(doAnyParmsExist);
    }

    public FullModelParameters getLoadedParameters()
    {
        return _parameterPanel.getLoadedParameters();
    }

    public SourceModelParameters getSelectedSourceModelParameters()
    {
        return _parameterPanel.getSelectedSourceModelParameters();
    }

    /**
     * @return The {@link JTabbedPane} that allows for adding additional components beyond just the control options
     *         panel, if necessary.
     */
    protected JTabbedPane getTabbedPane()
    {
        return _tabbedPane;
    }

    @Override
    public GenericEstimationPEStepProcessor getStepProcessor()
    {
        return (GenericEstimationPEStepProcessor)super.getStepProcessor();
    }

    @Override
    public List<? extends StepUnit> getStepUnitsToPerform()
    {
        return _locationPanel.getSelectedIdentifiers();
    }

    @Override
    @Subscribe
    public void reactToSelectedIdentifiersChanged(final SelectedIdentifiersChangedNotice evt)
    {
        updateCurrentlyEditedControlOptionsPanel(evt.getIdentifiers());

        //Update summary panel
        _locationPanel.getModel().setIdentifiers(evt.getIdentifiers());
        _locationPanel.refreshTable();
    }

    @Override
    public void gotoUnit(final Collection<StepUnit> units)
    {
        _locationPanel.selectIdentifiers(ListTools.convertCollection(units, (LocationAndDataTypeIdentifier)null));
        _tabbedPane.setSelectedIndex(0);
    }

    private class LogFileStatusProvider implements GenericSummaryTableStatusProvider
    {
        @Override
        public String getStatusColumnName()
        {
            return "Log File?";
        }

        @Override
        public String getToolTipForColumnHeader()
        {
            return "Indicates if log file was found for the estimated parameters.";
        }

        private File getLogFile(final LocationAndDataTypeIdentifier identifier)
        {
            return FileTools.newFile(getRunInfo().getBaseDirectory().getAbsolutePath(),
                                     "parameters",
                                     "logFiles",
                                     identifier.getLocationId() + "." + identifier.getParameterId() + ".txt");
        }

        @Override
        public StatusLabel getStatus(final LocationAndDataTypeIdentifier identifier)
        {
            //These status labels are not to be used to determine row readiness.
            if(getLogFile(identifier).exists())
            {
                return StatusLabel.make(true, "Log file exists for " + identifier.buildStringToDisplayInTree(), false);
            }
            else
            {
                return StatusLabel.make(false,
                                        "Log file does not exist for " + identifier.buildStringToDisplayInTree(),
                                        false);
            }
        }
    }

    private class BackupFileStatusProvider implements GenericSummaryTableStatusProvider
    {
        @Override
        public String getStatusColumnName()
        {
            return "Backup?";
        }

        @Override
        public String getToolTipForColumnHeader()
        {
            return "<html>Indicates if a backup file exists. The backup is the last set of good (estiamted<br>"
                + "error-free) parameters generated prior to the most recent attempt to estimate parameters.</html>";
        }

        @Override
        public StatusLabel getStatus(final LocationAndDataTypeIdentifier identifier)
        {
            //These status labels are not to be used to determine row readiness.
            if(areAllRequiredBackupFilesPresent(identifier))
            {
                return StatusLabel.make(true,
                                        "Backup files exist for "
                                            + identifier.buildStringToDisplayInTree()
                                            + " dated "
                                            + new Date(_backupParmHandler.getPrimaryParameterFile(identifier)
                                                                         .lastModified()),
                                        false);
            }
            else
            {
                return StatusLabel.make(false,
                                        "No backup files exist for " + identifier.buildStringToDisplayInTree(),
                                        false);
            }
        }
    }

    /**
     * Class exposes the scroll positions of edits to be undone or redone.
     * 
     * @author hankherr
     */
    private class EstimationOptionsUndoManager extends UndoManager
    {

        @Override
        public synchronized boolean addEdit(final UndoableEdit anEdit)
        {
            if(((ControlOptionUndoableEdit)anEdit).doesEditReflectAChange())
            {
                return super.addEdit(anEdit);
            }
            return false;
        }

        /**
         * @return The scroll position for the edit about to be undone.
         */
        public int getEditToBeUndoneScrollPosition()
        {
            final ControlOptionUndoableEdit edit = (ControlOptionUndoableEdit)editToBeUndone();
            if(edit == null)
            {
                return -1;
            }
            return edit.getScrollPosition();
        }

        /**
         * @return The scroll position for the edit about to be redone.
         */
        public int getEditToBeRedoneScrollPosition()
        {
            final ControlOptionUndoableEdit edit = (ControlOptionUndoableEdit)editToBeRedone();
            if(edit == null)
            {
                return -1;
            }
            return edit.getScrollPosition();
        }
    }
}
