package ohd.hseb.hefs.pe.estimation;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.ArrayList;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.ButtonGroup;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;

import ohd.hseb.hefs.pe.core.GenericParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.core.ParameterEstimatorObject;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.StepUnit;
import ohd.hseb.hefs.pe.estimation.options.EstimationControlOptions;
import ohd.hseb.hefs.pe.sources.ForecastSource;
import ohd.hseb.hefs.pe.sources.SourceDataHandler;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.tools.StringTools;
import ohd.hseb.hefs.utils.xml.GenericXMLReadingHandlerException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.util.misc.HString;

import com.google.common.base.Function;
import com.google.common.base.Objects;
import com.google.common.collect.Lists;

/**
 * Intended to be subclassed for estimation steps, it adds two methods to determine if the estimated parameter files
 * exist and if the files are current. It relies on a ParameterEstimatorRunInfo object to obtain its information.
 * 
 * @author hank.herr
 */
public abstract class GenericEstimationPEStepProcessor extends GenericParameterEstimatorStepProcessor implements
ParameterEstimatorObject
{
    private ParameterEstimatorRunInfo _runInfo;

    /**
     * Stores the user selected option for whether or not to backup parameters before estimation.
     */
    private boolean _backupParametersBeforeEstimation = true;

    /**
     * Stores the selected source for parameter estimation, or null if all sources are to be estimated.
     */
    private ForecastSource _selectedSource = null;

    protected void setRunInfo(final ParameterEstimatorRunInfo runInfo)
    {
        _runInfo = runInfo;
    }

    public void setBackupParametersBeforeEstimation(final boolean b)
    {
        _backupParametersBeforeEstimation = b;
    }

    /**
     * @return The user specified backup-parameters option derive from the confirm dialog via
     *         {@link #constructConfirmRunPanel(List)}. It will default to true.
     */
    public boolean getBackupParametersBeforeEstimation()
    {
        return _backupParametersBeforeEstimation;
    }

    public void setSelectedSource(final ForecastSource source)
    {
        _selectedSource = source;
    }

    /**
     * @return The user specified forecast source for which to estimate parameters, chosen via the confirm dialog
     *         returned by {@link #constructConfirmRunPanel(List)}. It will default to null, implying all sources.
     */
    public ForecastSource getSelectedSource()
    {
        return _selectedSource;
    }

    @Override
    public ParameterEstimatorRunInfo getRunInfo()
    {
        return _runInfo;
    }

    /**
     * @return True if all the files for the parameters exist, according to
     *         {@link EstimatedParametersFileHandler#haveParametersBeenCreatedAlready(LocationAndDataTypeIdentifier)}
     *         run for {@link #_runInfo}'s method {@link ParameterEstimatorRunInfo#getEstimatedParametersFileHandler()}.
     */
    public boolean doFilesExist(final LocationAndDataTypeIdentifier identifier)
    {
        return getRunInfo().getEstimatedParametersFileHandler().haveParametersBeenCreatedAlready(identifier);
    }

    /**
     * @return True if all the files for the backup parameters exist, according to
     *         {@link EstimatedParametersFileHandler#haveParametersBeenCreatedAlready(LocationAndDataTypeIdentifier)},
     *         run for {@link #_runInfo}'s method
     *         {@link ParameterEstimatorRunInfo#getEstimatedParametersBackupFileHandler()}.
     */
    protected boolean doBackupFilesExist(final LocationAndDataTypeIdentifier identifier)
    {
        return getRunInfo().getEstimatedParametersBackupFileHandler().haveParametersBeenCreatedAlready(identifier);
    }

    /**
     * @return Checks the parameter file last modified time against the files prepared by all source handlers. If any is
     *         newer than the parameter file, it returns false.
     */
    protected boolean areFilesCurrent(final LocationAndDataTypeIdentifier identifier)
    {
        final File parameterFile = getRunInfo().getEstimatedParametersFileHandler().getPrimaryParameterFile(identifier);
        final long parameterFileLastModifiedTime = parameterFile.lastModified();

        for(final SourceDataHandler handler: getRunInfo().getSourceDataHandlers())
        {
            for(final File fileOrDir: handler.generateListOfPreparedDataFiles(identifier))
            {
                if(fileOrDir.lastModified() > parameterFileLastModifiedTime)
                {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * @return True if a control options file exists next to the parameter file.
     */
    protected boolean doControlOptionsExist(final LocationAndDataTypeIdentifier identifier)
    {
        final File file = getRunInfo().getEstimatedParametersFileHandler().getControlOptionsXmlFile(identifier);
        return file.exists();
    }

    /**
     * @return {@link EstimationControlOptions} specifying the options used in the parameter estimation, as specified by
     *         those options stored parallel to the parameters. If users manually copy parameters into place, this can
     *         get out of sync with the true options stored in the parameter file itself, in some cases.
     */
    public EstimationControlOptions loadEstimationControlOptions(final LocationAndDataTypeIdentifier identifier)
    {
        final File file = getRunInfo().getEstimatedParametersFileHandler().getControlOptionsXmlFile(identifier);
        final EstimationControlOptions fileParms = instantiateEstimationControlOptions(identifier);

        //If the control file existence is properly checked before calling this,
        //this if should never be triggered.  If not, I've added this to avoid
        //triggering the run-time exception/e.printStackTrace() below.
        if(!file.exists())
        {
            return null;
        }
        try
        {
            XMLTools.readXMLFromFile(file, fileParms);
        }
        catch(final GenericXMLReadingHandlerException e)
        {
            //The next commented line turns it into a run-time exception, which prevents the table from painting.
            //throw Throwables.propagate(e); 
            e.printStackTrace();
            return null;
        }
        return fileParms;
    }

    /**
     * @return True if the estimation options for the provided identifier match those stored in a file parallel to the
     *         parameters file. False is returned for any other reason (such as no options file, unmatching options,
     *         etc).
     */
    protected boolean areControlOptionsCurrent(final LocationAndDataTypeIdentifier identifier)
    {
        final EstimationControlOptions fileParms = loadEstimationControlOptions(identifier);
        if(fileParms == null)
        {
            return false;
        }

        final EstimationControlOptions currentParms = getRunInfo().getEstimationControlOptions(identifier);

        return Objects.equal(fileParms, currentParms);
    }

    /**
     * Confirms if the user wants to backup the parameters prior to estimation, or just remove them. If
     * {@link #canParametersBeEstimatedForIndividualSources()} returns true, then it will ask the user to specify the
     * source for which to estimate parameters.
     */
    @Override
    public Component constructConfirmRunPanel(final List<? extends StepUnit> units)
    {
        setBackupParametersBeforeEstimation(true);
        setSelectedSource(null);

        //Super class provides list of locations.
        final Component c = super.constructConfirmRunPanel(units);

        //Forecast sources selection panel.
        JPanel sourcesPanel = null;
        if(canParametersBeEstimatedForIndividualSources())
        {
            final List<String> partialSourceIds = Lists.transform(getRunInfo().getForecastSources(),
                                                                  new Function<ForecastSource, String>()
                                                                  {
                                                                      @Override
                                                                      public String apply(final ForecastSource arg0)
                                                                      {
                                                                          return arg0.getSourceId();
                                                                      }
                                                                  });

            final List<String> sourceIds = new ArrayList<String>();
            sourceIds.add("All Sources");
            sourceIds.addAll(partialSourceIds);
            final JComboBox<String> sourceComboBox = new JComboBox(sourceIds.toArray());
            sourceComboBox.addActionListener(new ActionListener()
            {
                @Override
                public void actionPerformed(final ActionEvent e)
                {
                    if(sourceComboBox.getSelectedIndex() == 0)
                    {
                        setSelectedSource(null);
                    }
                    else
                    {
                        setSelectedSource(getRunInfo().getForecastSource((String)sourceComboBox.getSelectedItem()));
                    }
                }
            });

            final JPanel comboBoxPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
            comboBoxPanel.add(new JLabel("Select source for parameter estimation:"));
            comboBoxPanel.add(sourceComboBox);

            sourcesPanel = new JPanel(new BorderLayout());
            sourcesPanel.add(comboBoxPanel, BorderLayout.NORTH);
            final JLabel descriptionLabel = new JLabel(StringTools.htmlizeString(HString.wordWrap("If one source is selected, then parameter estimation will only be done for "
                                                                                                      + "those selected locations for which parameters already exist.  Also, general options used will be those in the "
                                                                                                      + "existing parameter file, while source-specific options will be those selected in the Estimation Options Panel.",
                                                                                                  104)));
            descriptionLabel.setForeground(Color.GRAY);
            sourcesPanel.add(descriptionLabel, BorderLayout.SOUTH);
            sourcesPanel.setBorder(HSwingFactory.createTitledBorder(BorderFactory.createBevelBorder(1),
                                                                    "Select Forecast Source",
                                                                    null));
        }

        //Backup parameters option.
        final JRadioButton backupRadioButton = new JRadioButton("Backup parameters (discard existing backup parameters)");
        backupRadioButton.setSelected(getBackupParametersBeforeEstimation());
        final JRadioButton noBackupRadioButton = new JRadioButton("Do NOT backup parameters (existing backup parameters will be unchanged)");
        final ButtonGroup group = new ButtonGroup();
        group.add(backupRadioButton);
        group.add(noBackupRadioButton);
        backupRadioButton.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                setBackupParametersBeforeEstimation(backupRadioButton.isSelected());
            }
        });
        noBackupRadioButton.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                setBackupParametersBeforeEstimation(backupRadioButton.isSelected());
            }
        });
        final JPanel radioPanel = new JPanel(new GridLayout(2, 1));
        radioPanel.add(backupRadioButton);
        radioPanel.add(noBackupRadioButton);
        radioPanel.setBorder(HSwingFactory.createTitledBorder(BorderFactory.createBevelBorder(1),
                                                              "Backup Parameters?",
                                                              null));

        //Wrapper panel for this contribution to the super class panel.
        final JPanel wrapperPanel = new JPanel(new BorderLayout());
        if(sourcesPanel != null)
        {
            wrapperPanel.add(sourcesPanel, BorderLayout.NORTH);
        }
        wrapperPanel.add(radioPanel, BorderLayout.SOUTH);

        //Put it together.
        final JPanel panel = new JPanel(new BorderLayout());
        panel.add(c, BorderLayout.CENTER);
        panel.add(wrapperPanel, BorderLayout.SOUTH);

        return panel;
    }

    /**
     * Override as needed!
     * 
     * @return True if parameters can be estimated for individual sources, false if its all-or-nothing.
     */
    protected boolean canParametersBeEstimatedForIndividualSources()
    {
        return false;
    }

    /**
     * Create an empty control parameters object which will be read in from the stored xml.
     */
    protected abstract EstimationControlOptions instantiateEstimationControlOptions(LocationAndDataTypeIdentifier identifier);
}
