package ohd.hseb.hefs.pe.estimation;

import java.io.File;

import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.estimation.options.EstimationControlOptions;
import ohd.hseb.hefs.pe.model.FullModelParameters;
import ohd.hseb.hefs.pe.model.ParameterEstimationModel;
import ohd.hseb.hefs.pe.sources.ForecastSource;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.jobs.JobMessenger;
import ohd.hseb.hefs.utils.log4j.LoggingTools;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.util.misc.HStopWatch;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

/**
 * A default class, it execute the parameter estimation process by calling the
 * {@link ParameterEstimationModel#estimateParameters(FullModelParameters, ParameterEstimatorRunInfo)} method of the
 * {@link ParameterEstimatorRunInfo#getModel(LocationAndDataTypeIdentifier)} returned model. It can be overridden as
 * needed.
 * 
 * @author hank.herr
 */
public class GenericParameterEstimationRunner
{
    private static final Logger LOG = LogManager.getLogger(GenericParameterEstimationRunner.class);

    private final ParameterEstimatorRunInfo _runInfo;

    private final LocationAndDataTypeIdentifier _runningIdentifier;

    private final int _numberOfSteps;

    /**
     * Indicates if the parameters should be backed up prior to estimation within this runner. Used in
     * {@link #prepForExecution()}.
     */
    private final boolean _backupParametersBeforeEstimation;

    /**
     * Either null to specify all forecast sources, or non-null to specify a single forecast source for which to
     * estimate parameters.
     */
    private final ForecastSource _forecastSource;

    /**
     * @param identifier The identifier for which to estimate parameters
     * @param runInfo The run information to use.
     * @param numberOfEstimationStepsToPerformForDisplayInDialog The number of estimation steps performed as part of the
     *            estimation. This is the number of times the job dialog will be instructed to increment progress. It
     *            should NOT include the step of writing the parameters to a file.
     */
    public GenericParameterEstimationRunner(final LocationAndDataTypeIdentifier identifier,
                                            final ParameterEstimatorRunInfo runInfo,
                                            final int numberOfEstimationStepsToPerformForDisplayInDialog,
                                            final boolean backupParametersBeforeEstimation,
                                            final ForecastSource forecastSource)
    {
        _runningIdentifier = identifier;
        _runInfo = runInfo;
        _numberOfSteps = numberOfEstimationStepsToPerformForDisplayInDialog;
        _backupParametersBeforeEstimation = backupParametersBeforeEstimation;
        _forecastSource = forecastSource;
    }

    public LocationAndDataTypeIdentifier getRunningIdentifier()
    {
        return _runningIdentifier;
    }

    /**
     * Creates a new subjob and sets the number of steps to _numberOfSteps + 1. Note that no log file appender is
     * initialized when this is run, so that log messages will only be visible via the CHPS logs panel or any other
     * associated Log4j appenders. There is no reason to override this, as it only deals with the {@link GenericJob}
     * interface.
     * 
     * @throws Exception
     */
    private void prepJob()
    {
        JobMessenger.newMonitorSubJob();
        JobMessenger.setMaximumNumberOfSteps(_numberOfSteps + 1);
    }

    /**
     * By default, backs up existing parameter files, control files, and log files. Note that no log file appender is
     * initialized when this is run, so that log messages will only be visible via the CHPS logs panel or any other
     * associated Log4j appenders.
     * 
     * @throws Exception
     */
    protected void prepForExecution() throws Exception
    {
        //Backup
        if(_backupParametersBeforeEstimation)
        {
            _runInfo.getEstimatedParametersFileHandler()
                    .moveOrSwapParameterFilesToAlternateDir(_runInfo.getEstimatedParametersBackupFileHandler()
                                                                    .getParametersDirectory(),
                                                            _runningIdentifier,
                                                            false);
            _runInfo.getEstimationLogFileHandler()
                    .moveOrSwapLogFileToAlternateDir(_runInfo.getEstimatedParametersBackupFileHandler()
                                                             .getParametersDirectory(),
                                                     _runningIdentifier,
                                                     false);
        }

        //Remove
        else
        {
            _runInfo.getEstimatedParametersFileHandler().removeParameterFiles(_runningIdentifier);
            _runInfo.getEstimationLogFileHandler().removeLogFile(_runningIdentifier);
        }
    }

    /**
     * Estimates the parameters given using {@link #_runInfo} for general purpose and {@link #_forecastSource} as the
     * one-forecast source to estimate parameters for (or null for all sources).
     * 
     * @param parameters The parameters object to populate during estimation. Note that the control options specified in
     *            those paraemters will be modified via the
     *            {@link ParameterEstimationModel#prepareEstimationOptions(FullModelParameters, ParameterEstimatorRunInfo, ForecastSource)}
     *            method.
     * @throws Exception
     */
    public void execute(final FullModelParameters parameters) throws Exception
    {
        String appenderName = null;

        //Prepare the control options.
        _runInfo.getModel(_runningIdentifier).prepareEstimationOptions(parameters, _runInfo, _forecastSource);
        final EstimationControlOptions controlOptions = parameters.getEstimationControlOptions();

        //Check control options for validity before anything else.
        try
        {
            controlOptions.assertValid();
        }
        catch(final Exception e)
        {
            throw new Exception("Invalid estimation control options: " + e.getMessage(), e);
        }

        //Used to time the run.
        final HStopWatch timer = new HStopWatch();

        try
        {
            prepJob(); //Creates a new child with step size initialized.
            prepForExecution();

            //Output the control parameters.  This must be after the prepForExecution, which clears out old results.
            try
            {
                final File file = _runInfo.getEstimatedParametersFileHandler()
                                          .getControlOptionsXmlFile(_runningIdentifier);
                XMLTools.writeXMLFileFromXMLWriter(file, controlOptions, true);
            }
            catch(final Exception e)
            {
                throw new Exception("Unable to record control options in XML file: " + e.getMessage());
            }

            //Initialize file logger for estimation.
            appenderName = LoggingTools.initializeLogFileLogger(_runInfo.getEstimationLogFileHandler()
                                                                        .determineFinalLogFile(parameters.getIdentifier()),
                                                                "ohd.hseb",
                                                                Level.DEBUG,
                                                                true); //Make it additive so that it does not cut off messaging unexpectedly

            //Do the estimation.
            _runInfo.getModel(_runningIdentifier).estimateParameters(parameters, _runInfo, _forecastSource);
            JobMessenger.madeProgress("Writing parameters to files...");
            LOG.info("Writing estimated parameters to parameter files...");
            LOG.info("Done estimating parameters in " + timer.getElapsedMillis() / 1000d + " seconds.");
            _runInfo.getEstimatedParametersFileHandler().writeParameterFile(parameters);
            JobMessenger.madeProgress("Done...");
        }
        catch(final Throwable e)
        {
            e.printStackTrace();
            LOG.error("Unable to complete parameter estimation: " + e.getMessage().replaceAll("\n", " "));

            //Make sure the stack trace gets into the log file, whose appender will be closed immediately upon throwing e.
            //See the finally section below.
            LoggingTools.outputStackTraceAsDebug(LOG, e);

            LOG.info("Time before stop: " + timer.getElapsedMillis() / 1000d + " seconds.");
            throw e;
        }
        finally
        {
            if(appenderName != null)
            {
                //Logger.getLogger("ohd.hseb").getAppender(appenderName).close();
                //Logger.getLogger("ohd.hseb").removeAppender(appenderName);
            	LoggingTools.removeLoggingAppender(appenderName, "ohd.hseb");
            }
            JobMessenger.clearMonitorSubJob();//The monitor subjob is created within prepJob.
        }
    }
}
