package ohd.hseb.hefs.pe.estimation;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JToolBar;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;

import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorSubPanel;
import ohd.hseb.hefs.pe.model.FullModelParameters;
import ohd.hseb.hefs.pe.model.OneTypeParameterValues;
import ohd.hseb.hefs.pe.sources.ForecastSource;
import ohd.hseb.hefs.pe.sources.SourceModelParameters;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.gui.jtable.GenericTable;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.MultiItemButton;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.jobs.HJobMonitorDialog;
import ohd.hseb.hefs.utils.jobs.JobListener;
import ohd.hseb.hefs.utils.tools.StringTools;

import org.jfree.util.Log;

import com.google.common.collect.Lists;

/**
 * Generic panel that displays loaded parameter files that are stored within the standard {@link OneTypeParameterValues}
 * by source structure. It should be directly usable without subclassing by passing an
 * {@link EstimatedParametersFileHandler} to the constructor.
 * 
 * @author hank.herr
 */
@SuppressWarnings("serial")
public class ParameterSummaryPanel extends ParameterEstimatorSubPanel implements TableModelListener
{
    private final EstimatedParametersFileHandler _handler;
    private FullModelParameters _parameters;

    private JComboBox<ForecastSource> _sourceChooser = null;
    private final GenericTable _table;
    private final ParametersFoundTableModel _tableModel;
    private final JScrollPane _centerPane;

    private final List<JobListener> _otherLoadJobListeners = new ArrayList<JobListener>();

    private LocationAndDataTypeIdentifier _identifier;

    private final ParameterDiagnosticPanelGenerator _diagnosticGenerator;

    /**
     * @param runInfo The run information for this PE.
     * @param handler The handler for loading the parameters.
     * @param additionalComponents Additional components to be displayed to the left of the view button.
     * @param diagnosticGenerator A tool that is called to generate diagnostic panels for displaying parameter values.
     */
    public ParameterSummaryPanel(final ParameterEstimatorRunInfo runInfo,
                                 final EstimatedParametersFileHandler handler,
                                 final List<Component> additionalComponents,
                                 final ParameterDiagnosticPanelGenerator diagnosticGenerator)
    {
        super(runInfo);
        _handler = handler;
        _identifier = null;
        _diagnosticGenerator = diagnosticGenerator;

        setLayout(new BorderLayout());
        updateBorder();

        updateSourceChooser();
        _table = createTable();
        _tableModel = (ParametersFoundTableModel)_table.getModel();
        _tableModel.addTableModelListener(this);
        _centerPane = createCenterPane();
        createViewButtonAndButtonPanel(additionalComponents);

        _sourceChooser.setEnabled(false);
    }

    @Override
    protected void initializeDisplay()
    {
    }

    protected FullModelParameters getLoadedParameters()
    {
        return this._parameters;
    }

    protected SourceModelParameters getSelectedSourceModelParameters()
    {
        return _tableModel.getSourceParameters();
    }

    /**
     * Pulls the sourceIDs from either {@link #_handler} or, if its loaded, {@link #getLoadedParameters()}. It updates
     * 
     * @return
     */
    private void updateSourceChooser()
    {
        List<? extends ForecastSource> sources = Lists.newArrayList(_handler.getForecastSources());
        if(getLoadedParameters() != null)
        {
            sources = getLoadedParameters().getOrderedForecastSources();

            //Remove any source id for which no source model parameters were found
            final List<ForecastSource> unfoundSources = new ArrayList<>();
            for(final ForecastSource source: sources)
            {
                if(!getLoadedParameters().getSourceModelParameters(source).wasParameterStorageInitializedForSource())
                {
                    unfoundSources.add(source);
                }
            }
            sources.removeAll(unfoundSources);
        }

        //source chooser is undefined so create it and add it to the panel.
        if(_sourceChooser == null)
        {
            //I chose not to use generics here, because if I specify JComboBox is <ForecastSource>, then I need
            //to cast the sources.toArray() results somehow to be <ForecastSource>, and I'd rather not do that.
            _sourceChooser = new JComboBox(sources.toArray());
            _sourceChooser.addActionListener(new ActionListener()
            {
                @Override
                public void actionPerformed(final ActionEvent e)
                {
                    notifySourcePicked();
                }
            });

            final JPanel panel = new JPanel(new FlowLayout(FlowLayout.CENTER));
            panel.add(new JLabel("Select Forecast Source:"));
            panel.add(_sourceChooser);

            this.add(panel, BorderLayout.NORTH);
        }
        //Update the existing source chooser and try to maintain the current selection if possible.  Otherwise, just select the first one.
        else
        {
            final ForecastSource selectedSource = (ForecastSource)_sourceChooser.getSelectedItem();
            int newSelectionIndex = sources.indexOf(selectedSource);
            if(newSelectionIndex < 0)
            {
                newSelectionIndex = 0;
            }
            _sourceChooser.setModel(new DefaultComboBoxModel<ForecastSource>(new Vector<ForecastSource>(sources)));
            _sourceChooser.setSelectedIndex(newSelectionIndex);
        }
    }

    private GenericTable createTable()
    {
        final GenericTable table = new GenericTable(new ParametersFoundTableModel(getRunInfo()));
        table.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
        return table;
    }

    private JScrollPane createCenterPane()
    {
        final JScrollPane centerPane = new JScrollPane();
        this.add(centerPane, BorderLayout.CENTER);
        return centerPane;
    }

    /**
     * Makes a button which causes this to fire a display diagnostic.
     * 
     * @param additionalComponents Additional Component to add to the left of the view button.
     */
    private void createViewButtonAndButtonPanel(final List<Component> additionalComponents)
    {
//        JButton viewButton = new SelfListeningButton("chart20x20")
        final MultiItemButton viewButton = new MultiItemButton("chart20x20")
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                if(_table.getSelectedRowCount() == 0)
                {
                    return;
                }

                //Check for all missing parameter values
                final int[] selectedRows = _table.getSelectedRows();
                final GenericJob viewParametersJob = new GenericJob()
                {
                    @Override
                    public void processJob()
                    {
                        setIndeterminate(true);
                        try
                        {
                            fireDiagnostic(_diagnosticGenerator.buildDiagnosticPanel(this,
                                                                                     _parameters,
                                                                                     _tableModel.getSourceParameters(),
                                                                                     _tableModel.getSelectedModelParameterTypes(selectedRows)));
                        }
                        catch(final Exception e)
                        {
                            e.printStackTrace();
                            fireProcessJobFailure(new Exception("Unable to display parameters:" + e.getMessage()), true);
                            return;
                        }
                        endTask();
                    }
                };

                final HJobMonitorDialog jobDialog = new HJobMonitorDialog(this,
                                                                          "Building Chart to Display Historical Data",
                                                                          viewParametersJob,
                                                                          false);
                viewParametersJob.addListener(new JobListener()
                {
                    @Override
                    public void processJobFailure(final Exception exc,
                                                  final GenericJob theJob,
                                                  final boolean displayMessage)
                    {
                        //exc.printStackTrace();
                        fireDiagnostic("Unable to build chart displaying parameters:", exc.getMessage());
                        JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(ParameterSummaryPanel.this),
                                                      StringTools.wordWrap(exc.getMessage(), 80),
                                                      "Unable to Build Chart",
                                                      JOptionPane.ERROR_MESSAGE);
                    }

                    @Override
                    public void processSuccessfulJobCompletion(final GenericJob theJob)
                    {
                    }
                });
                viewParametersJob.startJob();
                jobDialog.setMinimumSize(new Dimension(350, 10));
                jobDialog.setModal(true);
                jobDialog.setVisible(true);
            }
        };
        viewButton.setEnableOnOneSelection(true);

        _table.getRowSelectionBus().register(viewButton);

        viewButton.setEnabled(false);
        viewButton.setToolTipText("View Selected Parameters");

        final JToolBar bar = new JToolBar();
        bar.add(viewButton);
        for(final Component comp: additionalComponents)
        {
            bar.add(comp);
            if(comp instanceof JobListener)
            {
                _otherLoadJobListeners.add((JobListener)comp);
            }
        }
        bar.setFloatable(false);
        this.add(bar, BorderLayout.SOUTH);
    }

    private void updateBorder()
    {
        String title = "Parameter Summary Information for ";
        if(_identifier == null)
        {
            title += "<none loaded>";
        }
        else
        {
            title += _identifier.buildStringToDisplayInTree();
        }

        this.setBorder(HSwingFactory.createTitledBorder(BorderFactory.createEtchedBorder(), title, null));
    }

    private void notifySourcePicked()
    {
        //Just a precaution against null pointer exceptions.
        if(_parameters != null)
        {
            _tableModel.setSourceParameters(_parameters.getSourceModelParameters((ForecastSource)_sourceChooser.getSelectedItem()));
        }

        //This will force a recreation of the columns.  This must be called in order to reconstruct the column
        //headers, which may change with a source selection; setModel(...) is not enough.
        _table.createDefaultColumnsFromModel();
        if(!_tableModel.getDisplayMemberIndexColumn())
        {
            _table.getColumnModel().removeColumn(_table.getColumnModel().getColumn(2));
        }

        _centerPane.setViewportView(_table);
    }

    /**
     * Upon success or failure, all components passed into the constructor that are job listeners will be notified so
     * that they can respond accordingly.
     * 
     * @param identifier Identifier for which to load parameters.
     */
    public void loadParametersForIdentifier(final LocationAndDataTypeIdentifier identifier)
    {
        _identifier = identifier;
        if(identifier == null)
        {
            return;
        }

        //The job
        final GenericJob loadParametersJob = new GenericJob()
        {
            @Override
            public void processJob()
            {
                setIndeterminate(true);
                updateNote("Loading parameters from file...");
                try
                {
                    _parameters = _handler.readModelParameters(identifier);
                }
                catch(final Exception e)
                {
                    fireProcessJobFailure(e, true);
                    return;
                }
                endTask();
            }
        };

        final HJobMonitorDialog jobDialog = new HJobMonitorDialog(this, "Loading Parameters for "
            + _identifier.buildStringToDisplayInTree(), loadParametersJob, false);
        loadParametersJob.addListener(new JobListener()
        {
            @Override
            public void processJobFailure(final Exception exc, final GenericJob theJob, final boolean displayMessage)
            {
                //I put this in an invoke later because, on PC and Mac, I saw problems that may be due to timing
                //when rendering the parameter summary table.
                updateSourceChooser();
                updateBorder();
                _sourceChooser.setEnabled(false);

                final JPanel errorPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
                exc.printStackTrace();
                errorPanel.add(new JLabel("Error loading parameters for " + _identifier.buildStringToDisplayInTree()
                    + "!"));

                Log.debug("Error loading parameters for " + _identifier.buildStringToDisplayInTree() + ".  "
                    + exc.getClass());
                _centerPane.setViewportView(errorPanel);
            }

            @Override
            public void processSuccessfulJobCompletion(final GenericJob theJob)
            {
                //Invoke later is used because updating the source chooser could result in the 
                //table columns being reconstructed while the table is being painted, which is bad.  The invoke later
                //ensures that the source parameters and border are not updated until after any repainting is done.  It 
                //is then repainted against after the column model is updated.
                SwingUtilities.invokeLater(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        updateSourceChooser();
                        updateBorder();
                    }
                });
                _sourceChooser.setEnabled(true);
            }
        });
        for(final JobListener listener: _otherLoadJobListeners)
        {
            loadParametersJob.addListener(listener);
        }
        jobDialog.setMinimumSize(new Dimension(350, 10));
        loadParametersJob.startJob();
        jobDialog.setVisible(true);
    }

    @Override
    public void tableChanged(final TableModelEvent e)
    {
    }

}
