package ohd.hseb.hefs.pe.estimation;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.List;

import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorTableModel;
import ohd.hseb.hefs.pe.model.ModelParameterType;
import ohd.hseb.hefs.pe.model.OneTypeParameterValues;
import ohd.hseb.hefs.pe.sources.SourceModelParameters;
import ohd.hseb.hefs.utils.gui.jtable.models.MaxWidthsTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.MinWidthsTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.PreferredWidthsTableModel;

/**
 * Table model used for summarizing the parameters found in a model parameter file. The
 * {@link #setSourceParameters(SourceModelParameters)} method must be called given proper parameters before it can be
 * effectively used. The list of {@link ModelParameterType}s shown in the table is extracted from the
 * {@link SourceModelParameters} via {@link SourceModelParameters#getModelParameterTypesWithStoredValues()} when they
 * are provided.
 * 
 * @author hank.herr
 */
@SuppressWarnings("serial")
public class ParametersFoundTableModel extends ParameterEstimatorTableModel implements PreferredWidthsTableModel,
MaxWidthsTableModel, MinWidthsTableModel
{
    private static final int[] COLUMN_WIDTHS = {200, 50, 50, 75, 75};

    private SourceModelParameters _sourceParameters;
    private List<ModelParameterType> _modelParameterTypes;

    private final NumberFormat _formatter;

    private boolean _displayMemberIndexColumn = true;

    public ParametersFoundTableModel(final ParameterEstimatorRunInfo runInfo)
    {
        super(runInfo);
        _modelParameterTypes = new ArrayList<ModelParameterType>();
        _sourceParameters = null;
        _formatter = new DecimalFormat();
        _formatter.setMaximumFractionDigits(4);
    }

    /**
     * Sets the {@link #_sourceParameters} and {@link #_modelParameterTypes} from the given parametes.
     * {@link #_modelParameterTypes} are set based on
     * {@link SourceModelParameters#getModelParameterTypesWithStoredValues()}.
     * 
     * @param sourceParameters
     */
    public void setSourceParameters(final SourceModelParameters sourceParameters)
    {
        if(sourceParameters == null)
        {
            _sourceParameters = null;
            _modelParameterTypes = new ArrayList<ModelParameterType>();
        }
        else
        {
            _sourceParameters = sourceParameters;
            _modelParameterTypes = _sourceParameters.getModelParameterTypesWithStoredValues();
        }

        if(_sourceParameters.getColumnNameForEnsembleSizeIndex() == null)
        {
            _displayMemberIndexColumn = false;
        }
        
        //By using this instead of fireTableDataChanged(), I ensure the header row
        //is updated along with the data rows.
        fireTableStructureChanged();
    }

    /**
     * Calls {@link #setSourceParameters(SourceModelParameters)} passing in null.
     */
    public void clearSourceParameters()
    {
        setSourceParameters(null);
    }

    public SourceModelParameters getSourceParameters()
    {
        return this._sourceParameters;
    }

    public List<ModelParameterType> getModelParameterTypes()
    {
        return this._modelParameterTypes;
    }

    public List<ModelParameterType> getSelectedModelParameterTypes(final int[] selectedRows)
    {
        final List<ModelParameterType> types = new ArrayList<ModelParameterType>();
        for(int i = 0; i < selectedRows.length; i++)
        {
            types.add(getModelParameterTypes().get(selectedRows[i]));
        }
        return types;
    }

    public boolean getDisplayMemberIndexColumn()
    {
        return _displayMemberIndexColumn;
    }

    @Override
    public int getRowCount()
    {
        if(_modelParameterTypes == null)
        {
            return 0;
        }
        return _modelParameterTypes.size();
    }

    @Override
    public int getColumnCount()
    {
        return 5;
    }

    @Override
    public String getColumnName(final int column)
    {
        if(column == 0)
        {
            return "Parameter Type";
        }
        else if(column == 1)
        {
            if(_sourceParameters == null)
            {
                return "#";
            }
            return _sourceParameters.getColumnNameForDaysOfYearIndex();
        }
        else if(column == 2)
        {
            if(_sourceParameters == null)
            {
                return "#";
            }
            return _sourceParameters.getColumnNameForEnsembleSizeIndex();
        }
        else if(column == 3)
        {
            return "Minimum";
        }
        else if(column == 4)
        {
            return "Maximum";
        }
        return "INVALID";
    }

    @Override
    public Object getValueAt(final int row, final int column)
    {
        if(_sourceParameters == null)
        {
            return "";
        }
        if(this._modelParameterTypes.size() == 0)
        {
            return "";
        }

        final ModelParameterType type = _modelParameterTypes.get(row);
        final OneTypeParameterValues values = _sourceParameters.getParameterValues(type);
        if(column == 0)
        {
            return type.getParameterId() + " " + type.getName();
        }
        else if(column == 1)
        {
            return _sourceParameters.getNumberOfDaysOfYearWithValues();
        }
        else if(column == 2)
        {
            return _sourceParameters.getNumberOfEnsembleMembersForDiagnosticDisplay();
        }
        else if(column == 3)
        {
            if(values.areAllParameterValuesMissing())
            {
                return "No Data Found";
            }
            return _formatter.format(values.getSmallestValue());
        }
        else if(column == 4)
        {
            if(values.areAllParameterValuesMissing())
            {
                return "No Data Found";
            }
            return _formatter.format(values.getLargestValue());
        }
        return null;
    }

    @Override
    public String getCellToolTip(final int rowIndex, final int modelColIndex)
    {
        if(modelColIndex == 0)
        {
            return (String)getValueAt(rowIndex, modelColIndex);
        }
        else if(modelColIndex == 1)
        {
            return "The number of time periods for which parameters were estimated.";
        }
        else if(modelColIndex == 2)
        {
            return "The number of parameters estimated per time period.";
        }
        else if(modelColIndex == 3)
        {
            return "The smallest value of the parameters over all time periods.";
        }
        else if(modelColIndex == 4)
        {
            return "The largest value of the parameters over all time periods.";
        }
        return "";
    }

    @Override
    public String getColumnHeaderToolTip(final int modelColIndex)
    {
        return getCellToolTip(0, modelColIndex);
    }

    @Override
    public Integer getPreferredWidth(final int column)
    {
        if(0 <= column && column < COLUMN_WIDTHS.length)
        {
            return COLUMN_WIDTHS[column];
        }
        return null;
    }

    @Override
    public Integer getMinWidth(final int column)
    {
        if(column != 0)
        {
            return COLUMN_WIDTHS[column];
        }
        return null;
    }

    @Override
    public Integer getMaxWidth(final int column)
    {
        return getMinWidth(column);
    }
}
