package ohd.hseb.hefs.pe.estimation.options;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JComboBox;
import javax.swing.JComponent;

import ohd.hseb.hefs.utils.VariableSetNotice;
import ohd.hseb.hefs.utils.tools.MapTools;
import ohd.hseb.hefs.utils.tools.NumberTools;
import ohd.hseb.hefs.utils.xml.vars.XMLInteger;

import com.google.common.collect.BiMap;
import com.google.common.eventbus.Subscribe;
import com.google.common.primitives.Ints;

@SuppressWarnings("serial")
public class ChoiceEstimationOption extends EditableLeafControlOption<Integer>
{
    private String[] _choices;

    private final BiMap<Integer, Integer> _indexToValue;

    public ChoiceEstimationOption(final String xmlTag, final String label, final Integer value, final String... choices)
    {
        super(new XMLInteger(xmlTag, value), label);
        _choices = choices;
        _indexToValue = MapTools.createIndexMap(NumberTools.iota(_choices.length)); // Each index maps to itself.
    }

    /**
     * @param xmlTag the xml tag
     * @param fortranTag the fortran tag
     * @param label the description label in the gui
     * @param initialValue the initial value
     * @param choices the names of all values to choose from
     * @param values the successive values of each choice
     */
    public ChoiceEstimationOption(final String xmlTag,
                                  final String label,
                                  final Integer initialValue,
                                  final String[] choices,
                                  final int[] values)
    {
        super(new XMLInteger(xmlTag, initialValue), label);
        _choices = choices;
        _indexToValue = MapTools.createIndexMap(Ints.asList(values));
    }

    @Override
    public JComponent makeEditorEditor()
    {
        if(getLabel() == null)
        {
            return null;
        }
        return new Editor();
    }

    private class Editor extends JComboBox implements VariableSetNotice.Subscriber
    {
        private final ActionListener _comboBoxListener = new ActionListener()
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                set(_indexToValue.get(getSelectedIndex()), this);
            }
        };

        private Editor()
        {
            super(_choices);

            update();

            register(this);
        }

        private void update()
        {
            this.removeActionListener(_comboBoxListener);
            final Integer value = get();
            if(value != null)
            {
                setSelectedIndex(_indexToValue.inverse().get(value));
            }
            this.addActionListener(_comboBoxListener);
        }

        @Override
        @Subscribe
        public void reactToVariableSet(final VariableSetNotice evt)
        {
            update();
        }
    }

    @Override
    public Object clone() throws CloneNotSupportedException
    {
        //Clone must be created using the _indexToValue map in order to ensure that the correct possible values are used.
        final ChoiceEstimationOption copy = new ChoiceEstimationOption(getXMLTagName(),
                                                                       _label,
                                                                       get(),
                                                                       _choices,
                                                                       NumberTools.convertNumbersToIntsArray(_indexToValue.values()));
        copy.setDefaultValue(getDefaultValue());
        return copy;
    };

    @Override
    public void copyFrom(final ControlOption base)
    {
        final ChoiceEstimationOption toCopy = (ChoiceEstimationOption)base;
        setXMLTagName(base.getXMLTagName());
        _label = toCopy._label;
        _choices = toCopy._choices;
        _indexToValue.clear();
        _indexToValue.putAll(toCopy._indexToValue);
        set(((ChoiceEstimationOption)base).get());
        setDefaultValue(((GenericLeafControlOption<Integer>)base).getDefaultValue());
    }
}
