package ohd.hseb.hefs.pe.estimation.options;

import java.util.LinkedHashMap;

import ohd.hseb.hefs.pe.model.AlgorithmModelParameters;
import ohd.hseb.hefs.pe.model.ParameterEstimationModel;
import ohd.hseb.hefs.utils.tools.IterableTools;
import ohd.hseb.hefs.utils.tools.MapTools;
import ohd.hseb.hefs.utils.tools.ParameterId;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;

/**
 * General top-level class for storing estimation control parameters. It is designed to store a set of parameters
 * specifying options for each of the SourceDataHandlers and several instances of ParameterEstimationModels.
 * 
 * @author hank.herr
 */
public abstract class EstimationControlOptions extends GenericCompositeControlOption
{

    /**
     * Must be a {@link LinkedHashMap} in order to facilitate copyFrom, which will require remapping the suppliers.
     */
    private final LinkedHashMap<ControlOptionSupplier, ControlOption> _supplierMap;
    private ParameterEstimationModel _model;

    /**
     * Does nothing but set the XML tag.
     * 
     * @param xmlTag
     */
    protected EstimationControlOptions(final String xmlTag)
    {
        super(xmlTag);
        _supplierMap = new LinkedHashMap<ControlOptionSupplier, ControlOption>();
    }

    /**
     * @param type The parameter type (e.g., ParameterId.Type.PRECIPITATION)
     * @param xmlTag The tag associated with this XML element.
     * @param model The Parameter estimation model used.
     * @param suppliers List of control parameters.
     */
    protected EstimationControlOptions(final ParameterId.Type type,
                                       final String xmlTag,
                                       final ParameterEstimationModel model,
                                       final Iterable<? extends ControlOptionSupplier> suppliers)
    {
        super(xmlTag, Iterables.transform(IterableTools.concat(model, suppliers), getParametersByType(type)));
        _model = model;
        _supplierMap = MapTools.createLinkedHashMap(IterableTools.concat(model, suppliers), _parameters);
    }

    /**
     * Creates a function that calls
     * {@link ControlOptionSupplier#createControlOptions(ohd.hseb.hefs.utils.tools.ParameterId.Type)} with {@code type}.
     * 
     * @param type the type to pass
     */
    private static final Function<ControlOptionSupplier, ControlOption> getParametersByType(final ParameterId.Type type)
    {
        return new Function<ControlOptionSupplier, ControlOption>()
        {
            @Override
            public ControlOption apply(final ControlOptionSupplier input)
            {
                return input.createControlOptions(type);
            }
        };
    }

    public ControlOption getControlOptions(final ControlOptionSupplier supplier)
    {
        return _supplierMap.get(supplier);
    }

    public void removeControlOptions(final ControlOptionSupplier supplier)
    {
        _supplierMap.remove(supplier);
    }

    /**
     * Set any algorithm parameters that need to be held within the algorithm parameters, but are specified by
     * estimation control parameters. This method is called when finalizing reading for FullModelParameters.
     * 
     * @param algorithmParameters The algorithm parameters to populate based on estimation control parameters.
     */
    public void populateAlgorithmModelParameters(final AlgorithmModelParameters algorithmParameters)
    {
        //Does nothing by default.
    }

    public ParameterEstimationModel getModel()
    {
        return _model;
    }

    public ControlOption getModelParameters()
    {
        return getControlOptions(_model);
    }

    /**
     * Useful for cloning.
     * 
     * @param base That which is to be copied, which elements cloned.
     */
    @Override
    public void copyFrom(final ControlOption base)
    {
        super.copyFrom(base);

        final EstimationControlOptions toCopy = (EstimationControlOptions)base;

        //Copy supplier map -- The keys are not copied, but they are repointed.
        _model = toCopy._model; //NOT A COPY!
        _supplierMap.clear();
        int index = 0;
        for(final ControlOptionSupplier key: toCopy._supplierMap.keySet())
        {
            _supplierMap.put(key, _parameters.get(index)); //Using index requires a LinkedHashMap for _supplierMap!
            index++;
        }
    }

    @Override
    public EstimationControlOptions clone()
    {
        try
        {
            final EstimationControlOptions cloned = (EstimationControlOptions)super.clone();
            return cloned;
        }
        catch(final CloneNotSupportedException e)
        {
            return null;
        }
    }

}
