package ohd.hseb.hefs.pe.estimation.options;

import static com.google.common.collect.Lists.newArrayList;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JPanel;

import ohd.hseb.hefs.utils.VariableSetNotice;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SelfListeningButton;
import ohd.hseb.hefs.utils.notify.NoticeForwarder;
import ohd.hseb.hefs.utils.xml.CompositeXMLReader;
import ohd.hseb.hefs.utils.xml.CompositeXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLWriter;

import com.google.common.base.Objects;
import com.google.common.collect.Lists;
import com.google.common.eventbus.Subscribe;

/**
 * The basic class for a {@link ControlOption} composed of many sub-parameters.
 * 
 * @author alexander.garbarino
 */
public class GenericCompositeControlOption extends NoticeForwarder implements
ControlOption<List<? extends ControlOption>>
{
    private String _xmlTag;

    private String _editorBorderTitle = null;

    /**
     * The list of all sub-parameters, in the order they are processed.
     */
    protected final List<ControlOption> _parameters;

    protected GenericCompositeControlOption(final String xmlTag)
    {
        _xmlTag = xmlTag;
        _parameters = new ArrayList<ControlOption>();
    }

    protected GenericCompositeControlOption(final String xmlTag, final Iterable<? extends ControlOption> parameters)
    {
        _xmlTag = xmlTag;
        _parameters = newArrayList(parameters);
        subscribeAll(_parameters);
    }

    protected GenericCompositeControlOption(final String xmlTag, final ControlOption... parameters)
    {
        _xmlTag = xmlTag;
        _parameters = newArrayList(parameters);
        subscribeAll(_parameters);
    }

    /**
     * @return The {@link ControlOption} found with the specified XML tag name.
     */
    public ControlOption getOptionWithName(final String xmlTagName)
    {
        for(final ControlOption option: _parameters)
        {
            if(option.getXMLTagName().equals(xmlTagName))
            {
                return option;
            }
        }
        return null;
    }

    /**
     * Gets the parameter at {@code index}.
     */
    public ControlOption getParameter(final int index)
    {
        return _parameters.get(index);
    }

    /**
     * {@link #makeEditor()} returns this called with {@link #_parameters}. This is defined so that a subclass can hide
     * certain parameters from the user.
     */
    @SuppressWarnings("serial")
    protected JPanel makeEditor(final List<ControlOption> parameters)
    {
        final JPanel panel = new JPanel();
        if(_editorBorderTitle != null)
        {
            panel.setBorder(HSwingFactory.createTitledBorder(BorderFactory.createEtchedBorder(),
                                                             _editorBorderTitle,
                                                             null));
        }
        else
        {
            panel.setBorder(BorderFactory.createEtchedBorder());
        }
        panel.setLayout(new GridBagLayout());

        final GridBagConstraints cons = new GridBagConstraints();
        cons.gridy = -1;
        cons.weightx = 1;
        cons.anchor = GridBagConstraints.NORTHWEST;
        cons.fill = GridBagConstraints.HORIZONTAL;
        cons.insets = new Insets(1, 5, 1, 5);

        for(final ControlOption parameter: parameters)
        {
            if(parameter instanceof EditableLeafControlOption)
            {
                final EditableLeafControlOption leafParm = (EditableLeafControlOption)parameter;
                final JComponent editorEditor = leafParm.makeEditorEditor();
                if(editorEditor != null)
                {
                    cons.gridx = 0;
                    cons.gridy++;
                    cons.gridwidth = 1;
                    cons.weightx = 0;
                    panel.add(leafParm.makeEditorLabel(), cons);
                    cons.gridx = 1;
                    cons.weightx = 1;
                    panel.add(editorEditor, cons);

                    //Default button.
                    final JButton defaultButton = new SelfListeningButton("Default")
                    {
                        @Override
                        public void actionPerformed(final ActionEvent e)
                        {
                            leafParm.setToDefault();
                        }
                    };
                    defaultButton.setEnabled(!leafParm.isDefaultValue());
                    leafParm.register(new VariableSetNotice.Subscriber()
                    {
                        @Override
                        @Subscribe
                        public void reactToVariableSet(final VariableSetNotice notice)
                        {
                            defaultButton.setEnabled(!leafParm.isDefaultValue());
                        }
                    });

                    cons.gridx = 2;
                    cons.weightx = 0;
                    panel.add(defaultButton, cons);
                }
            }
            else
            {
                final JComponent editor = parameter.makeEditor();
                if(editor != null)
                {
                    cons.gridx = 0;
                    cons.gridy++;
                    cons.gridwidth = 3;
                    cons.weightx = 1;
                    panel.add(editor, cons);
                }
            }
        }

        cons.gridx = 0;
        cons.gridy++;
        cons.gridwidth = 2;
        cons.weighty = 1;
        cons.insets = new Insets(0, 0, 0, 0);
        cons.fill = GridBagConstraints.BOTH;
        panel.add(new JPanel(), cons);

        return panel;
    }

    @Override
    public XMLReader getReader()
    {
        return new CompositeXMLReader(getXMLTagName(), _parameters);
    }

    @Override
    public String getXMLTagName()
    {
        return _xmlTag;
    }

    protected void setXMLTagName(final String tag)
    {
        _xmlTag = tag;
    }

    public String getEditorBorderTitle()
    {
        return _editorBorderTitle;
    }

    public void setEditorBorderTitle(final String title)
    {
        _editorBorderTitle = title;
    }

    @Override
    public XMLWriter getWriter()
    {
        return new CompositeXMLWriter(getXMLTagName(), _parameters);
    }

    /**
     * Simply calls {@link ControlOption#assertValid()} on each sub-parameter.
     */
    @Override
    public void assertValid() throws IllegalStateException
    {
        for(final ControlOption parameter: _parameters)
        {
            parameter.assertValid();
        }
    }

    @Override
    public JPanel makeEditor()
    {
        return makeEditor(_parameters);
    }

    @Override
    public List<? extends ControlOption> get()
    {
        return Lists.newArrayList(_parameters);
    }

    @Override
    public void setToDefault()
    {
        //XXX This does not work when I accidentally called it with a button click.  I think it may be posting too
        //many messages to the bus without giving the bus a chance to process???  Not sure, but since this is not
        //called, I won't worry about it now.
//        for(int i = 0; i < _parameters.size(); i++)
//        {
//            _parameters.get(i).setToDefault();
//        }
        throw new IllegalStateException("Do not call this version of setToDefault(); the implementation does not work due to the software freezing.");
    }

    @Override
    public void setDefaultValue(final List<? extends ControlOption> value)
    {
        for(int i = 0; i < _parameters.size(); i++)
        {
            _parameters.get(i).setDefaultValue(value.get(i).get());
        }
    }

    @Override
    public int hashCode()
    {
        return Objects.hashCode(_xmlTag, _parameters);
    }

    @Override
    public boolean equals(final Object other)
    {
        if(this == other)
        {
            return true;
        }

        if(!(other instanceof GenericCompositeControlOption))
        {
            return false;
        }

        final GenericCompositeControlOption that = (GenericCompositeControlOption)other;
        return Objects.equal(this._xmlTag, that._xmlTag) && Objects.equal(this._parameters, that._parameters);
    }

    @Override
    public Object clone() throws CloneNotSupportedException
    {
        final List<ControlOption> copyParms = new ArrayList<ControlOption>();
        for(final ControlOption parm: this._parameters)
        {
            copyParms.add((ControlOption)parm.clone());
        }
        return new GenericCompositeControlOption(getXMLTagName(), copyParms);
    }

    @Override
    public void copyFrom(final ControlOption base)
    {
        //This creates a new list of parameters.  It does not keep existing parameters in place and just copy 
        //the values.  
        //
        //Frankly I don't know which way will be more useful in the future.  Will it need to keep the same 
        //storage variables and just change values?  
        final GenericCompositeControlOption toCopy = (GenericCompositeControlOption)base;
        _xmlTag = base.getXMLTagName();
        _parameters.clear();
        for(final ControlOption parm: toCopy._parameters)
        {
            try
            {
                _parameters.add((ControlOption)parm.clone());
            }
            catch(final CloneNotSupportedException e)
            {
                //Should not happen
                e.printStackTrace();
            }
        }
        subscribeAll(_parameters);
    }

}
