package ohd.hseb.hefs.pe.estimation.options;

import static com.google.common.collect.Lists.newArrayList;
import static ohd.hseb.hefs.utils.tools.ListTools.concat;

import java.awt.BorderLayout;
import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JCheckBox;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;

import ohd.hseb.hefs.utils.VariableSetNotice;

import com.google.common.collect.Lists;
import com.google.common.eventbus.Subscribe;

/**
 * A number of sub-parameters whose visibility are all controlled by a single boolean parameter.
 * 
 * @author alexander.garbarino
 */
public class OptionalCompositeControlOption extends GenericCompositeControlOption
{
    protected BooleanEstimationOption _useParameter;
    private final List<ControlOption> _additionalParameters;

    /**
     * This should only be called by subclasses that are about to fill in the {@link #_useParameter} and
     * {@link #_additionalParameters} fields on their own, possibly by calling {@link #copyFrom(ControlOption)}.
     * 
     * @param xmlTag
     */
    protected OptionalCompositeControlOption(final String xmlTag)
    {
        super(xmlTag);
        _additionalParameters = Lists.newArrayList();
        _useParameter = null;
    }

    protected OptionalCompositeControlOption(final String xmlTag,
                                                final BooleanEstimationOption useParameter,
                                                final ControlOption... additionalParameters)
    {
        this(xmlTag, useParameter, Lists.newArrayList(additionalParameters));
    }

    protected OptionalCompositeControlOption(final String xmlTag,
                                                final BooleanEstimationOption useParameter,
                                                final List<ControlOption> additionalParameters)
    {
        super(xmlTag, concat(newArrayList(useParameter), additionalParameters));
        _useParameter = useParameter;
        _additionalParameters = additionalParameters;
    }

    public boolean isEnabled()
    {
        return _useParameter.get();
    }

    public void setEnabled(final boolean enabled)
    {
        _useParameter.set(enabled, this);
    }

    @Override
    public JPanel makeEditor()
    {
        final JCheckBox check = _useParameter.makeEditor();
        final JPanel innerPanel = super.makeEditor(_additionalParameters);
        innerPanel.setBorder(BorderFactory.createEtchedBorder());
        innerPanel.setVisible(_useParameter.get());

        final JPanel outerPanel = new JPanel(new BorderLayout());
        outerPanel.add(check, BorderLayout.NORTH);
        outerPanel.add(innerPanel, BorderLayout.CENTER);

        _useParameter.register(new VariableSetNotice.Subscriber()
        {
            @Override
            @Subscribe
            public void reactToVariableSet(final VariableSetNotice evt)
            {
                innerPanel.setVisible(_useParameter.get());

                // Scroll to panel.
                if(innerPanel.isVisible())
                {
                    SwingUtilities.invokeLater(new Runnable()
                    {
                        @Override
                        public void run()
                        {
                            // The min/max seems to confuse it enough so that it doesn't scroll horizontally.
                            // The 5 height includes all off the etched border.
                            outerPanel.scrollRectToVisible(new Rectangle(Integer.MIN_VALUE,
                                                                         outerPanel.getHeight(),
                                                                         Integer.MAX_VALUE,
                                                                         5));
                        }
                    });
                }
            }
        });

        return outerPanel;
    }

    @Override
    public Object clone() throws CloneNotSupportedException
    {
        //Creates a list of copies of the additional parameters and passes those into a constructor along
        //with a copy of the 
        final List<ControlOption> copyAddParms = new ArrayList<ControlOption>();
        for(int i = 0; i < _additionalParameters.size(); i++)
        {
            final ControlOption parm = _parameters.get(i);
            copyAddParms.add((ControlOption)parm.clone());
        }
        return new OptionalCompositeControlOption(getXMLTagName(),
                                                     (BooleanEstimationOption)_useParameter.clone(),
                                                     copyAddParms);
    }

    @Override
    public void copyFrom(final ControlOption base)
    {
        super.copyFrom(base);

        //The underlying variables are changed via the copyFrom, so I need to recollect them.
        _useParameter = (BooleanEstimationOption)_parameters.get(0);
        _additionalParameters.clear();
        for(final ControlOption oneParm: _parameters.subList(1, _parameters.size()))
        {
            _additionalParameters.add(oneParm);
        }
    }
}
