package ohd.hseb.hefs.pe.gui;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.util.Arrays;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JRootPane;
import javax.swing.JScrollPane;
import javax.swing.KeyStroke;

import ohd.hseb.hefs.pe.core.ParameterEstimatorDiagnosticPanel;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.notice.ClearDiagnosticNotice;
import ohd.hseb.hefs.pe.notice.DiagnosticSubscriber;
import ohd.hseb.hefs.pe.notice.DisplayDiagnosticMessageNotice;
import ohd.hseb.hefs.pe.notice.DisplayDiagnosticPanelNotice;
import ohd.hseb.hefs.utils.gui.components.ClickableJLabel;
import ohd.hseb.hefs.utils.gui.components.ComponentPanelGlassPane;
import ohd.hseb.hefs.utils.gui.help.HelpFile;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.tools.IconTools;

import com.google.common.collect.Lists;
import com.google.common.eventbus.Subscribe;

/**
 * Displays a {@link ParameterEstimatorDiagnosticPanel} inside of a {@link JRootPane} with a
 * {@link ComponentPanelGlassPane} displaying two buttons: a clear button and a detach button.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
//@HelpFile("helpManual.htm#DiagnosticDisplayPanelRef")
@HelpFile("PDFHELP:helpManual.pdf#DiagnosticDisplayPanelRef")
public class DiagnosticsDisplayPanel extends JRootPane implements DiagnosticSubscriber
{

    private JPanel _currentlyDisplayedPanel = null;
    private ParameterEstimatorRunInfo _runInfo = null;

    /**
     * Records the components passed in to {@link #addGlassPane(Component[])}. These components are carried with the
     * diagnostic if it is ever spun off.
     */
    private Component[] _componentsAddedToGlassPane = null;

    public DiagnosticsDisplayPanel()
    {
        getContentPane().setLayout(new BorderLayout());
        addGlassPane(null);
    }

    /**
     * Adds the tool bar glass pane.
     */
    private void addGlassPane(final Component[] components)
    {
        //Record the additional components for the glass pane
        if(components == null)
        {
            _componentsAddedToGlassPane = null;
        }
        else
        {
            _componentsAddedToGlassPane = Arrays.copyOf(components, components.length);
        }

        final ClearAction clearAction = new ClearAction();
        final DetachDiagnosticPanelAction duplicateAction = new DetachDiagnosticPanelAction();

        //Clear diagnostic by clicking on a label
        final ClickableJLabel clearLabel = new ClickableJLabel(IconTools.getHSEBIcon("delete14x14"), clearAction);
        clearLabel.setIconToUseWhenCursorIsNotOverLabel(IconTools.getHSEBIcon("translucentDelete14x14"));
        clearLabel.setToolTipText("Clear diagnostic display");
        clearLabel.createTransparentMargin(2, 2, 2, 2);

        //Duplicate is a bit of a misnomer.  It takes the panel and puts it in a frame for display.
        final ClickableJLabel detachWindowLabel = new ClickableJLabel(IconTools.getHSEBIcon("detachWindow14x14"),
                                                                      duplicateAction);
        detachWindowLabel.setIconToUseWhenCursorIsNotOverLabel(IconTools.getHSEBIcon("translucentDetachWindow14x14"));
        detachWindowLabel.createTransparentMargin(2, 2, 2, 2);
        detachWindowLabel.setToolTipText("Move diagnostic display to a separate window");

        // Clear Diagnostic with F5
        final String actionName = GenericParameterEstimatorExplorerPlugIn.class.getCanonicalName() + "#clearDiagnostic";
        getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(KeyStroke.getKeyStroke("F5"),
                                                                       clearAction.getName());
        getActionMap().put(actionName, clearAction);

        //Construct the list of components to include on the glass pane in the upper left corner.
        final List<Component> addedComps = Lists.newArrayList((Component)clearLabel, detachWindowLabel);
        if(_componentsAddedToGlassPane != null)
        {
            addedComps.addAll(Arrays.asList(_componentsAddedToGlassPane));
        }

        //Create the glass panel and add it to the glass pane for this.
        final ComponentPanelGlassPane glassPanel = new ComponentPanelGlassPane(addedComps, FlowLayout.LEADING);
        glassPanel.addToPane(this);
        glassPanel.setVisible(true);
        getGlassPane().setVisible(false);
    }

    public ParameterEstimatorRunInfo getRunInfo()
    {
        return _runInfo;
    }

    public void setRunInfo(final ParameterEstimatorRunInfo runInfo)
    {
        _runInfo = runInfo;
        _runInfo.register(this);
    }

    @Override
    @Subscribe
    public void reactToClearDiagnostic(final ClearDiagnosticNotice evt)
    {
        getContentPane().removeAll();
        getGlassPane().setVisible(false);
        repaint();
    }

    @Override
    @Subscribe
    public void reactToDisplayDiagnosticPanel(final DisplayDiagnosticPanelNotice evt)
    {
        _currentlyDisplayedPanel = evt.getDiagnosticPanel();

        getContentPane().removeAll();
        getGlassPane().setVisible(false);
        if(_currentlyDisplayedPanel == null)
        {
            return;
        }

        try
        {
            //Put the panel in place.
            getContentPane().add(_currentlyDisplayedPanel, BorderLayout.CENTER);

            //Build the glass pane and add it to this.
            if(_currentlyDisplayedPanel instanceof ParameterEstimatorDiagnosticPanel)
            {
                final Component[] comps = ((ParameterEstimatorDiagnosticPanel)_currentlyDisplayedPanel).getComponentsForDiagnosticsToolBar();
                addGlassPane(comps);
            }

            //Make it visible.
            getGlassPane().setVisible(true);
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            final JScrollPane badResultsScrollArea = new JScrollPane();
            final String headerMessage = "Failed to Display Diagnostics:";
            final String errMessage = e.getMessage().replaceAll("\n", "<br>");
            badResultsScrollArea.setViewportView(HSwingFactory.createErrorMessagePane(headerMessage, errMessage));
            getContentPane().add(badResultsScrollArea, BorderLayout.CENTER);
            _currentlyDisplayedPanel = null;
        }

        SwingTools.forceComponentRedraw(getContentPane());
    }

    @Override
    @Subscribe
    public void reactToDisplayDiagnosticMessage(final DisplayDiagnosticMessageNotice evt)
    {
        getContentPane().removeAll();
        final JScrollPane badResultsScrollArea = new JScrollPane();
        final String headerMessage = evt.getHeader();
        badResultsScrollArea.setViewportView(HSwingFactory.createErrorMessagePane(headerMessage, evt.getBody()));
        getContentPane().add(badResultsScrollArea, BorderLayout.CENTER);
        getGlassPane().setVisible(false);
        setVisible(false);
        setVisible(true);
    }

    /**
     * Action clears the diagnostic by firing a ClearDiagnosticEvent which is listened to above.
     * 
     * @author hank.herr
     */
    private class ClearAction extends AbstractAction
    {
        public ClearAction()
        {
            super(GenericParameterEstimatorExplorerPlugIn.class.getCanonicalName() + "#clearDiagnostic",
                  IconTools.getHSEBIcon("translucentDelete14x14"));
        }

        public String getName()
        {
            return (String)getValue(Action.NAME);
        }

        public void clearDiagnostic()
        {
            getRunInfo().post(new ClearDiagnosticNotice(DiagnosticsDisplayPanel.this));
        }

        @Override
        public void actionPerformed(final ActionEvent e)
        {
            clearDiagnostic();
        }
    }

    /**
     * Action to spinoff the chart into its own frame. The chart will take _componentsAddedToGlassPane with it.
     * 
     * @author hankherr
     */
    private class DetachDiagnosticPanelAction extends AbstractAction
    {
        public void actionPerformed(final ActionEvent e)
        {
            final JPanel displayPanel = new JPanel(new BorderLayout());
            displayPanel.add(_currentlyDisplayedPanel, BorderLayout.CENTER);

            final JFrame panelFrame = new JFrame("Diagnostic Display");
            panelFrame.setContentPane(displayPanel);
            panelFrame.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
            panelFrame.setSize(DiagnosticsDisplayPanel.this.getParent().getSize());
            panelFrame.setLocationRelativeTo(DiagnosticsDisplayPanel.this.getParent());

            //Construct the list of components to include on the glass pane in the upper left corner.
            final List<Component> addedComps = Lists.newArrayList();
            if(_componentsAddedToGlassPane != null)
            {
                addedComps.addAll(Arrays.asList(_componentsAddedToGlassPane));
            }

            //Create the glass panel and add it to the glass pane for this.
            final ComponentPanelGlassPane glassPanel = new ComponentPanelGlassPane(addedComps, FlowLayout.LEADING);
            glassPanel.addToPane(panelFrame.getRootPane());
            glassPanel.setVisible(true);
            getGlassPane().setVisible(true);

            _currentlyDisplayedPanel = null;
            reactToClearDiagnostic(null);

            panelFrame.setVisible(true);
        }
    }
}
