package ohd.hseb.hefs.pe.gui;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Dimension;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JRootPane;
import javax.swing.JScrollPane;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.noos.xing.mydoggy.AggregationPosition;
import org.noos.xing.mydoggy.ContentManager;
import org.noos.xing.mydoggy.MultiSplitConstraint;
import org.noos.xing.mydoggy.MultiSplitContentManagerUI;
import org.noos.xing.mydoggy.TabbedContentManagerUI;
import org.noos.xing.mydoggy.TabbedContentUI;
import org.noos.xing.mydoggy.ToolWindow;
import org.noos.xing.mydoggy.plaf.MyDoggyToolWindowManager;

import com.google.common.eventbus.Subscribe;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;

import nl.wldelft.fews.common.config.GlobalProperties;
import nl.wldelft.fews.gui.explorer.FewsEnvironment;
import nl.wldelft.fews.gui.explorer.FewsExplorerPlugin;
import nl.wldelft.fews.gui.plugin.consumers.EmbeddedPiServerPortConsumer;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.notice.RefreshFullDisplayNotice;
import ohd.hseb.hefs.pe.sources.ForecastSource;
import ohd.hseb.hefs.utils.gui.help.CurrentFocusHelpManager;
import ohd.hseb.hefs.utils.gui.mydoggy.OHDMyDoggyMultiSplitContentManagerUI;
import ohd.hseb.hefs.utils.gui.mydoggy.OHDMyDoggyToolWindowManager;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.jobs.HJobMonitorDialog;
import ohd.hseb.hefs.utils.jobs.JobListener;
import ohd.hseb.hefs.utils.piservice.FewsPiServiceProvider;
import ohd.hseb.hefs.utils.tools.ThreadTools;

/**
 * This must be subclassed for each instance of a Parameter Estimator. The methods to override involve initializing
 * 
 * @author hank.herr
 */
@SuppressWarnings("serial")
public abstract class GenericParameterEstimatorExplorerPlugIn extends JPanel
implements FewsExplorerPlugin, RefreshFullDisplayNotice.Subscriber, EmbeddedPiServerPortConsumer
{
    private static final Logger LOG = LogManager.getLogger(GenericParameterEstimatorExplorerPlugIn.class);

    public static final String BASE_DIR_PROPERTY = "HEFSMODELSDIR";
    public static final String REGION_HOME_PROPERTY = "REGION_HOME";

    private final ListenableFuture<ParameterEstimatorRunInfo> _runInfo;

    private ParameterEstimatorStepsPanel _stepsPanel;
    private LocationSummaryTablePanel _locationPanel;
    protected final DiagnosticsDisplayPanel _diagnosticsPanel = new DiagnosticsDisplayPanel();
    private List<ParameterEstimatorStepProcessor> _steps;

    private boolean _alive;

    /**
     * Constructor called when run in CHPS.
     */
    public GenericParameterEstimatorExplorerPlugIn()
    {
        this(getDefaultBaseDirectory(), getDefaultConfigDirectory());
    }

    @Override
    public void setPortNumber(final int arg0)
    {
        FewsPiServiceProvider.initializeDefaultService(true, Integer.toString(arg0));
    }

    /**
     * Constructor called when run for testing (outside CHPS).
     * 
     * @param baseDirectory
     * @param configDirectory
     */
    public GenericParameterEstimatorExplorerPlugIn(final File baseDirectory, final File configDirectory)
    {
        LOG.info(this.getClass().getSimpleName() + " run area directory is " + baseDirectory.getAbsolutePath());

        final ListeningExecutorService executor = MoreExecutors.listeningDecorator(Executors.newCachedThreadPool());

        // Start creating the run info right away.
        // Later need to integrate with jobs and such - but for now the job calls it anyway, so no issue.
        final Callable<ParameterEstimatorRunInfo> runInfoCall = new Callable<ParameterEstimatorRunInfo>()
        {
            @Override
            public ParameterEstimatorRunInfo call() throws Exception
            {
                final ParameterEstimatorRunInfo runInfo =
                                                        createDefaultRunInfo(executor, baseDirectory, configDirectory);
                runInfo.register(GenericParameterEstimatorExplorerPlugIn.this);
                _diagnosticsPanel.setRunInfo(runInfo);
//debugging               runInfo.register(new EventBusObserver("Main"));
                return runInfo;
            }
        };
        _runInfo = executor.submit(runInfoCall);

        LOG.info(this.getClass().getSimpleName() + " run area directory is " + baseDirectory.getAbsolutePath());

        if(!baseDirectory.exists())
        {
            displayOverallMessage("Unable to initialize " + this.getClass().getSimpleName() + " Explorer Plug-in:",
                                  "Run area directory should have been created during "
                                      + "installation but does not exist. Please perform "
                                      + this.getClass().getSimpleName()
                                      + " installation steps and make sure the global property " + BASE_DIR_PROPERTY
                                      + " has been set to point to Models/hefs.");
        }
        else
        {
            initialize(baseDirectory, configDirectory);
            setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
            displayOverallMessage("Initializing " + this.getClass().getSimpleName() + "...", "");
        }

        //DEBUG ONLY -- tracks focus ownership
//        FocusOwnerTracker tracker = new FocusOwnerTracker(this)
//        {
//            @Override
//            public void focusLost()
//            {
//            }
//
//            @Override
//            public void focusGained()
//            {
//            }
//        };
//        tracker.start();
    }

    /**
     * @return The value of {@link #BASE_DIR_PROPERTY} within the {@link GlobalProperties} of FEWS.
     */
    protected static File getDefaultBaseDirectory()
    {
        if(GlobalProperties.get(BASE_DIR_PROPERTY) == null)
        {
            return null;
        }
        return new File(GlobalProperties.get(BASE_DIR_PROPERTY));
    }

    /**
     * @return The value of {@value #REGION_HOME_PROPERTY} within the {@link GlobalProperties} of FEWS.
     */
    protected static File getDefaultConfigDirectory()
    {
        final String regionDir = GlobalProperties.get(REGION_HOME_PROPERTY);
        return new File(regionDir, "Config");
    }

    /**
     * @param executor the executor service which the run info will use
     * @param baseDirectory The base directory (run area) for the estimator
     * @param configDirectory The configuration directory for the associated CHPS configuration
     * @return A ParameterEstimatorRunInfo instance to be used for the parameter estimator
     * @throws Exception
     */
    protected abstract ParameterEstimatorRunInfo createDefaultRunInfo(ExecutorService executor,
                                                                      File baseDirectory,
                                                                      File configDirectory) throws Exception;

    /**
     * @return A list of ParameterEstimatorStepProcessor instances specifying the steps to be performed. By default, the
     *         list will include the {@link ParameterEstimatorStepProcessor} returned by
     *         {@link ForecastSource#getSourceStepProcessor(ParameterEstimatorRunInfo)} for each source provided in
     *         {@link #_runInfo}.
     */
    protected List<ParameterEstimatorStepProcessor> initializeStepProcessors()
    {
        final List<ParameterEstimatorStepProcessor> steps = new ArrayList<ParameterEstimatorStepProcessor>();
        for(final ForecastSource source: getRunInfo().getForecastSources())
        {
            steps.add(source.getSourceStepProcessor(getRunInfo()));
        }
        return steps;
    }

    /**
     * @return A PESetupPanel which allows the users to setup for parameter estimation.
     */
    protected abstract PESetupPanel initializeParameterEstimatorSetupPanel();

    /**
     * Called when initialization is complete and after the display is created, allowing a subclass to perform any
     * actions necessary after initialization.
     */
    protected abstract void reactToSuccessfulInitialization();

    /**
     * Called to initialize the display. It does its job in a thread so that it does not freeze the CHPS GUI. However,
     * the progress dialog it creates is modal.
     * 
     * @param baseDirectory
     * @param configDirectory
     */
    public void initialize(final File baseDirectory, final File configDirectory)
    {
        final GenericJob initJob = new GenericJob()
        {
            @Override
            public void processJob()
            {
                LOG.info("Initializing " + this.getClass().getSimpleName() + "...");
                setIndeterminate(true);
                updateNote("Initializing " + this.getClass().getSimpleName()
                    + ".\nCheck the CHPS Logs Panel for status.");

                //If I want to capture messages from within the calls below, I may want to create my
                //own AppenderSkeleton to capture the Log4j message and send them to the job dialog.
                //See GraphGen Conductor for examples.

                //Initialize the run info
                try
                {
                    // Help
                    final File helpFile = getRunInfo().getHelpDirectory();
                    new CurrentFocusHelpManager(helpFile, GenericParameterEstimatorExplorerPlugIn.this);
                }
                catch(final Exception e)
                {
                    fireProcessJobFailure(e, true);
                    return;
                }

                _steps = initializeStepProcessors();
                LOG.info("Done initializing " + this.getClass().getSimpleName() + ".");

                endTask();
            }
        };

        final HJobMonitorDialog jobDialog = new HJobMonitorDialog(this,
                                                                  "Initializing " + this.getClass().getSimpleName(),
                                                                  initJob,
                                                                  false);
        initJob.addListener(new JobListener()
        {
            @Override
            public void processJobFailure(final Exception exc, final GenericJob theJob, final boolean displayMessage)
            {
                exc.printStackTrace();
                displayOverallMessage("Unable to initialize " + this.getClass().getSimpleName() + ":",
                                      exc.getMessage());
            }

            @Override
            public void processSuccessfulJobCompletion(final GenericJob theJob)
            {
                ThreadTools.sleep(500); //Force a half-second sleep to give it a chance to displays the init panel.
                removeAll();
                initializeDisplay();
                reactToSuccessfulInitialization();
                setCursor(Cursor.getDefaultCursor());
                setVisible(false);
                setVisible(true);
            }
        });
        initJob.startJob();
        jobDialog.setMinimumSize(new Dimension(300, 10));
        jobDialog.setModal(false); //Don't block
        jobDialog.setVisible(true);
    }

    /**
     * Initializes the display.
     */
    protected void initializeDisplay()
    {
        final JPanel myDoggyWrapperPanel = new JPanel(new BorderLayout());
        _stepsPanel = new ParameterEstimatorStepsPanel(getRunInfo(),
                                                       _steps,
                                                       initializeParameterEstimatorSetupPanel(),
                                                       myDoggyWrapperPanel);
        _locationPanel = new LocationSummaryTablePanel(getRunInfo(), _steps, getRunInfo().getSupportedDataTypes());
        myDoggyWrapperPanel.add(createMyDoggyDisplayPanel(), BorderLayout.CENTER);

        //For some reason, I cannot make this panel a JRootPane.  If I do and set its content pane to myDoggyWrapper
        //it shows a clear screen.  Therefore, I'm leaving this as a JPanel and adding to it a JRootPane that contains
        //the myDoggyWrapperPanel as its content.  Therefore, when the help manager looks for a parent JRootPane,
        //it will find innerRootPane.
        final JRootPane innerRootPane = new JRootPane();
        innerRootPane.setContentPane(myDoggyWrapperPanel);
        this.setLayout(new BorderLayout());
        this.add(innerRootPane, BorderLayout.CENTER);

        LOG.info("Initiating thread to save the run-time information once per minute.");
        getRunInfo().startSaveThread();
    }

    /**
     * @return A MyDoggy panel to serve as the main display for this plugin.
     */
    private Component createMyDoggyDisplayPanel()
    {
        // Create a new instance of MyDoggyToolWindowManager passing the frame.
        final MyDoggyToolWindowManager myDoggyToolWindowManager = new OHDMyDoggyToolWindowManager();
        final MyDoggyToolWindowManager _toolWindowManager = myDoggyToolWindowManager;
        final ContentManager contentManager = _toolWindowManager.getContentManager();
        final MultiSplitContentManagerUI contentManagerUI = new OHDMyDoggyMultiSplitContentManagerUI();
        contentManager.setContentManagerUI(contentManagerUI);

        //Add the three contents.
        contentManager.addContent("Steps",
                                  "Estimation Steps Panel",
                                  null, // An icon
                                  _stepsPanel,
                                  "Perform steps in MEFP parameter estimation process.",
                                  new MultiSplitConstraint(AggregationPosition.LEFT));
        contentManager.addContent("Locations",
                                  "Estimation Location Summary Panel",
                                  null, // An icon
                                  _locationPanel,
                                  "Displays a summary of the step performed for the available locations, based on historical data.",
                                  new MultiSplitConstraint(contentManager.getContent(0), AggregationPosition.RIGHT)); //Make this RIGHT if template parameters included
        contentManager.addContent("Diagnostics",
                                  "Diagnostics Panel",
                                  null, // An icon
                                  _diagnosticsPanel,
                                  "Allows for display of diagnostics pertaining to active step.",
                                  new MultiSplitConstraint(contentManager.getContent(1), //(1),
                                                           AggregationPosition.BOTTOM));

        //Setup the content manager.
        contentManagerUI.setTabPlacement(TabbedContentManagerUI.TabPlacement.TOP);
        contentManagerUI.setShowAlwaysTab(false);

        //Setup the three content ui's.
        TabbedContentUI contentUI;
        contentUI = (TabbedContentUI)_toolWindowManager.getContentManager().getContent(0).getContentUI();
        contentUI.setDetachable(false);
        contentUI.setCloseable(false);
        contentUI.setMinimizable(false);
        contentUI.getContent().setPopupMenu(new JPopupMenu()); //Shuts off the right-click tab popup menu

        contentUI = (TabbedContentUI)_toolWindowManager.getContentManager().getContent(1).getContentUI();
        contentUI.setDetachable(true);
        contentUI.setCloseable(false);
        contentUI.setMinimizable(false);
        contentUI.getContent().setPopupMenu(new JPopupMenu()); //Shuts off the right-click tab popup menu

        contentUI = (TabbedContentUI)_toolWindowManager.getContentManager().getContent(2).getContentUI();
        contentUI.setDetachable(true);
        contentUI.setCloseable(false);
        contentUI.setMinimizable(false);
        contentUI.getContent().setPopupMenu(new JPopupMenu()); //Shuts off the right-click tab popup menu

        // Make all tools available
        for(final ToolWindow window: _toolWindowManager.getToolWindows())
        {
            window.setAvailable(true);
        }

        return myDoggyToolWindowManager;
    }

    /**
     * Called to display a message in the main panel instead of the GUI. This is called if an error prevents the
     * parameter estimator from being used.
     * 
     * @param headerLine Print in bold at the top.
     * @param messageBody Regular font just below.
     */
    protected void displayOverallMessage(final String headerLine, final String messageBody)
    {
        removeAll();
        setLayout(new BorderLayout());
        final JScrollPane failedInitScrollPane = new JScrollPane();
        failedInitScrollPane.setViewportView(HSwingFactory.createErrorMessagePane(headerLine, messageBody));
        failedInitScrollPane.getViewport().getView().setBackground(failedInitScrollPane.getBackground());
        add(failedInitScrollPane, BorderLayout.CENTER);
        setVisible(false);
        setVisible(true);
    }

    protected LocationSummaryTablePanel getLocationPanel()
    {
        return this._locationPanel;
    }

    public ParameterEstimatorRunInfo getRunInfo()
    {
        return Futures.getUnchecked(_runInfo);
    }

    @Override
    public void dispose()
    {
        if(isAlive())
        {
            try
            {
                final ParameterEstimatorRunInfo info = getRunInfo();
                if(info != null)
                {
                    info.killSaveThread();
                    info.saveWorkingRunInfoToStandardFile();
                }
                _stepsPanel = null;
                _locationPanel = null;
                _steps = null;
                LOG.info("Run-time information saved successfully.");
            }
            catch(final Exception e)
            {
                e.printStackTrace();
                JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(this),
                                              "Unable to save run-time information:\n" + e.getMessage() + "\n"
                                                  + this.getClass().getSimpleName()
                                                  + " will not be able to recover previous run settings when next executed.",
                                              "Error Recording Run-time Information!",
                                              JOptionPane.ERROR_MESSAGE);
            }
        }
        _alive = false;
    }

    @Override
    public boolean isAlive()
    {
        return _alive;
    }

    @Override
    public void run(final FewsEnvironment fewsEnv, final String arg1)
    {
        _alive = true;
    }

    @Override
    public void toFront()
    {
    }

    @Override
    @Subscribe
    public void reactToRefreshFullDisplay(final RefreshFullDisplayNotice evt)
    {
        _locationPanel.refreshTable();
    }
}
