package ohd.hseb.hefs.pe.gui;

import java.awt.Color;
import java.awt.Component;
import java.util.List;

import javax.swing.JTable;
import javax.swing.table.TableCellRenderer;

import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.core.ParameterEstimatorTableModel;
import ohd.hseb.hefs.pe.notice.SelectedIdentifiersChangedNotice;
import ohd.hseb.hefs.pe.notice.StepStatusRefreshAllNotice;
import ohd.hseb.hefs.pe.notice.StepUnitsUpdatedNotice;
import ohd.hseb.hefs.pe.notice.StepUpdatedNotice;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifierList;
import ohd.hseb.hefs.utils.gui.jtable.models.SelectionMaintainingTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.WrapRendererTableModel;
import ohd.hseb.hefs.utils.gui.jtable.renderers.ForwardingTableCellRenderer;
import ohd.hseb.hefs.utils.gui.tools.ColorTools;
import ohd.hseb.hefs.utils.status.StatusLabel;

import com.google.common.eventbus.Subscribe;

/**
 * Model used in the table displayed within a LocationSummaryTablePanel. There should be no reason to subclass this. It
 * subscribes to changes in the selected identifiers via {@link SelectedIdentifiersChangedNotice} events and requests to
 * refresh all status via {@link StepStatusRefreshAllNotice} events.
 * 
 * @author hank.herr
 */
@SuppressWarnings("serial")
public class LocationSummaryTableModel extends ParameterEstimatorTableModel implements
SelectedIdentifiersChangedNotice.Subscriber, StepStatusRefreshAllNotice.Subscriber, StepUpdatedNotice.Subscriber,
SelectionMaintainingTableModel<LocationAndDataTypeIdentifier>, WrapRendererTableModel
{
    public final static String PRECIPITATION_PARAMETER_CELL_ENTRY = "precip";
    public final static String TEMPERATURE_PARAMETER_CELL_ENTRY = "temp";

    private List<ParameterEstimatorStepProcessor> _steps = null;
    private LocationAndDataTypeIdentifierList _identifiers = new LocationAndDataTypeIdentifierList();

    public LocationSummaryTableModel(final ParameterEstimatorRunInfo runInfo,
                                     final List<ParameterEstimatorStepProcessor> steps)
    {
        super(runInfo);
        _steps = steps;
        _identifiers = getRunInfo().getSelectedIdentifiers();
    }

    public List<LocationAndDataTypeIdentifier> getIdentifiers()
    {
        return this._identifiers;
    }

    public ParameterEstimatorStepProcessor getStepForColumn(final int colIndex)
    {
        if(colIndex >= 2)
        {
            return this._steps.get(colIndex - 2);
        }
        return null;
    }

    public int getColumnForStep(final ParameterEstimatorStepProcessor step)
    {
        return _steps.indexOf(step) + 2;
    }

    public int getColumnForStep(final Class<? extends ParameterEstimatorStepProcessor> step)
    {
        int index = 2;
        for(final ParameterEstimatorStepProcessor s: _steps)
        {
            if(step.isInstance(s))
            {
                return index;
            }
            index++;
        }
        return -1;
    }

    public LocationAndDataTypeIdentifier getIdentifierForRow(final int row)
    {
        if(row >= 0)
        {
            return getIdentifiers().get(row);
        }
        return null;
    }

    public int getRowForIdentifier(final LocationAndDataTypeIdentifier identifier)
    {
        for(int row = 0; row < _identifiers.size(); row++)
        {
            if(_identifiers.get(row).equals(identifier))
            {
                return row;
            }
        }
        return -1;
    }

    public int getRowForLocation(final String locationId)
    {
        for(int row = 0; row < _identifiers.size(); row++)
        {
            if(_identifiers.get(row).getLocationId().equals(locationId))
            {
                return row;
            }
        }
        return -1;
    }

    @Override
    public int getRowCount()
    {
        return _identifiers.size();
    }

    @Override
    public int getColumnCount()
    {
        //+2 -- locationId and parameterId.
        return _steps.size() + 2;
    }

    @Override
    public Object getValueAt(final int rowIndex, final int columnIndex)
    {
        final LocationAndDataTypeIdentifier identifier = this._identifiers.get(rowIndex);
        if(columnIndex == 0)
        {
            return identifier.getLocationId();
        }
        else if(columnIndex == 1)
        {
            if(identifier.isPrecipitationDataType())
            {
                return PRECIPITATION_PARAMETER_CELL_ENTRY;
            }
            else if(identifier.isTemperatureDataType())
            {
                return TEMPERATURE_PARAMETER_CELL_ENTRY;
            }
            else
            {
                return null;
            }
        }
        else if(columnIndex >= 2)
        {
            final int stepIndex = columnIndex - 2;
            return StatusLabel.make(_steps.get(stepIndex).getStatus(identifier));
        }
        return null;
    }

    @Override
    public boolean isCellEditable(final int rowIndex, final int columnIndex)
    {
        return false;
    }

    @Override
    public Class<?> getColumnClass(final int columnIndex)
    {
        if(columnIndex >= 2)
        {
            return StatusLabel.class;
        }
        return String.class;
    }

    @Override
    public String getColumnName(final int column)
    {
        if(column == 0)
        {
            return "Location ID";
        }
        else if(column == 1)
        {
            return "Parameter ID";
        }
        else
        {
            return _steps.get(column - 2).getShortNameForIdentifierTableColumn();
        }
    }

    @Override
    public String getColumnHeaderToolTip(final int modelColIndex)
    {
        //1 is used here because it is the visible column index, and the parameter column was made invisible.
        if(modelColIndex > 1)
        {
            return this._steps.get(modelColIndex - 2).getToolTipTextForIdentifierTableColumnHeader();
        }
        return "";
    }

    @Override
    public String getCellToolTip(final int rowIndex, final int modelColIndex)
    {
        //1 is used here because it is the visible column index, and the parameter column was made invisible.
        if(modelColIndex > 1)
        {
            return _steps.get(modelColIndex - 2).getStatus(getIdentifierForRow(rowIndex)).getDescription();
        }
        return "";
    }

    @Override
    @Subscribe
    public void reactToStepUpdated(final StepUpdatedNotice evt)
    {
        //For a step unit, just fire cell updated for the specific identifiers.  Note that if we update everything (ala below),
        //flicker may be caused (wherein it repaints the table unsorted and then sorted) if the rows are sorted.  This avoids
        //that problem because the table will not resort from scratch.
        if(evt instanceof StepUnitsUpdatedNotice)
        {
            final StepUnitsUpdatedNotice<ParameterEstimatorStepProcessor, LocationAndDataTypeIdentifier> unitNotice = (StepUnitsUpdatedNotice)evt;
            final int col = getColumnForStep(unitNotice.getStep());
            for(final LocationAndDataTypeIdentifier identifier: unitNotice.getUnits())
            {
                //The parameterId within this identifier may not necessary be the standard id.  For example, it may be FMAP instead of
                //MAP.  Hence, we only want to check the location and general data type.
                final int row = getRowForLocation(identifier.getLocationId());
                if((row >= 0)
                    && (identifier.isPrecipitationDataType() == getIdentifierForRow(row).isPrecipitationDataType()))
                {
                    fireTableCellUpdated(row, col);
                }
            }
        }
        //Otherwise, update the whole shebang.
        else
        {
            fireTableDataChanged();
        }
    }

    @Override
    @Subscribe
    public void reactToSelectedIdentifiersChanged(final SelectedIdentifiersChangedNotice evt)
    {
        _identifiers = getRunInfo().getSelectedIdentifiers();
        fireTableDataChanged();
    }

    @Override
    @Subscribe
    public void reactToStepStatusRefreshAll(final StepStatusRefreshAllNotice notice)
    {
        this.fireTableDataChanged();
    }

    @Override
    public LocationAndDataTypeIdentifier getRow(final int index)
    {
        return _identifiers.get(index);
    }

    @Override
    public int getRowIndex(final LocationAndDataTypeIdentifier row)
    {
        return _identifiers.indexOf(row);
    }

    @Override
    public TableCellRenderer wrapRenderer(final TableCellRenderer baseRenderer)
    {
        return new FadeRenderer(baseRenderer);
    }

    /**
     * If the colomn or row of a cell is selected, but the cell is not, this applies the half selected color as the
     * background to the cell.
     * 
     * @author hankherr
     */
    private class FadeRenderer extends ForwardingTableCellRenderer
    {
        private Color _halfSelectColor = null;

        public FadeRenderer(final TableCellRenderer delegate)
        {
            super(delegate);
        }

        @Override
        public Component getTableCellRendererComponent(final JTable table,
                                                       final Object value,
                                                       final boolean isSelected,
                                                       final boolean hasFocus,
                                                       final int row,
                                                       final int column)
        {
            if(_halfSelectColor == null)
            {
                _halfSelectColor = ColorTools.mixColors(getTable().getSelectionBackground(), Color.white);
            }
            final Component c = super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);

            if((getTable().isRowSelected(row) || getTable().isColumnSelected(column)) && !isSelected)
            {
                c.setBackground(_halfSelectColor);
            }
            return c;
        }
    }

    @Override
    public boolean applyAfterAllOtherRenderers()
    {
        return true;
    }

}
