package ohd.hseb.hefs.pe.gui;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JToolBar;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.core.ParameterEstimatorSubPanel;
import ohd.hseb.hefs.pe.notice.GotoStepAndUnitNotice;
import ohd.hseb.hefs.pe.notice.StepStatusRefreshAllNotice;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.gui.help.HelpFile;
import ohd.hseb.hefs.utils.gui.jtable.GenericTable;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.jobs.HJobMonitorDialog;
import ohd.hseb.hefs.utils.jobs.JobListener;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tools.StringTools;

import com.google.common.collect.ImmutableList;

/**
 * Wraps a location summary table, displaying data from a LocationSummaryTableModel. Multiple summary tables can be
 * displayed, one per estimation data type, and the active table is selected based on a choice box. The constructor is
 * general enough that there should be no reason to subclass this class.
 * 
 * @author hank.herr
 */
//@HelpFile("helpManual.htm#LocationSummaryPanelRef")
@HelpFile("PDFHELP:helpManual.pdf#LocationSummaryPanelRef")
@SuppressWarnings("serial")
public class LocationSummaryTablePanel extends ParameterEstimatorSubPanel implements ListSelectionListener,
ActionListener
{
    private final List<ParameterEstimatorStepProcessor> _steps;

    private final List<ParameterId.Type> _dataTypes;
    private JComboBox _dataTypeChoices = null;
    private final GenericTable<LocationAndDataTypeIdentifier> _table;
    private JScrollPane _tablePane;
    private final JButton _refreshButton = HSwingFactory.createJButtonWithIcon("hefsIcons/refresh20x20.png",
                                                                               "Refresh Status Columns",
                                                                               this);
    private final JButton _gotoSelectedButton = HSwingFactory.createJButtonWithIcon("hefsIcons/gotoSelected20x20.png",
                                                                                    "Goto Step Panel for Selected Cell(s)",
                                                                                    this);
    private final JButton _runAllSelectedButton = HSwingFactory.createJButtonWithIcon("hefsIcons/playAllBlueGreen20x20.png",
                                                                                      "Run All Steps for Selected Rows",
                                                                                      this);

    private Class<? extends PerformAllStepsJob> _performAllJobClass = PerformAllStepsJob.class;

    /**
     * @param runInformation The ParameterEstimatorRunInfo, used to acquire identifiers to display.
     * @param steps The ParameterEstimatorStepProcessor instances, used to determine the state of the steps.
     * @param dataTypeParameterIds List of different data type example parameterIds for which parameters can be
     *            estimated. HEFSUtils is used to determine if the provided parameter id is precipitation, temperature,
     *            stream flow, or any other type of data is works with.
     * @param dataTypesDisplayedStrings List of string to display for the data types.
     */
    public LocationSummaryTablePanel(final ParameterEstimatorRunInfo runInfo,
                                     final List<ParameterEstimatorStepProcessor> steps,
                                     final EnumSet<ParameterId.Type> dataTypeParameterIds)
    {
        super(runInfo);
        _steps = steps;
        _dataTypes = ImmutableList.copyOf(dataTypeParameterIds);
        _dataTypeChoices = HSwingFactory.createJComboBox(ParameterId.Type.makePrintNameList(dataTypeParameterIds), null);
        _table = new GenericTable<LocationAndDataTypeIdentifier>(new LocationSummaryTableModel(runInfo, _steps));

        initializeDisplay();
        updateButtonEnabledness();
    }

    /**
     * Call to use a subclass of PerformAllStepsJob to do a run all if necessary.
     * 
     * @param job Instance of a subclass of PerformAllStepsJob that will be used to run all steps. Its initialize method
     *            will be called whenever a run is to be performed.
     */
    public void setPerformAllJobToUse(final Class<? extends PerformAllStepsJob> job)
    {
        _performAllJobClass = job;
    }

    @Override
    protected void initializeDisplay()
    {
        this.setLayout(new BorderLayout());

        _dataTypeChoices.setSelectedIndex(0);
        _dataTypeChoices.addActionListener(this);
        final JPanel choicesPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        choicesPanel.add(new JLabel("Select type of data for estimation: "));
        choicesPanel.add(_dataTypeChoices);
        add(choicesPanel, BorderLayout.NORTH);

        _table.setAutoCreateRowSorter(true);
        _table.setColumnSelectionAllowed(true);
        _table.getSelectionModel().setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
        _table.getSelectionModel().addListSelectionListener(this);
        _table.getColumnModel().getSelectionModel().addListSelectionListener(this);
        _table.getColumnModel().getSelectionModel().setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        initializeColumnWidths(_table);

        _tablePane = new JScrollPane(_table);
        final JPanel tablePaneSubPanel = new JPanel(new BorderLayout());
        tablePaneSubPanel.add(new JLabel("Summary of locations for parameter estimation:"), BorderLayout.NORTH);
        tablePaneSubPanel.add(_tablePane, BorderLayout.CENTER);
        add(tablePaneSubPanel, BorderLayout.CENTER);

        //Button panel
        final JToolBar toolBar = new JToolBar();
        toolBar.setFloatable(false);
        toolBar.add(this._gotoSelectedButton);
        toolBar.add(_runAllSelectedButton);
        toolBar.add(HSwingFactory.createFillerJPanel());
        toolBar.add(_refreshButton);
        add(toolBar, BorderLayout.SOUTH);
    }

    private void initializeColumnWidths(final JTable table)
    {
        table.getColumnModel().getColumn(0).setPreferredWidth(500); //Very large so that it gets most of the initial width
        table.getColumnModel().getColumn(1).setPreferredWidth(0);
        for(int i = 2; i < table.getModel().getColumnCount(); i++)
        {
            //Allow column width change by user
            table.getColumnModel().getColumn(i).setPreferredWidth(45);
            table.getColumnModel().getColumn(1).setMinWidth(35);//Allow column width change
        }

        table.getColumnModel().removeColumn(table.getColumnModel().getColumn(1));
    }

    private void reactToNewDataTypeSelected()
    {
        final int index = this._dataTypeChoices.getSelectedIndex();
        getRunInfo().setSelectedDataType(_dataTypes.get(index), this);
    }

    private void fireGotoStepForLocation()
    {
        final ParameterEstimatorStepProcessor stepProc = ((LocationSummaryTableModel)_table.getModel()).getStepForColumn(_table.convertColumnIndexToModel(_table.getSelectedColumn()));
        if(stepProc != null)
        {
            post(new GotoStepAndUnitNotice<LocationAndDataTypeIdentifier>(this, stepProc, _table.getSelectedWholeRows()));
        }
    }

    public void refreshTable()
    {
        post(new StepStatusRefreshAllNotice(this)); //This should force a full redraw of the table without calling setVisible(...) twice.
    }

    private void updateButtonEnabledness()
    {
        if(this._table.getSelectedRowCount() == 1)
        {
            this._gotoSelectedButton.setEnabled(true);
            this._runAllSelectedButton.setEnabled(true);
        }
        else if(this._table.getSelectedRowCount() > 1)
        {
            this._gotoSelectedButton.setEnabled(true);
            this._runAllSelectedButton.setEnabled(true);
        }
        else
        //zero
        {
            this._gotoSelectedButton.setEnabled(false);
            this._runAllSelectedButton.setEnabled(false);
        }

        //If column 0 is selected, disable goto
        if(this._table.getSelectedColumn() == 0)
        {
            this._gotoSelectedButton.setEnabled(false);
        }
    }

    private void performAllStepsForSelected()
    {
        if(_table.getSelectedRowCount() == 0)
        {
            JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(LocationSummaryTablePanel.this),
                                          "No location has been selected for which to perform the step.\nPlease select a location to perform step.",
                                          "Step Failed!",
                                          JOptionPane.ERROR_MESSAGE);
            return;
        }

        //Build list of selected identifiers
        final List<LocationAndDataTypeIdentifier> selectedIdentifiers = new ArrayList<LocationAndDataTypeIdentifier>();
        for(final int row: _table.getSelectedRows())
        {
            selectedIdentifiers.add(_table.getWholeRow(row));
        }

        //Verify step
        //String allLocationsStr = identifier.buildStringToDisplayInTree();

        //Initialize the job
        PerformAllStepsJob performAllJob = null;
        try
        {
            performAllJob = _performAllJobClass.newInstance();
            performAllJob.initialize(getRunInfo(), _steps, selectedIdentifiers, true);
        }
        catch(final Exception e)
        {
            JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(this),
                                          StringTools.wordWrap("Unable to perform all steps:\n" + e.getMessage(), 100),
                                          "Error Performing All Steps!",
                                          JOptionPane.ERROR_MESSAGE);
            return;
        }
        final HJobMonitorDialog jobDialog = new HJobMonitorDialog(SwingTools.getGlobalDialogParent(this),
                                                                  "Perform All Steps",
                                                                  performAllJob,
                                                                  true);
        performAllJob.setParentComponentForMessages(SwingTools.getGlobalDialogParent(this));
        performAllJob.addListener(new JobListener()
        {
            @Override
            public void processJobFailure(final Exception exc, final GenericJob theJob, final boolean displayMessage)
            {
                if(theJob.isCanceled())
                {
                    ((PerformAllStepsJob)theJob).displayExceptions("Perform all steps job has been cancelled.");
                }
                else
                {
                    JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(LocationSummaryTablePanel.this),
                                                  StringTools.wordWrap(exc.getMessage(), 100),
                                                  "Perform All Error Occurred!",
                                                  JOptionPane.ERROR_MESSAGE);
                }
                jobDialog.setVisible(false);
                setVisible(false);
                setVisible(true);
                SwingTools.forceComponentRedraw(_table); //Make sure the table status icons are fully painted.
            }

            @Override
            public void processSuccessfulJobCompletion(final GenericJob theJob)
            {
                ((PerformAllStepsJob)theJob).displayExceptions("Perform all steps job has completed.");
                setVisible(false);
                setVisible(true);
                SwingTools.forceComponentRedraw(_table); //Make sure the table status icons are fully painted.
            }
        });

        //Run all steps.  I need to create another thread to run the job because I do not plan on 
        //reconstructing _performAllJob.  Since you cannot stop and restart a thread in Java, I need
        //to perform the run() method of _performAllJob within another thread that can be disposed of.
        performAllJob.startJob();
        jobDialog.setMinimumSize(new Dimension(700, 10));
        jobDialog.setModal(true);
        jobDialog.setVisible(true);

    }

    @Override
    public void valueChanged(final ListSelectionEvent e)
    {
        //Prevents painting issues due to the fancy, schmancy selected row/column coloring
        SwingTools.forceComponentRedraw(_table);

        updateButtonEnabledness();
    }

    @Override
    public void actionPerformed(final ActionEvent e)
    {
        if(e.getSource() == this._dataTypeChoices)
        {
            reactToNewDataTypeSelected();
        }
        else if(e.getSource() == this._refreshButton)
        {
            refreshTable();
        }
        else if(e.getSource() == this._gotoSelectedButton)
        {
            fireGotoStepForLocation();
        }
        else if(e.getSource() == this._runAllSelectedButton)
        {
            performAllStepsForSelected();
        }
    }

}
