package ohd.hseb.hefs.pe.gui;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import javax.swing.JToolBar;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepOptionsPanel;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.core.ParameterEstimatorSubPanel;
import ohd.hseb.hefs.pe.core.StepUnit;
import ohd.hseb.hefs.pe.notice.GotoStepAndUnitNotice;
import ohd.hseb.hefs.pe.notice.RefreshFullDisplayNotice;
import ohd.hseb.hefs.pe.notice.StepUnitsSelectedNotice;
import ohd.hseb.hefs.utils.gui.about.AboutDialog;
import ohd.hseb.hefs.utils.gui.help.ClickHelpManager;
import ohd.hseb.hefs.utils.gui.help.HelpFile;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.jobs.HJobMonitorDialog;
import ohd.hseb.hefs.utils.jobs.JobListener;
import ohd.hseb.hefs.utils.tools.StringTools;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.eventbus.Subscribe;

/**
 * General steps panel that should not need to be subclassed. The constructor allows for specifying run information,
 * steps, and a setup panel. Given those three, the rest of the panel is generally applicable to any type of estimation
 * program that fits in the general PE framework.
 * 
 * @author hank.herr
 */
@SuppressWarnings("serial")
@HelpFile("PDFHELP:helpManual.pdf#EstimationStepsPanelRef")
public class ParameterEstimatorStepsPanel extends ParameterEstimatorSubPanel implements ActionListener,
StepUnitsSelectedNotice.Subscriber, GotoStepAndUnitNotice.Subscriber, RefreshFullDisplayNotice.Subscriber
{
    private static final Logger LOG = LogManager.getLogger(ParameterEstimatorStepsPanel.class);

    private final PESetupPanel _setupPanel;
    private final Component _componentForHelpSearch;
    private List<ParameterEstimatorStepProcessor> _steps = null;
    private final List<ParameterEstimatorStepOptionsPanel> _tabbedPanels = new ArrayList<ParameterEstimatorStepOptionsPanel>();
    private JTabbedPane _tabbedPane;
    private final JButton _backButton = HSwingFactory.createJButtonWithIcon("hefsIcons/navigate_left_24x24.png",
                                                                            "Back",
                                                                            this);
    private final JButton _nextButton = HSwingFactory.createJButtonWithIcon("hefsIcons/navigate_right_24x24.png",
                                                                            "Next",
                                                                            this);
    private final JButton _runButton = HSwingFactory.createJButtonWithIcon("hefsIcons/playBlue24x24.png",
                                                                           "Perform Step",
                                                                           this);
    private final JButton _saveRunInfoButton = HSwingFactory.createJButtonWithIcon("hefsIcons/save24x24.png",
                                                                                   "Save Run-time Information",
                                                                                   this);

    /**
     * @param runInfo ParameterEstimatorRunInfo instance that is used by the type of parameter estimator.
     * @param steps Steps that are performed for parameter estimation.
     * @param setupPanel A setup panel to be displayed to setup for estimation.
     * @param componentForHelpSearch If null, this is used. Its first parent which is a JRootPane will be used for
     *            handling help clicks.
     */
    public ParameterEstimatorStepsPanel(final ParameterEstimatorRunInfo runInfo,
                                        final List<ParameterEstimatorStepProcessor> steps,
                                        final PESetupPanel setupPanel,
                                        final Component componentForHelpSearch)
    {
        super(runInfo);
        _componentForHelpSearch = componentForHelpSearch;
        _steps = steps;
        _setupPanel = setupPanel;
        initializeDisplay();
        updateRunButton(new ArrayList<StepUnit>());
    }

    @Override
    protected void initializeDisplay()
    {
        //Tabbed Pane
        _tabbedPane = new JTabbedPane();
        _tabbedPane.addTab("Setup", _setupPanel);
        for(final ParameterEstimatorStepProcessor step: _steps)
        {
            final ParameterEstimatorStepOptionsPanel panel = step.constructOptionsPanel();
            if(panel != null)
            {
                _tabbedPane.addTab(step.getTabNameForStep(),
                                   null,
                                   panel,
                                   Strings.nullToEmpty(step.getToolTipTextDescribingStep()));
                _tabbedPanels.add(panel);
                panel.reactToSelectedIdentifiersChanged(getRunInfo().generateDataTypeSelectedEvent(this));
                // Changed to BiMap - should only need to place once.
                // _processorToPanelMap.put(step, panel);
            }
        }
        _tabbedPane.addChangeListener(new ChangeListener()
        {
            @Override
            public void stateChanged(final ChangeEvent e)
            {
                updateEnablednessOfBackAndNextButton();
                updateRunButton(getStepUnitsToPerform());
            }
        });

        //Button panel
//        JPanel leftButtonPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
//        leftButtonPanel.add(_backButton);
//        leftButtonPanel.add(_runButton);
//        leftButtonPanel.add(_nextButton);
//        leftButtonPanel.add(_saveRunInfoButton);
//        JPanel rightButtonPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
//        rightButtonPanel.add(AboutDialog.createAboutDialogButton(this, getRunInfo().getProgramNameForAboutDialog()));
//        rightButtonPanel.add(new JButton(new ClickHelpManager(getRunInfo().getHelpDirectory(), _componentForHelpSearch).startHelpAction));
//        JPanel buttonPanel = new JPanel(new BorderLayout());
//        buttonPanel.add(leftButtonPanel, BorderLayout.WEST);
//        buttonPanel.add(new BufferJPanel(), BorderLayout.CENTER);
//        buttonPanel.add(rightButtonPanel, BorderLayout.EAST);

        final JToolBar buttonBar = new JToolBar();
        buttonBar.add(_backButton);
        buttonBar.add(_runButton);
        buttonBar.add(_nextButton);
        buttonBar.add(_saveRunInfoButton);
        buttonBar.add(new JPanel(new BorderLayout()));
        buttonBar.add(AboutDialog.createAboutDialogButton(this, getRunInfo().getProgramNameForAboutDialog()));
        buttonBar.add(ClickHelpManager.createHelpButton(getRunInfo().getHelpDirectory(), _componentForHelpSearch));
        buttonBar.setFloatable(false);

        //Main panel
        this.setLayout(new BorderLayout());
        this.add(_tabbedPane, BorderLayout.CENTER);
//        this.add(buttonPanel, BorderLayout.SOUTH);
        this.add(buttonBar, BorderLayout.SOUTH);
    }

    private void updateEnablednessOfBackAndNextButton()
    {
        //Set the buttons accordingly
        _backButton.setEnabled(true);
        _nextButton.setEnabled(true);
        if(_tabbedPane.getSelectedIndex() == 0)
        {
            _backButton.setEnabled(false);
        }
        if(_tabbedPane.getSelectedIndex() == _tabbedPane.getTabCount() - 1)
        {
            _nextButton.setEnabled(false);
        }
    }

    /**
     * Updates the enabledness of the run button based on the number of selected step units and whether a run button
     * exists for the panel.
     */
    private void updateRunButton(final List<? extends StepUnit> unitsSelected)
    {
        //Set run button  name
        if(getSelectedStep() == null)
        {
            _runButton.setVisible(false);
        }
        else if(getSelectedStep().getStepNameForRunButton() == null)
        {
            if(getSelectedStep().getToolTipTextDescribingStep() == null)
            {
                _runButton.setVisible(false);
            }
            else
            {
                _runButton.setToolTipText(getSelectedStep().getToolTipTextDescribingStep());
                _runButton.setEnabled(false);
                _runButton.setVisible(true);
            }
        }
        else
        {
            _runButton.setToolTipText("<html>" + getSelectedStep().getStepNameForRunButton() + "<br>"
                + getSelectedStep().getToolTipTextDescribingStep() + "</html>");
            _runButton.setEnabled(true);
            _runButton.setVisible(true);
            if(unitsSelected.size() <= 0)
            {
                _runButton.setEnabled(false);
            }
            else
            {
                _runButton.setEnabled(true);
            }
        }
    }

    /**
     * @return The currently active {@link ParameterEstimatorSubPanel}.
     */
    private ParameterEstimatorSubPanel getSelectedPanel()
    {
        return (ParameterEstimatorSubPanel)_tabbedPane.getSelectedComponent();
    }

    /**
     * @return The currently selected {@link ParameterEstimatorStepProcessor}.
     */
    private ParameterEstimatorStepProcessor getSelectedStep()
    {
        final JPanel panel = getSelectedPanel();
        if(panel instanceof ParameterEstimatorStepOptionsPanel)
        {
            return ((ParameterEstimatorStepOptionsPanel)panel).getStepProcessor();
        }
        else
        {
            return null;
        }
    }

    /**
     * @return A List of {@link StepUnit} instances, each being a unit for which the current step is selected to be
     *         performed.
     */
    private List<? extends StepUnit> getStepUnitsToPerform()
    {
        final JPanel panel = getSelectedPanel();
        if(panel instanceof ParameterEstimatorStepOptionsPanel)
        {
            return ((ParameterEstimatorStepOptionsPanel)panel).getStepUnitsToPerform();
        }
        else
        {
            return Lists.newArrayList();
        }
    }

    /**
     * Whether successful or not, a
     */
    private void performStepForSelectedPanel()
    {
        //Check for identifiers
        final List<? extends StepUnit> units = getStepUnitsToPerform();
        if(units.isEmpty())
        {
            JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(ParameterEstimatorStepsPanel.this),
                                          "Nothing has been selected for which to perform the step.\nPlease select a location or group to perform step.",
                                          "Step Failed!",
                                          JOptionPane.ERROR_MESSAGE);
            return;
        }

        // Create step
        final ParameterEstimatorStepProcessor step = getSelectedStep();

        // Pre-step check
        try
        {
            step.prepareStep(false);
        }
        catch(final Exception e)
        {
            JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(this),
                                          "Could not prepare for step:\n" + e.getMessage() + "\nAborting.",
                                          "Error Preparing Step!",
                                          JOptionPane.ERROR_MESSAGE);
            return;
        }

        //Verify step
        final int result = JOptionPane.showOptionDialog(SwingTools.getGlobalDialogParent(this),
                                                        step.constructConfirmRunPanel(units),
                                                        "Continue?",
                                                        JOptionPane.OK_CANCEL_OPTION,
                                                        JOptionPane.QUESTION_MESSAGE,
                                                        null,
                                                        new String[]{"Continue", "Cancel"},
                                                        "Continue");
        if(result == 1)
        {
            return;
        }

        //Perform the step
        final PerformStepJob task = new PerformStepJob(getRunInfo(), step, units);
        final HJobMonitorDialog monitorDialog = new HJobMonitorDialog(this,
                                                                      step.getStepNameForRunButton(),
                                                                      task,
                                                                      true,
                                                                      step.concurrencySupported());
        task.setParentComponentForMessages(SwingTools.getGlobalDialogParent(this));
        task.addListener(new JobListener()
        {
            @Override
            public void processJobFailure(final Exception exc, final GenericJob theJob, final boolean displayMessage)
            {
                monitorDialog.setVisible(false);
                if(displayMessage)
                {
                    //Message may include \n, which should display fine in the message dialog.
                    JOptionPane.showMessageDialog(ParameterEstimatorStepsPanel.this,
                                                  StringTools.wordWrap(exc.getMessage(), 100),
                                                  "Step Failed!",
                                                  JOptionPane.ERROR_MESSAGE);
                }
                setVisible(false);
                setVisible(true);
            }

            @Override
            public void processSuccessfulJobCompletion(final GenericJob theJob)
            {
                setVisible(false);
                setVisible(true);
            }
        });
        monitorDialog.setMinimumSize(new Dimension(700, 10));
        task.startJob();

        //XXX To allow this to perform multiple estimations at once, I **think** the monitorDialog 
        //will need to be made NOT modal.  Also, the step processor keeps track of the currently
        //running identifier; it will need to be updated to allow for multiple identifiers and 
        //to ensure that the user does not run an already running identifier.
        monitorDialog.setAlwaysOnTop(false);
        monitorDialog.setModal(true);
        monitorDialog.setVisible(true);

        //For future reference, this can be done to make it so that only one job runs at a time.
        //The HJobManager should be a static created somewhere else.  The above start command can be
        //removed... it is handled by the job manager.
        //private final HJobManager _jobManager = new HJobManager(this);
        //_jobManager.addJob(task);
    }

    @Override
    @Subscribe
    public void reactToStepProcessEnabled(final StepUnitsSelectedNotice evt)
    {
        updateRunButton(evt.getUnits());
    }

    @Override
    @Subscribe
    public void reactToGotoStepAndUnit(final GotoStepAndUnitNotice evt)
    {
        _tabbedPane.setSelectedIndex(_steps.indexOf(evt.getStep()) + 1); //+1 accounts for Setup.
    }

    @Override
    public void actionPerformed(final ActionEvent e)
    {
        if(e.getSource() == this._backButton)
        {
            _tabbedPane.setSelectedIndex(_tabbedPane.getSelectedIndex() - 1);
        }
        else if(e.getSource() == this._nextButton)
        {
            _tabbedPane.setSelectedIndex(_tabbedPane.getSelectedIndex() + 1);
        }
        else if(e.getSource() == this._runButton)
        {
            performStepForSelectedPanel();
        }
        else if(e.getSource() == this._saveRunInfoButton)
        {
            try
            {
                if(getRunInfo() != null)
                {
                    getRunInfo().saveWorkingRunInfoToStandardFile();
                    LOG.info("Run-time information saved successfully.");
                }
            }
            catch(final Exception exception)
            {
                exception.printStackTrace();
                JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(this),
                                              "Unable to save run-time information:\n" + exception.getMessage(),
                                              "Error Recording Run-time Information!",
                                              JOptionPane.ERROR_MESSAGE);
            }
        }
    }

    @Override
    @Subscribe
    public void reactToRefreshFullDisplay(final RefreshFullDisplayNotice evt)
    {
        this.setVisible(false);
        this.setVisible(true);
    }
}
