package ohd.hseb.hefs.pe.gui;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.JTree;

import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.notice.StepUnitsUpdatedNotice;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.jtree.JTreeTools;
import ohd.hseb.hefs.utils.tools.ListTools;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

/**
 * Performs all steps. It will open an options panel associated with the run that allows for specifying whether to exit
 * if any failure occurs and specifying whether to rerun steps that are already completed and up to date. Instances of
 * this class are meant to allow for multiple calls of run. Whenever a new run-all job is required, call the initialize
 * method accordingly.<br>
 * <br>
 * This class may need to be overridden if additional options are required or if specific steps need to be performed
 * differently.
 * 
 * @author hank.herr
 */
public class PerformAllStepsJob extends GenericJob
{
    private static final Logger LOG = LogManager.getLogger(PerformAllStepsJob.class);

    /**
     * Identifiers for which to perform steps.
     */
    private final List<LocationAndDataTypeIdentifier> _selectedIdentifiers = new ArrayList<LocationAndDataTypeIdentifier>();

    /**
     * The steps to perform.
     */
    private final List<ParameterEstimatorStepProcessor> _steps = new ArrayList<ParameterEstimatorStepProcessor>();

    /**
     * {@link Component} used for the location of dialogs.
     */
    private Component _parentComponentForMessaging = null;

    /**
     * Flag indicates if the process should exit out with any failure. Defaults to false.
     */
    private boolean _exitIfAnyFailureOccurs = false;

    /**
     * Flag indicates if steps already complete should be run again. Defaults to false.
     */
    private boolean _rerunCompletedStepsFlag = false;

    /**
     * The run information object.
     */
    private ParameterEstimatorRunInfo _runInformation;

    /**
     * String is displayed in a lot of messaging and can be overridden to use different language.
     */
    private String _locationMessageStr = "Steps being performed";

    /**
     * Keeps track of the exceptions accumulated for all steps and all locations.
     */
    private final LinkedHashMap<LocationAndDataTypeIdentifier, LinkedHashMap<ParameterEstimatorStepProcessor, Exception>> _allExceptions = new LinkedHashMap<LocationAndDataTypeIdentifier, LinkedHashMap<ParameterEstimatorStepProcessor, Exception>>();

    /**
     * Empty constructor. Call {@link #initialize(ParameterEstimatorRunInfo, List, List, boolean)} after construction.
     */
    public PerformAllStepsJob()
    {
    }

    /**
     * If the constructor throws an Exception, the user has canceled the job.
     * 
     * @param runInfo The {@link ParameterEstimatorRunInfo} for this run.
     * @param steps Steps involved.
     * @param identifiers Identifiers to run with.
     * @param displayOptionsPane If true, then the options pane will be displayed, providing specific options based on
     *            the steps.
     * @throws Exception If the user cancels the job. No need to do anything when this happens, except do not run this
     *             job!
     */
    public void initialize(final ParameterEstimatorRunInfo runInfo,
                           final List<ParameterEstimatorStepProcessor> steps,
                           final List<LocationAndDataTypeIdentifier> identifiers,
                           final boolean displayOptionsPane) throws Exception
    {
        _runInformation = runInfo;

        //Job flags...
        setDone(false);
        setCanceledAndUpdateProgress(false);
        clearJobMonitorAttr();

        _allExceptions.clear();
        _selectedIdentifiers.clear();
        _selectedIdentifiers.addAll(identifiers);
        _steps.clear();
        _steps.addAll(steps);

        if(displayOptionsPane)
        {
            openRunOptionsPane();
        }
    }

    /**
     * Override to define your own options. By default, the panel returned follows a BoxLayout (vertical) and includes
     * controls for the exit on failure and rerun completed steps options. You can use this return value as a starting
     * point and then continue to add your own options.
     * 
     * @return Panel allowing the user to select options.
     */
    protected JPanel constructOptionsPanel()
    {
        final JLabel exitOnErrorOptionLabel = new JLabel("Do you want to exit estimation if any step fails?");
        final JRadioButton exitOnErrorYesButton = new JRadioButton("Yes");
        final JRadioButton exitOnErrorNoButton = new JRadioButton("No");
        final ButtonGroup exitOnErrorButtonGroup = new ButtonGroup();
        exitOnErrorNoButton.setSelected(true);
        exitOnErrorButtonGroup.add(exitOnErrorYesButton);
        exitOnErrorButtonGroup.add(exitOnErrorNoButton);
        final JPanel exitOnErrorButtonPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        exitOnErrorButtonPanel.add(exitOnErrorYesButton);
        exitOnErrorButtonPanel.add(exitOnErrorNoButton);
        final JPanel exitOnErrorPanel = new JPanel(new BorderLayout());
        exitOnErrorPanel.add(exitOnErrorOptionLabel, BorderLayout.NORTH);
        exitOnErrorPanel.add(exitOnErrorButtonPanel, BorderLayout.CENTER);
        exitOnErrorPanel.setBorder(BorderFactory.createEtchedBorder());

        final ActionListener exitOnErrorActionListener = new ActionListener()
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                _exitIfAnyFailureOccurs = exitOnErrorYesButton.isSelected();
            }
        };
        exitOnErrorYesButton.addActionListener(exitOnErrorActionListener);
        exitOnErrorNoButton.addActionListener(exitOnErrorActionListener);

        final JLabel rerunLabel = new JLabel("Do you want to rerun steps already completed that do not need updating?");
        final JRadioButton rerunYesButton = new JRadioButton("Yes");
        final JRadioButton rerunNoButton = new JRadioButton("No");
        final ButtonGroup rerunButtonGroup = new ButtonGroup();
        rerunButtonGroup.add(rerunYesButton);
        rerunButtonGroup.add(rerunNoButton);
        rerunNoButton.setSelected(true);
        final JPanel rerunButtonPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        rerunButtonPanel.add(rerunYesButton);
        rerunButtonPanel.add(rerunNoButton);
        final JPanel rerunPanel = new JPanel(new BorderLayout());
        rerunPanel.add(rerunLabel, BorderLayout.NORTH);
        rerunPanel.add(rerunButtonPanel, BorderLayout.CENTER);
        rerunPanel.setBorder(BorderFactory.createEtchedBorder());

        final ActionListener rerunActionListener = new ActionListener()
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                _rerunCompletedStepsFlag = rerunYesButton.isSelected();
            }
        };
        rerunYesButton.addActionListener(rerunActionListener);
        rerunNoButton.addActionListener(rerunActionListener);

        final JPanel optionPanel = new JPanel();
        optionPanel.setLayout(new BoxLayout(optionPanel, BoxLayout.Y_AXIS));
        optionPanel.add(exitOnErrorPanel);
        optionPanel.add(rerunPanel);

        return optionPanel;
    }

    /**
     * Opens the panel returned by {@link #constructOptionsPanel()}.
     * 
     * @throws Exception If the user canceled or another exception occurs.
     */
    private void openRunOptionsPane() throws Exception
    {
        final JLabel openingLabel = new JLabel("<html>All steps will be performed for all selected locations.<br>"
            + "All existing estimated parameter files will be backed up prior to any estimation steps.<br>"
            + "If you wish to continue, please choose from the following options and click OK.</html>");

        final JPanel overallPanel = new JPanel(new BorderLayout());
        overallPanel.add(openingLabel, BorderLayout.NORTH);
        overallPanel.add(constructOptionsPanel(), BorderLayout.CENTER);

        final int option = JOptionPane.showConfirmDialog(_parentComponentForMessaging,
                                                         overallPanel,
                                                         "Run All Options",
                                                         JOptionPane.OK_CANCEL_OPTION,
                                                         JOptionPane.QUESTION_MESSAGE);
        if(option != JOptionPane.OK_OPTION)
        {
            throw new Exception("User canceled!");
        }
    }

    public void setParentComponentForMessages(final Component component)
    {
        _parentComponentForMessaging = component;
    }

    public List<LocationAndDataTypeIdentifier> getSelectedIdentifiers()
    {
        return _selectedIdentifiers;
    }

    public void setLocationMessageStr(final String str)
    {
        _locationMessageStr = str;
    }

    /**
     * Adds the exception to {@link #_allExceptions}.
     */
    private void addException(final LocationAndDataTypeIdentifier identifier,
                              final ParameterEstimatorStepProcessor step,
                              final Exception exc)
    {
        //Dump the stack trace so that errors can be seen in stdout if needed.
        exc.printStackTrace();

        LinkedHashMap<ParameterEstimatorStepProcessor, Exception> identifierExceptions = _allExceptions.get(identifier);
        if(identifierExceptions == null)
        {

            identifierExceptions = new LinkedHashMap<ParameterEstimatorStepProcessor, Exception>();
            _allExceptions.put(identifier, identifierExceptions);
        }
        identifierExceptions.put(step, exc);
    }

    /**
     * Override if different steps need to be run in different ways.
     * 
     * @param step
     * @param identifier
     * @throws Exception
     */
    protected void executeStep(final ParameterEstimatorStepProcessor step,
                               final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        step.performStep(identifier);
    }

    /**
     * Runs the steps to perform for the given location. This calls
     * {@link #executeStep(ParameterEstimatorStepProcessor, LocationAndDataTypeIdentifier)} as needed.
     * 
     * @throws Exception If any problems occur.
     */
    private void runStepsForIdentifier(final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        newMonitorSubJob();
        setMaximumNumberOfSteps(_steps.size());
        setIndeterminate(false);
        updateNote(_locationMessageStr + " for " + identifier.buildStringToDisplayInTree() + "...");

        //Loop through the steps, checking for cancellation regularly.
        int index = 0;
        for(final ParameterEstimatorStepProcessor step: this._steps)
        {
            if(this.isCanceled())
            {
                return;
            }

            //Some steps cannot be performed via a run button, and, therefore, cannot be performed via the run-all button.
            //For example plug-in steps that must be prepared via the Setup Panel.  Skip those steps.
            if(step.getStepNameForRunButton() == null)
            {
                LOG.info("Step " + index + " (of " + _steps.size() + "): " + step.getPerformStepPrefix()
                    + " is not a step that can be performed via run-all. Skipping...");
                madeProgress();
                continue;
            }

            index++;
            LOG.info("Step " + index + " (of " + _steps.size() + "): " + step.getPerformStepPrefix() + " for "
                + identifier.buildStringToDisplayInTree() + "...");
            updateNote("Step " + index + " (of " + _steps.size() + "): " + step.getPerformStepPrefix() + " for "
                + identifier.buildStringToDisplayInTree() + "...");

            //If the _rerunCompletedStepsFlag is true or the step needs updating, run the step.
            if((_rerunCompletedStepsFlag) || !step.getStatus(identifier).isReady())
            {
                try
                {
                    executeStep(step, identifier);
                }
                catch(final Exception e)
                {
                    addException(identifier, step, e);

                    //Do not throw an exception unless the _exitIfAnyFailureOccurs flag is true.
                    if(_exitIfAnyFailureOccurs)
                    {
                        clearMonitorSubJob();
                        throw new Exception("Error occurred executing " + step.getPerformStepPrefix() + " for "
                            + identifier.buildStringToDisplayInTree() + ": " + e.getMessage());
                    }
                }
                finally
                {
                    _runInformation.post(new StepUnitsUpdatedNotice(this, step, identifier));
                }
            }

            madeProgress();
        }

        clearMonitorSubJob();
    }

    /**
     * Calls {@link #displayExceptions(String, String)} passing in 'Perform All Errors' for the dialog title.
     */
    public void displayExceptions(final String messageLabelPrefix)
    {
        displayExceptions("Perform All Errors", messageLabelPrefix);
    }

    /**
     * Opens a dialog to display the exceptions contained in {@link #_allExceptions}.
     * 
     * @param dialogTitle The title of the dialog displayed.
     * @param messageLabelPrefix Phrase that preceded "The following errors..." in the header label.
     */
    public void displayExceptions(final String dialogTitle, final String messageLabelPrefix)
    {
        Map map;
        //Only one step... The tree needs to only be one deep.
        if(_steps.size() == 1)
        {
            // Prepare map to use short strings
            map = new LinkedHashMap<String, Exception>();
            for(final Map.Entry<LocationAndDataTypeIdentifier, LinkedHashMap<ParameterEstimatorStepProcessor, Exception>> entry: _allExceptions.entrySet())
            {
                map.put(entry.getKey().buildStringToDisplayInTree(), ListTools.first(entry.getValue().values()));
            }
        }
        //Otherwise it must be deeper.
        else
        {
            // Prepare map to use short strings
            map = new LinkedHashMap<String, LinkedHashMap<ParameterEstimatorStepProcessor, Exception>>();
            for(final Map.Entry<LocationAndDataTypeIdentifier, LinkedHashMap<ParameterEstimatorStepProcessor, Exception>> entry: _allExceptions.entrySet())
            {
                map.put(entry.getKey().buildStringToDisplayInTree(), entry.getValue());
            }
        }

        final JLabel messageLabel = new JLabel(messageLabelPrefix
            + " The following errors occurred while performing all steps:");

        JTree tree = JTreeTools.createTreeFromMap("Exceptions by Location and Step", map);
        if(map.isEmpty()) // no errors found, put in a message stating so.
        {
            tree = JTreeTools.createTreeFromMap("No errors occurred", map);
        }
        final JScrollPane scrollPane = new JScrollPane(tree);
        scrollPane.setPreferredSize(new Dimension(500, 400));
        final JPanel messagePanel = new JPanel(new BorderLayout());
        messagePanel.add(messageLabel, BorderLayout.NORTH);
        messagePanel.add(scrollPane, BorderLayout.CENTER);

        JOptionPane.showMessageDialog(_parentComponentForMessaging,
                                      messagePanel,
                                      dialogTitle,
                                      JOptionPane.WARNING_MESSAGE);
    }

    public boolean isExitIfAnyFailureOccurs()
    {
        return _exitIfAnyFailureOccurs;
    }

    public void setExitIfAnyFailureOccurs(final boolean exitIfAnyFailureOccurs)
    {
        _exitIfAnyFailureOccurs = exitIfAnyFailureOccurs;
    }

    public boolean isRerunCompletedStepsFlag()
    {
        return _rerunCompletedStepsFlag;
    }

    public void setRerunCompletedStepsFlag(final boolean rerunCompletedStepsFlag)
    {
        _rerunCompletedStepsFlag = rerunCompletedStepsFlag;
    }

    public List<ParameterEstimatorStepProcessor> getSteps()
    {
        return _steps;
    }

    public Component getParentComponentForMessaging()
    {
        return _parentComponentForMessaging;
    }

    @Override
    public void processJob()
    {
        //In debug mode, there are issues where the software freezes as the dialog is being
        //drawn, which may be related to initial job progress while building the dialog.
        //As such, I will pause for .1 seconds here to give the dialog time to draw itself.
        try
        {
            Thread.sleep(100);
        }
        catch(final InterruptedException e1)
        {
            e1.printStackTrace();
        }

        LOG.info("All " + _locationMessageStr.toLowerCase() + " for " + _selectedIdentifiers.size() + " locations...");

        setMaximumNumberOfSteps(_selectedIdentifiers.size());
        setIndeterminate(false);
        updateNote("All " + _locationMessageStr.toLowerCase() + " for " + _selectedIdentifiers.size() + " locations...");

        try
        {
            //Perform preparation for each step.  This is identifier independent.
            for(final ParameterEstimatorStepProcessor step: this._steps)
            {
                try
                {
                    step.prepareStep(true);
                }
                catch(final Exception e)
                {
                    final int option = JOptionPane.showConfirmDialog(_parentComponentForMessaging,
                                                                     "While attempting to prepare for running all steps, step preparation failed for "
                                                                         + step.getPerformStepPrefix()
                                                                         + ":\n"
                                                                         + e.getMessage()
                                                                         + "\n"
                                                                         + ".\nDo you want to continue?  (Click No to abort running all steps.)",
                                                                     "Error in Run All Preparation!",
                                                                     JOptionPane.YES_NO_OPTION,
                                                                     JOptionPane.ERROR_MESSAGE);
                    if(option == JOptionPane.NO_OPTION)
                    {
                        throw new Exception("User canceled operation after step preparation failed.");
                    }
                }
            }

            //For each identifier, run the steps.
            int index = 0;
            for(final LocationAndDataTypeIdentifier identifier: _selectedIdentifiers)
            {
                index++;
                LOG.info("All " + _locationMessageStr.toLowerCase() + " for location " + index + " (of "
                    + _selectedIdentifiers.size() + "): " + identifier.buildStringToDisplayInTree() + "...");
                updateNote("All " + _locationMessageStr.toLowerCase() + " for location " + index + " (of "
                    + _selectedIdentifiers.size() + "): " + identifier.buildStringToDisplayInTree() + "...");

                if(isCanceled())
                {
                    throw new InterruptedException("User cancelled.");
                }

                runStepsForIdentifier(identifier);

                if(isCanceled())
                {
                    throw new InterruptedException("User cancelled.");
                }

                //Increment progress of top level job.
                madeProgress();
            }
        }
        catch(final InterruptedException e)
        {
            LOG.error("Run all failed: " + e.getMessage()
                + "\nSome steps may have been performed for some locations prior to failure.");
            fireProcessJobFailure(new Exception("Run all failed: " + e.getMessage()), true);
            return;

        }
        catch(final Exception e)
        {
            e.printStackTrace();
            LOG.error("Run all failed: " + e.getMessage());
            fireProcessJobFailure(new Exception("Run all failed: " + e.getMessage()), true);
            return;
        }
        finally
        {
            //Perform cleanup for each step, regardless of what happened above.  This will close
            //FTP connections, among possible other activities.
            for(final ParameterEstimatorStepProcessor step: this._steps)
            {
                step.cleanupAfterStep();
            }
        }

        LOG.info("All steps complete for all selected locations.");
        updateNote("All steps complete for all selected locations.");

        endTask();
    }

    @Override
    public void endTask()
    {
        //Close the dialog
        setDone(true);

        //Call the default end task activity (which calls the owner's process success).
        super.endTask();

    }
}
