package ohd.hseb.hefs.pe.gui;

import java.awt.Component;
import java.util.Iterator;
import java.util.List;

import javax.swing.JOptionPane;

import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.core.StepUnit;
import ohd.hseb.hefs.pe.notice.StepUnitsUpdatedNotice;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.log4j.LoggingTools;
import ohd.hseb.hefs.utils.tools.StringTools;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

/**
 * Performs a single step for a list of identifiers. The ParameterEstimatorStepProcessor performStep method is called on
 * location at a time, by default.
 * 
 * @author hank.herr
 */
public class PerformStepJob extends GenericJob
{
    private static final Logger LOG = LogManager.getLogger(PerformStepJob.class);

    private final ParameterEstimatorRunInfo _runInformation;
    private final List<? extends StepUnit> _units;
    private final ParameterEstimatorStepProcessor _step;
    private Component _parentComponent = null;

    public PerformStepJob(final ParameterEstimatorRunInfo runInfo,
                          final ParameterEstimatorStepProcessor step,
                          final List<? extends StepUnit> units)
    {
        _runInformation = runInfo;
        _units = units;
        _step = step;
    }

    public void setParentComponentForMessages(final Component component)
    {
        _parentComponent = component;
    }

    public ParameterEstimatorStepProcessor getStep()
    {
        return _step;
    }

    public List<? extends StepUnit> getChosenUnits()
    {
        return _units;
    }

    @Override
    public void processJob()
    {
        LOG.info(_step.getPerformStepPrefix() + " for " + _units.size() + " locations...");
        setMaximumNumberOfSteps(_units.size());
        updateNote(_step.getPerformStepPrefix() + " for " + _units.size() + " locations...");

        try
        {
            final Iterator<? extends StepUnit> iter = _units.iterator();
            while(iter.hasNext())
            {
                final StepUnit unit = iter.next();
                if(isCanceled())
                {
                    throw new InterruptedException("User cancelled.");
                }

                LOG.info(_step.getPerformStepPrefix() + " for " + unit.getMessageName() + "...");
                updateNote(_step.getPerformStepPrefix() + " for " + unit.getMessageName() + "...");

                try
                {
                    _step.performStep(unit);
                }
                catch(final Exception e)
                {
                    LoggingTools.outputStackTraceAsDebug(LOG, e);

                    //If the user canceled the run or there are no more locations, then we will not 
                    //offer the option of continuing to the next location.  
                    if((!isCanceled()) && (iter.hasNext()))
                    {
                        LOG.error(_step.getStepNameForRunButton() + " step failed for " + unit.getMessageName() + ": "
                            + e.getMessage().replaceAll("\n", " "));
                        LOG.info("If a log file is created, check it for the reason.");
                        LOG.info("(To see stack trace, set the root priority value to be 'DEBUG' in the Log4jConfig.xml file.)");
                        final int option = JOptionPane.showConfirmDialog(_parentComponent,
                                                                         _step.getStepNameForRunButton()
                                                                             + " step failed for "
                                                                             + unit.getMessageName()
                                                                             + ":\n"
                                                                             + StringTools.wordWrap(e.getMessage(), 100)
                                                                             + "\nContinue to next location? (Click No to stop processing.)",
                                                                         "Error Performing Step!",
                                                                         JOptionPane.YES_NO_OPTION,
                                                                         JOptionPane.WARNING_MESSAGE);
                        if(option != JOptionPane.YES_OPTION)
                        {
                            throw new InterruptedException("Step was cancelled likely due to problems encountered.");
                        }
                    }
                    else
                    {
                        throw new InterruptedException(e.getMessage());
                    }
                }
                finally
                {
                    _runInformation.post(new StepUnitsUpdatedNotice(this, _step, _units));
                }

                madeProgress();
            }
        }
        catch(final InterruptedException e)
        {
            LoggingTools.outputStackTraceAsDebug(LOG, e);
            LOG.error(_step.getPerformStepPrefix() + " step failed: " + e.getMessage().replaceAll("\n", " ") //remove new-lines
                + "   The step may have been performed for some locations prior to failure.");
            fireProcessJobFailure(new Exception(_step.getPerformStepPrefix() + " step failed:\n\n" + e.getMessage()
                + "\n\nThe step may have been performed for some locations prior to failure."), true);
            return;

        }
        catch(final Exception e)
        {
            e.printStackTrace();
            LOG.error(_step.getStepNameForRunButton() + " step failed: " + e.getMessage().replaceAll("\n", " "));
            fireProcessJobFailure(new Exception("Error occurred while executing step:\n\n" + e.getMessage()), true);
            return;
        }
        finally
        {
            _step.cleanupAfterStep();
        }

        LOG.info("Step has been performed for all selected locations.");
        updateNote("Step has been performed for all selected locations.");
        endTask();
    }
}
