package ohd.hseb.hefs.pe.model;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;

import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.estimation.options.EstimationControlOptions;
import ohd.hseb.hefs.pe.sources.ForecastSource;
import ohd.hseb.hefs.pe.sources.ForecastSourceTools;
import ohd.hseb.hefs.pe.sources.SourceDataHandler;
import ohd.hseb.hefs.pe.sources.SourceModelParameters;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifierSupplier;
import ohd.hseb.hefs.utils.gui.about.AboutFile;
import ohd.hseb.hefs.utils.gui.about.OHDConfigInfo;
import ohd.hseb.hefs.utils.tools.TarTools;
import ohd.hseb.hefs.utils.tools.TarTools.OHDWrapperTarArchiveInputStream;
import ohd.hseb.hefs.utils.xml.CompositeXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReadable;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLReaderWrapper;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLWritable;
import ohd.hseb.hefs.utils.xml.XMLWriter;
import ohd.hseb.hefs.utils.xml.vars.XMLString;

import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.xml.sax.Attributes;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

/**
 * Handles all parameters for a model execution: algorithm parameters and source-specific parameters. The model to which
 * the parameters apply is passed into the constructor as a ParameterEstimationModel. The algorithm model parameters are
 * pulled from that model.<br>
 * <br>
 * Note that the ordering of the sources, {@link #_orderedForecastSources}, is important in order to keep it in synch
 * with {@link #_orderedSourceModelParameters}, but also in that it reflects priorities used by underlying algorithms.
 * However, the order is no longer tied to the order of files in a parameter tar-ball. Though files generated will
 * follow that order, the contents can be out of order and still be read (though the relative ordering of files
 * associated with a source must be maintained and those files must be consecutive; see
 * {@link #readParametersTarArchive(File)}).<br>
 * <br>
 * For the output parameter tar-balls to include a version, the subclass must include the {@link AboutFile} annotation.
 * See {@link MEFPFullModelParameters} for example.
 * 
 * @author hank.herr
 */
public abstract class FullModelParameters implements XMLWritable, XMLReadable
{
    private static final Logger LOG = LogManager.getLogger(FullModelParameters.class);

    private LocationAndDataTypeIdentifier _identifier;

    /**
     * The model for estimating/using parameters. This should be set whenever a model is executed to estimate these
     * parameters. It should be the model that is doing the estimation.
     */
    private ParameterEstimationModel _model;

    /**
     * The control options used to estimate the parameters.
     */
    private EstimationControlOptions _estimationControlOptions;

    /**
     * The model parameters specific for the algorithm.
     */
    private AlgorithmModelParameters _algorithmModelParameters;

    /**
     * Order is important in this list. It must match the order of sources declared in the main parameters source list.
     * If the {@link SourceModelParameters} framework does not work for the model involved, this list should be kept
     * empty. In such a case, all parameters must be put within the _algorithmModelParameters, a much more general
     * framework.
     */
    private final List<SourceModelParameters> _orderedSourceModelParameters = new ArrayList<SourceModelParameters>();

    /**
     * List of {@link SourceDataHandler}s providing {@link SourceModelParameters}.
     */
    private final List<ForecastSource> _orderedForecastSources;

    /**
     * Call this if you know the identifier for the parameters and the control options.
     */
    public FullModelParameters(final LocationAndDataTypeIdentifier identifier,
                               final EstimationControlOptions estimationControlOptions,
                               final List<? extends ForecastSource> orderedForecastSources)
    {
        _identifier = identifier;
        _estimationControlOptions = estimationControlOptions;
        _model = buildModelInstance();
        _algorithmModelParameters = _model.getAlgorithmModelParameters();
        _orderedForecastSources = Lists.newArrayList(orderedForecastSources);
        initializeSourceModelParameters();
    }

    /**
     * Call this if you know the identifier for the parameters, but not the control options.
     */
    public FullModelParameters(final LocationAndDataTypeIdentifier identifier,
                               final List<? extends ForecastSource> orderedForecastSources)
    {
        _identifier = identifier;
        _orderedForecastSources = Lists.newArrayList(orderedForecastSources);
        _estimationControlOptions = buildEstimationControlOptions();
        _model = buildModelInstance();
        _algorithmModelParameters = _model.getAlgorithmModelParameters();
        initializeSourceModelParameters();
    }

    /**
     * Call this if parameters are to be read in from a file for an unknown location and control options.
     */
    public FullModelParameters(final List<? extends ForecastSource> forecastSourcesOrderedBasedOnParameterFiles)
    {
        _estimationControlOptions = null;
        _algorithmModelParameters = null;
        _model = null;
        _identifier = null;
        _orderedForecastSources = Lists.newArrayList(forecastSourcesOrderedBasedOnParameterFiles);
    }

    /**
     * @throws Exception If parameters cannot be found for any of the required {@link ModelParameterType}s.
     */
    public void validateParameters() throws Exception
    {
        for(final SourceModelParameters parms: _orderedSourceModelParameters)
        {
            for(final ModelParameterType type: _model.getAllParametersRequiredForModel(parms.getForecastSource()))
            {
                if(parms.getParameterValues(type) == null)
                {
                    throw new Exception("Parameters of type " + type.getName() + " were not found.");
                }
            }
            parms.validateParameters();
        }
        getAlgorithmModelParameters().validateParameters();
    }

    /**
     * @param source The source to check
     * @return True if parameters were estimated for the source. False if either source model parameters do not exist
     *         for the source or, if they do exist, then
     *         {@link SourceModelParameters#wereParametersEstimatedForSource()} returns false.
     */
    public boolean wereParametersEstimatedForSource(final ForecastSource source)
    {
        return ((getSourceModelParameters(source) != null) && (getSourceModelParameters(source).wereParametersEstimatedForSource()));
    }

    /**
     * Updates the stored parameter values to reflect those in the provided {@link OneSetParameterValues}. The
     * {@link SourceModelParameters} updated are those that correspond to
     * {@link OneSetParameterValues#getForecastSource()}.
     * 
     * @param values Values for a single day and value of the secondary index.
     */
    public void recordValuesFrom(final OneSetParameterValues values)
    {
        final SourceModelParameters parms = this.getSourceModelParameters(values.getForecastSource());
        parms.recordValuesFrom(values);
    }

    /**
     * Populates the given {@link OneSetParameterValues} with {@link SourceModelParameters} values. The values are
     * extracted based on {@link OneSetParameterValues#getDayOfYearIndex()} and
     * {@link OneSetParameterValues#getSecondaryIndex()}. The source of the numbers is the {@link SourceModelParameters}
     * associated with the {@link SourceDataHandler} returned by {@link OneSetParameterValues#getSourceDataHandler()}.
     * 
     * @param values {@link OneSetParameterValues} instance specifying the day of year and secondary index. The values
     *            put into the provided object are for the nearest day of estimated parameters (not linear
     *            interpolation!).
     */
    public void putValuesInto(final OneSetParameterValues values)
    {
        final SourceModelParameters parms = this.getSourceModelParameters(values.getForecastSource());
        parms.putValuesInto(values, findNearestComputationalDay(values.getDayOfYearIndex()));
    }

    /**
     * Override to return a {@link OneSetParameterValues} containing all of the parameters for the given day of the year
     * and secondary index.
     * 
     * @param source The source for which to acquire parameters.
     * @param dayOfYear The day of the year.
     * @param secondaryIndex The secondary index for which to acquire parameters.
     * @return The parameters necessary to run a model for the given inputs.
     */
    public abstract OneSetParameterValues getSourceEstimatedModelParameterValues(ForecastSource source,
                                                                                 int dayOfYear,
                                                                                 int secondaryIndex);

    public ParameterEstimationModel getModel()
    {
        return _model;
    }

    public void setModel(final ParameterEstimationModel model)
    {
        _model = model;
    }

    public AlgorithmModelParameters getAlgorithmModelParameters()
    {
        return _algorithmModelParameters;
    }

    public EstimationControlOptions getEstimationControlOptions()
    {
        return _estimationControlOptions;
    }

    public LocationAndDataTypeIdentifier getIdentifier()
    {
        return _identifier;
    }

    /**
     * Sets the identifier here and passes it down into the sources. It is not recommended that this be called lightly.
     * It is only used in MEFPPE for testing in order to switch identifiers in parameter files.
     * 
     * @param identifier The new identifier.
     */
    public void setIdentifier(final LocationAndDataTypeIdentifier identifier)
    {
        _identifier = identifier;
        for(final SourceModelParameters sourceParms: getOrderedSourceModelParameters())
        {
            sourceParms.setIdentifier(identifier);
        }
    }

    public List<SourceModelParameters> getOrderedSourceModelParameters()
    {
        return this._orderedSourceModelParameters;
    }

    /**
     * Call to set the {@link #_orderedForecastSources}. Useful if you need to reset the sources within a set of
     * parameters to a different list.
     */
    protected void setOrderedForecastSources(final Collection<? extends ForecastSource> sources)
    {
        _orderedForecastSources.clear();
        _orderedForecastSources.addAll(sources);
    }

    public List<? extends ForecastSource> getOrderedForecastSources()
    {
        return _orderedForecastSources;
    }

    public SourceModelParameters getSourceModelParameters(final int sourceIndex)
    {
        return this._orderedSourceModelParameters.get(sourceIndex);
    }

    public SourceModelParameters getSourceModelParameters(final ForecastSource source)
    {
        if(_orderedForecastSources.contains(source))
        {
            return _orderedSourceModelParameters.get(_orderedForecastSources.indexOf(source));
        }
        return null;
    }

    public int getSourceCount()
    {
        return this._orderedSourceModelParameters.size();
    }

    /**
     * @param operationalRun True for operational run, false for hindcasting run or other type of run (e.g.e, parameter
     *            estimation).
     * @param source The source to check for need.
     * @return True if, given that this is or is not an operational run, parameters must be read in for the source.
     */
    protected abstract boolean needParametersForSource(boolean operationalRun, ForecastSource source);

    /**
     * @return Return {@link ParameterEstimationModel} based on the {@link #_identifier}.
     */
    protected abstract ParameterEstimationModel buildModelInstance();

    /**
     * @return Return {@link EstimationControlOptions} based on the {@link #_identifier}.
     */
    protected abstract EstimationControlOptions buildEstimationControlOptions();

    /**
     * @return The days of the year for which to estimate parameters or for which parameters have been estimated. The
     *         returned numbers correspond to the first index in the {@link OneTypeParameterValues#getValue(int, int)}
     *         method and lists the possible values of the first index for which parameters are directly computed (other
     *         values are interpolated somehow).
     */
    protected abstract List<Integer> generateDaysOfTheYearForWhichToEstimateParameters();

    /**
     * The nearest day for which parameters were estimated to the desired day.
     * 
     * @param desiredDay
     * @return
     */
    public abstract int findNearestComputationalDay(final int desiredDay);

    /**
     * Current interpolation scheme is nearest neighbor, so this algorithm determines the closes neighbor (1 day only is
     * returned).
     * 
     * @param algoParameters The algorithm parameters for the model to run.
     * @param daysForWhichToAcquireParameters Days of the year for which parameters are desired.
     * @return The days of the year for which to read parameters in order to ensure that all days of the year for which
     *         to acquire parameters can be interpolated.
     */
    public SortedSet<Integer> determineDaysForWhichToReadParameters(final List<Integer> daysForWhichToAcquireParameters)
    {
//Commented out code below should be uncommented to use linear interpolation!
        final TreeSet<Integer> results = Sets.newTreeSet();
//        final List<Integer> fullListOfDays = algoParameters.generateDaysOfTheYearForWhichToEstimateParameters();
        for(final Integer desiredDay: daysForWhichToAcquireParameters)
        {
            results.add(findNearestComputationalDay(desiredDay));

//            final Integer[] boundingDays = NumberTools.determineBoundingIntegers(fullListOfDays, desiredDay);
//            if(boundingDays[1] == null)
//            {
//                boundingDays[1] = 1;
//            }
//            for(final Integer day: boundingDays)
//            {
//                results.add(day);
//            }
        }
        return results;
    }

    /**
     * This must only be called if parameter estimation is to be done for only one source. It will update the estimation
     * options stored with these parameters (acquired via {@link #getEstimationControlOptions()} to incorporate those
     * specified in for the source in the provided run time options.<br>
     * <br>
     * Note that, in some cases, its easier to reverse the logic, starting with the run time options and overwriting
     * them so that they match the parameter options, except for those specified for the desired source. See
     * {@link MEFPFullModelParameters} for an example of this.
     * 
     * @param currentRunTimeOptions Current run-time estimation options selected by a user and acquired via a
     *            {@link ParameterEstimatorRunInfo} object.
     * @param estimatedSource The forecast source for which parameter estimation to be performed and for which its
     *            options must be pulled from the run time options and incorporated herein.
     */
    public abstract void prepareEstimationOptionsForComputationOfOneSource(final EstimationControlOptions currentRunTimeOptions,
                                                                           final ForecastSource estimatedSource);

    /**
     * Must be overridden in order to initialize {@link #_orderedSourceModelParameters} for reading after all other
     * parameters are read. It should be possible to prepare them for reading based on the contents of
     * {@link #getIdentifier()}, {@link #getEstimationControlOptions()}, and {@link #getAlgorithmModelParameters()}.
     */
    protected abstract void prepareSourceModelParameters();

    /**
     * Constructs {@link SourceModelParameters} and populates {@link #_orderedSourceModelParameters} based on the
     * {@link SourceDataHandler#getSourceModelParameters(LocationAndDataTypeIdentifier, AlgorithmModelParameters)}. This
     * should be called at the top of {@link #prepareSourceModelParameters()}.
     */
    protected void initializeSourceModelParameters()
    {
        _orderedSourceModelParameters.clear();
        for(final ForecastSource source: _orderedForecastSources)
        {
            final SourceModelParameters srcParms = source.getSourceModelParameters(_identifier,
                                                                                   _algorithmModelParameters);
            srcParms.setAlgorithmModelParameters(_algorithmModelParameters);
            _orderedSourceModelParameters.add(srcParms);
        }
    }

    /**
     * Pulled this out in order to unclutter the {@link #writeParametersTarArchive(File, boolean, boolean)} method.
     * 
     * @return A {@link CompositeXMLWriter} for writing out the main parameters file within a parameter tarball if a
     *         parameter tar ball is used.
     */
    private CompositeXMLWriter getMainParametersWriter()
    {
        final CompositeXMLWriter writer = new CompositeXMLWriter("mefpModelParameters");
        addVersionXMLString(writer);
        writer.addComponent(_identifier);
        writer.addComponent(ForecastSourceTools.createSourcesWriter(null, _orderedForecastSources, false));
        writer.addComponent(_estimationControlOptions);
        writer.addComponent(new XMLString("modelClass", this._model.getClass().getName()));
        writer.addComponent(_algorithmModelParameters);
        return writer;
    }

    /**
     * Add an {@link XMLString} to the writer to specify the version based on an {@link AboutFile} annotation on this
     * subclass.
     */
    private void addVersionXMLString(final CompositeXMLWriter writer)
    {
        //Output the version.  
        final AboutFile aboutAnnotation = getClass().getAnnotation(AboutFile.class);
        if(aboutAnnotation == null)
        {
            LOG.warn("No version information availble; Class: " + getClass().getName()
                + " does not have an @AboutFile annotation value");
        }
        else
        {
            final OHDConfigInfo aboutInfo = OHDConfigInfo.loadOHDConfigInfo(aboutAnnotation.value());
            if(aboutInfo != null)
            {
                writer.addComponent(new XMLString("version", aboutInfo.getVersion()));
            }
        }
    }

    /**
     * By default, this method checks to see if 'SourceModelParameters' is in the file name. If so, it is assumed to be
     * a source model parameters file. If a different mechanism should be used, override this method as needed.
     * 
     * @return True if the provided entry file name is a source model parameter entry file. This presumes that all
     *         source model entry files are named in a manner that can be identified by a subclass and cannot be
     *         confused with the additional files processed via
     *         {@link #readAdditionalInputFromTarArchive(File, TarArchiveInputStream, boolean)}. This, then, becomes a
     *         requirement of the tarball reading process.
     */
    protected boolean isSourceModelParameterEntryFile(final String entryName)
    {
        if(entryName == null)
        {
            return false;
        }
        return entryName.contains("SourceModelParameters");
    }

    /**
     * Writes parameters as individual files: one file each per source and one file for the rest.
     * 
     * @param tarOutputFileName The output file name of the tar file. It must end with .tar.gz or .tgz, as the file will
     *            be a gzipped tarball.
     * @param performanceMode True if the file to be read was written for performance in terms of size and speed; false
     *            if it was written for testing, meaning all files are human readable XML. Note that fast infoset will
     *            not be used for I/O because it appears to be slower than XML when reading a gzipped tarball.
     * @throws Exception
     */
    public void writeParametersTarArchive(final File tarOutputFile, final boolean performanceMode) throws Exception
    {
        writeParametersTarArchive(tarOutputFile, performanceMode, false);
    }

    /**
     * Writes parameters as individual files: one file each per source and one file for the rest. <br>
     * <br>
     * Note that the source written to the estimation options will NOT include reforecast preparation steps; only a
     * basic list of the sources. This is because when reading/writing a parameter file, those steps need not be known.
     * Only the presence of the source needs to be known.
     * 
     * @param tarOutputFileName The output file name of the tar file. It must end with .tar.gz or .tgz, as the file will
     *            be a gzipped tarball.
     * @param performanceMode True if the file to be read was written for performance in terms of size and speed; false
     *            if it was written for testing, meaning all files are human readable XML. Note that fast infoset will
     *            not be used for I/O because it appears to be slower than XML when reading a gzipped tarball.
     * @param writeOnlyNecessaryData If true, then the parameters that will be output are only those that are actually
     *            computed. If this is used for binary to XML file conversion, then the binary file includes parameters
     *            for all days. To limit the output to only computed days relying on interpolation for the rest, pass in
     *            true.
     * @throws Exception
     */
    public void writeParametersTarArchive(final File tarOutputFile,
                                          final boolean performanceMode,
                                          final boolean writeOnlyNecessaryData) throws Exception
    {
        if((!tarOutputFile.getAbsolutePath().endsWith(".tar.gz"))
            && (!tarOutputFile.getAbsolutePath().endsWith(".tgz")))
        {
            throw new Exception("Output file name does not end with .tar.gz or .tgz: " + tarOutputFile.getName());
        }
        if(tarOutputFile.exists() && !tarOutputFile.canWrite())
        {
            throw new Exception("The parameter file to write, " + tarOutputFile.getAbsolutePath()
                + ", exists and cannot be written to due to permissions.");
        }

        //Determine the days of the year for which to read parameters, if restricted.  This MUST be done after the 
        //algorithm and control parameters are read via the main parameters file!
        Collection<Integer> daysOfTheYearForWhichToReadParameters = Lists.newArrayList();
        if(writeOnlyNecessaryData)
        {
            daysOfTheYearForWhichToReadParameters = generateDaysOfTheYearForWhichToEstimateParameters();
        }

        //Create the tar archive stream.
        final TarArchiveOutputStream tarStream = TarTools.openTarArchiveForWriting(tarOutputFile, true);

        //Write main parm file.
        try
        {
            //Writer must be defined explicitly by called getMainParametersWriter() because the default getWriter() method
            //will create a writer that includes parameter values when the main parametes, which is not what we want here.
            final CompositeXMLWriter writer = getMainParametersWriter();
            TarTools.addXMLEntryToTarArchive(tarStream,
                                             "mainParameters." + XMLTools.getXMLFileExtension(false),
                                             writer,
                                             false);

            //Sources -- handle the writeONlyNecessaryData flag below, using the daysOfTheYearForWhichToReadParameters list computed above.
            for(final SourceModelParameters srcParms: _orderedSourceModelParameters)
            {
                srcParms.clearDaysOfYearToReadWriteParameters(); //just being safe, though it shouldn't be needed!
                if(writeOnlyNecessaryData)
                {
                    srcParms.addAllDaysOfYearToReadWrite(daysOfTheYearForWhichToReadParameters);
                }
                srcParms.writeToTarArchive(tarStream, performanceMode);
                if(writeOnlyNecessaryData)
                {
                    srcParms.clearDaysOfYearToReadWriteParameters();
                }
            }

            //Write additional subclass specific info.
            writeAdditionalOutputToTarArchive(tarStream);
        }
        finally
        {
            tarStream.flush();
            tarStream.close();

            //Make the file read only.
            tarOutputFile.setReadOnly();
        }

    }

    /**
     * Override to write out additional information to the XML tar file that it subclass specific. Note that this must
     * be kept in synch with reading. By default, this does nothing.<br>
     * <br>
     * Any file added by this method must be named such that the method {@link #isSourceModelParameterEntryFile(String)}
     * returns false. The read method first reads the main parameters, then the source parameters. It stops with the
     * source parameters once {@link #isSourceModelParameterEntryFile(String)} returns false.
     * 
     * @param stream Stream to which to write.
     * @throws Exception
     */
    protected void writeAdditionalOutputToTarArchive(final TarArchiveOutputStream stream) throws Exception
    {
    }

    /**
     * Calls {@link #readParametersTarArchive(File, List)} passing in any empty list.
     */
    public void readParametersTarArchive(final File tarInputFile) throws Exception
    {
        readParametersTarArchive(tarInputFile, new ArrayList<Integer>());
    }

    /**
     * Calls {@link #readParametersTarArchive(File, List, boolean)} passing in false for mainParametersOnly flag.
     */
    public void readParametersTarArchive(final File tarInputFile, final List<Integer> daysForWhichToAcquireParameters) throws Exception
    {
        readParametersTarArchive(tarInputFile, daysForWhichToAcquireParameters, false, false);
    }

    /**
     * Note that the nature of the commons tar stream code used forces files to be read from the tar ball in the exact
     * same order as they were written to the tar ball. Check against
     * {@link #writeParametersTarArchive(File, boolean, boolean)} to make sure the order is maintained if overriding
     * this method or changing it.<br>
     * <br>
     * Note that the source written to the estimation options will NOT include reforecast preparation steps; only a
     * basic list of the sources. This is because when reading/writing a parameter file, those steps need not be known.
     * Only the presence of the source needs to be known.
     * 
     * @param tarOutputFileName The input file name of the tar file. It must end with .tar.gz or .tgz, as the file will
     *            be a gzipped tarball.
     * @param daysForWhichToAcquireParameters If the days for which to read parameters are to be restricted, pass in an
     *            array here with a list of the days for which to read parameters. If the list is empty, all days are
     *            read. null is not currently allowed.
     * @param mainParametersOnly If true, then only the mainParameters.xml is read, possibly in order to acquire only
     *            the estimation control parameters or algorithm parameters without the rest of it.
     * @param operationalRead If true, then the parameters are being read for the purposes of an operational run. If
     *            false, parameters are being read during parameter estimation and the read should be a full one.
     * @throws Exception
     */
    public void readParametersTarArchive(final File tarInputFile,
                                         final List<Integer> daysForWhichToAcquireParameters,
                                         final boolean mainParametersOnly,
                                         final boolean operationalRead) throws Exception
    {
        if((!tarInputFile.getAbsolutePath().endsWith(".tar.gz")) && (!tarInputFile.getAbsolutePath().endsWith(".tgz")))
        {
            throw new Exception("Output file name does not end with .tar.gz or .tgz: " + tarInputFile.getName());
        }

        //Use openTar... so that the XML parser does not close the stream.
        final OHDWrapperTarArchiveInputStream tarStream = (OHDWrapperTarArchiveInputStream)TarTools.openTarArchiveForReading(tarInputFile,
                                                                                                                             true,
                                                                                                                             true);

        try
        {
            //Read main parameters first.
            LOG.debug("Reading main parameters from " + tarInputFile.getAbsolutePath() + "...");
            TarTools.readXMLFromTarStream(tarStream,
                                          true,
                                          "mainParameters." + XMLTools.getXMLFileExtension(false),
                                          getReader(),
                                          false);
            LOG.debug("Done reading main parameters from " + tarInputFile.getAbsolutePath() + ".");
            if(mainParametersOnly)
            {
                return;
            }

            //Go to the next entry as preparation for the source model parameters.
            TarTools.gotoNextEntry(tarStream);

            //Determine the days of the year for which to read parameters, if restricted.  This MUST be done after the 
            //algorithm and control parameters are read via the main parameters file!
            Collection<Integer> daysOfTheYearForWhichToReadParameters = daysForWhichToAcquireParameters;
            if(!daysForWhichToAcquireParameters.isEmpty())
            {
                daysOfTheYearForWhichToReadParameters = determineDaysForWhichToReadParameters(daysForWhichToAcquireParameters);
                LOG.debug("Given desired day(s) " + daysForWhichToAcquireParameters.toString()
                    + ", parameters will be loaded for day(s) " + daysOfTheYearForWhichToReadParameters.toString()
                    + ".");
            }

            //Prep source model parms for reading and set the days of the year appropriately for the source model parameters.
            prepareSourceModelParameters(); //Sets number of forecast days to be 0.
            for(int i = 0; i < this._orderedSourceModelParameters.size(); i++)
            {
                _orderedSourceModelParameters.get(i).clearDaysOfYearToReadWriteParameters(); //just being safe, though it shouldn't be needed!
                _orderedSourceModelParameters.get(i).addAllDaysOfYearToReadWrite(daysOfTheYearForWhichToReadParameters);
            }

            //We don't care about the order of the source model parameter files, except that they follow the mainParameters.xml file.
            //This loop will identify which source is being read and read that source.  As soon as an entry file is NOT found, it is assumed
            //that we are at the end of the source model parameters part of the parameter tgz file.  Hence it moves on to reading the 
            //additional stuff (see below).
            String currentEntryName = TarTools.getCurrentEntryName(tarStream);
            while(isSourceModelParameterEntryFile(currentEntryName))
            {
                //NOTE: It is NOT an error if the current entry is not processable by any source.  That may mean that the source is no longer
                //usable, so just ignore it.  In general, such files are ignored until the next file is found that can be processed or 
                //is not a source model parameter entry file.
                int index = 0;
                for(index = 0; index < _orderedSourceModelParameters.size(); index++)
                {
                    final SourceModelParameters srcParms = _orderedSourceModelParameters.get(index);
                    if(srcParms.buildFirstSourceParametersFileEntryName().equals(currentEntryName))
                    {
                        LOG.debug("The current tar entry file " + currentEntryName
                            + " matches that expected for source " + srcParms.getForecastSource().getName()
                            + "; reading source parameters...");
                        srcParms.readFromTarArchive(tarStream,
                                                    operationalRead,
                                                    !needParametersForSource(operationalRead,
                                                                             srcParms.getForecastSource()));
                        LOG.debug("Done reading tarred parameter files for source "
                            + srcParms.getForecastSource().getName() + ".");
                        break;
                    }
                }

                //Nothing was found... skip the file.
                if(index == _orderedSourceModelParameters.size())
                {
                    TarTools.gotoNextEntry(tarStream);
                }

                currentEntryName = TarTools.getCurrentEntryName(tarStream);
            }

            //Read additional stuff that is subclass specific.
            LOG.debug("Current entry file " + currentEntryName
                + " was not found for a source. Continuing with application-specific additional tar-file reading.");
            readAdditionalInputFromTarArchive(tarInputFile, tarStream, operationalRead);
        }
        //Close the tar stream now!
        finally
        {
            tarStream.reallyClose();
        }
    }

    /**
     * Override to read in additional information from the tar file that is subclass specific. By default, this does
     * nothing. Note that the nature of the commons tar stream code used forces files to be read from the tar ball in
     * the exact same order as they were written to the tar ball. Check against
     * {@link #writeAdditionalOutputToTarArchive(TarArchiveOutputStream)} to make sure the order is maintained if
     * overriding this method.
     * 
     * @param tarInputFile The tar input file to read; note that this is a file, not a stream.
     * @param operationalRead If true, then the parameters are being read for the purposes of an operational run. If
     *            false, parameters are being read during parameter estimation and the read should be a full one.
     * @throws Exception
     */
    protected void readAdditionalInputFromTarArchive(final File tarInputFile,
                                                     final TarArchiveInputStream tarStream,
                                                     final boolean operationalRead) throws Exception
    {
    }

    /**
     * For writing parameters files when all of the parameters are in a single XML file. Hence, this writer includes
     * {@link #_orderedSourceModelParameters}. However, this is not true for MEFP (which uses a tarball) and XML is not
     * used for EnsPost. Hence, this is currently never called.
     */
    @Override
    public XMLWriter getWriter()
    {
        final CompositeXMLWriter writer = new CompositeXMLWriter("mefpModelParameters");
        writer.addComponent(_identifier);
        writer.addComponent(_estimationControlOptions);
        writer.addComponent(new XMLString("modelClass", this._model.getClass().getName()));
        writer.addComponent(_algorithmModelParameters);
        for(final SourceModelParameters srcParms: _orderedSourceModelParameters)
        {
            writer.addComponent(srcParms);
        }
        return writer;
    }

    /**
     * For reading both the main parameters file in a tarball as well as an overall XML file that contains estimated
     * parameter values as well.
     */
    @Override
    public XMLReader getReader()
    {
        return new XMLReader()
        {
            /**
             * A reader that wraps the _algorithmModelParameters reader and attaches a finalize that initializes the
             * source model parameters for reading. This attribute is initialized after the location is read.
             */
            private XMLReader _algorithmReaderWrapper = null;

            /**
             * Reads the location. When finalizing, it sets the _model and builds the estimation control options.
             */
            private final XMLReader _locationReader = new LocationAndDataTypeIdentifierSupplier()
            {
                @Override
                public void finalizeReading() throws XMLReaderException
                {
                    _identifier = this.get();

                    //Initialize the estimation control options, models, and algorithm parameters based on DEFAULT sources,
                    //which are those already set.  These may be reconstructed if the sources are specified int he parameter file
                    //(see below).
                    _estimationControlOptions = buildEstimationControlOptions();
                    _model = buildModelInstance();
                    _algorithmModelParameters = _model.getAlgorithmModelParameters();

                    _algorithmReaderWrapper = new XMLReaderWrapper(_algorithmModelParameters.getReader())
                    {
                        @Override
                        public void finalizeReading() throws XMLReaderException
                        {
                            //After the algorithm model parameters are read, the source model parameters can be constructed.
                            super.finalizeReading();
                            FullModelParameters.this.prepareSourceModelParameters();
                        }
                    };
                }
            };

            /**
             * Finalize within the returned reader should setup the sources list handed to it. For whatever reason, the
             * list passed into createSourcesReader is NOT the list being passed to it; its translated somehow. As
             * such,this finalize method must update the forecast source list manually.
             */
            private final XMLReaderWrapper _sourcesReader = new XMLReaderWrapper(ForecastSourceTools.createSourcesReader(null,
                                                                                                                         _orderedForecastSources))
            {
                @Override
                public void finalizeReading() throws XMLReaderException
                {
                    super.finalizeReading();

                    //Reset the estimation control options, model, and algorithm parameters based on the sources specified in the parameter file!
                    _estimationControlOptions = buildEstimationControlOptions();
                    _model = buildModelInstance();
                    _algorithmModelParameters = _model.getAlgorithmModelParameters();

                    _algorithmReaderWrapper = new XMLReaderWrapper(_algorithmModelParameters.getReader())
                    {
                        @Override
                        public void finalizeReading() throws XMLReaderException
                        {
                            //After the algorithm model parameters are read, the source model parameters can be constructed.
                            super.finalizeReading();
                            FullModelParameters.this.prepareSourceModelParameters();
                        }
                    };
                }
            };

            /**
             * Reads the name of the model. The finalize just checks if the name is valid.
             */
            private final XMLReader _modelNameReader = new XMLString("modelClass")
            {
                @Override
                public void finalizeReading() throws XMLReaderException
                {
                    if(!FullModelParameters.this._model.getClass().getName().equals(get()))
                    {
                        throw new XMLReaderException("Value of modelClass, " + get() + ", is not as expected, "
                            + _model.getClass().getName() + ".");
                    }

                }
            };

            /**
             * Indicates the index of the source model parameters to read next.
             */
            private int _sourceToReadIndex = 0;

            @Override
            public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
            {
            }

            @Override
            public XMLReader readInPropertyFromXMLElement(final String elementName, final Attributes attr) throws XMLReaderException
            {
                if(elementName.equals(getXMLTagName()))
                {
                    return null;
                }
                else if(elementName.equals("version"))
                {
                    return new XMLString("version");
                }
                else if(elementName.equals(_locationReader.getXMLTagName()))
                {
                    return _locationReader;
                }
                else if(elementName.equals(_sourcesReader.getXMLTagName()))
                {
                    return _sourcesReader;
                }
                else if((_estimationControlOptions != null)
                    && (elementName.equals(_estimationControlOptions.getXMLTagName())))
                {
                    return _estimationControlOptions.getReader();
                }
                else if(elementName.equals(_modelNameReader.getXMLTagName()))
                {
                    return _modelNameReader;
                }
                else if(elementName.equals(_algorithmModelParameters.getXMLTagName()))
                {
                    return _algorithmReaderWrapper;
                }

                //This reader assumes that the parameters are output within the same file as the main parameters (above).  That is not the case for 
                //MEFP, so this loop is never entered into for MEFP.  It would be the case for EnsPost if they used this parameter writer, which it
                //does not use.
                else if((_sourceToReadIndex < _orderedSourceModelParameters.size())
                    && (elementName.equals(_orderedSourceModelParameters.get(_sourceToReadIndex).getXMLTagName())))
                {
                    _sourceToReadIndex++;
                    return _orderedSourceModelParameters.get(_sourceToReadIndex - 1).getReader();
                }
                throw new XMLReaderException("Unrecognized element tag name '" + elementName + "'.");
            }

            @Override
            public void finalizeReading() throws XMLReaderException
            {
                _estimationControlOptions.populateAlgorithmModelParameters(getAlgorithmModelParameters());
            }

            @Override
            public void validate() throws XMLReaderException
            {
            }

            @Override
            public XMLReader getReader()
            {
                return this;
            }

            @Override
            public String getXMLTagName()
            {
                return "mefpModelParameters";
            }
        };
    }

}
