package ohd.hseb.hefs.pe.model;

import java.util.HashMap;

import ohd.hseb.charter.parameters.ChartDrawingParameters;
import ohd.hseb.hefs.utils.tools.ParameterId;

import com.google.common.base.Objects;

/**
 * This abstract class sets the hashCode() and equals() methods so that two instances of the same
 * {@link ModelParameterType} will be considered equal for determining the key of a {@link HashMap}. The hashCode method
 * returns a code derived from a String specifying the class name, while equals() returns true if the Object given
 * matches the class of this.<br>
 * <br>
 * All model parameter types must implement this. The method {@link #computeValue(double[], double[])} assumes that the
 * parameters of a model can be computed independently of each other from a set of forecasts and observations, which may
 * not always be true (hence this method need not always be overridden). The raw parameter values are assumed to be
 * numerical, hence {@link #getValueTypeClass()} returns a class that extends {@link Number}.<br>
 * <br>
 * Be sure to override the {@link #getUnit()} method if the parameter is unitless or cannot be determined based on the
 * {@link ParameterId} corresponding to {@link #getParameterId()}.
 * 
 * @author hank.herr
 */
public abstract class ModelParameterType
{

    private String _parameterId = "UNSPECIFIED";

    private int _parameterIndex = 0;

    private Object _optionalExtraIdentifier = null;

    /**
     * This assumes that this parameter can be computed independently of the other parameters.
     * 
     * @param forecasts Array of forecast values.
     * @param observations Array of observed values corresponding to the forecasts.
     * @return The parameter value.
     */
    public abstract double computeValue(double[] forecasts, double[] observations);

    /**
     * Override if you want to validate the parameters. By default, it does nothing.
     * 
     * @param values The values to validate.
     * @throws Exception If validation fails with a message indicating the reason.
     */
    public void validateParameters(final OneTypeParameterValues values) throws Exception
    {
    }

    /**
     * @return A legible name for the model parameter type.
     */
    public abstract String getName();

    /**
     * @return The unit of the parameter, displayed in diagnostics or wherever else appropriate. By default, this
     *         returns the default unit specified for the {@link ParameterId} corresponding to {@link #getParameterId()}
     *         . Null is a valid return and implies that the parameter is unitless.
     */
    public String getUnit()
    {
        return ParameterId.valueOf(getParameterId())._defaultUnit.getName();
    }

    /**
     * @return The class of the object used to store/retrieve values. Use Float.class for float, Integer.class for int.
     *         It must always extend {@link Number}.
     */
    public abstract Class<? extends Number> getValueTypeClass();

    public void setParameterId(final String parameterId)
    {
        _parameterId = parameterId;
    }

    public String getParameterId()
    {
        return _parameterId;
    }

    public void setParameterIndex(final int index)
    {
        this._parameterIndex = index;
    }

    public int getParameterIndex()
    {
        return this._parameterIndex;
    }

    /**
     * @param identifierObj An identifying object whose toString() function is defined.
     */
    public void setOptionalExtraIdentifier(final Object identifierObj)
    {
        this._optionalExtraIdentifier = identifierObj;
    }

    public Object getOptionalExtraIdentifier()
    {
        return this._optionalExtraIdentifier;
    }

    /**
     * Override if you need to set model parameter type specific chart drawing parameters, overriding the general
     * display created by using the product parameter xml file combined with the DefaultModelParameterDiagnosticDisplay
     * class. This method will be called once per ModelParameterType, but may be called many times total: once per
     * parameter type displayed on a chart.
     * 
     * @param parameters The ChartDrawingParameters to modify.
     */
    public void modifyDiagnosticDisplayParameters(final ChartDrawingParameters parameters)
    {
    }

    /**
     * Called by some tools that display parameter values on a plot. Override as needed.
     * 
     * @return A value that provides the smallest value to include on an axis displaying this parameter. If
     *         {@link Double#NaN} is returned, then the axis minimum is not specified for this parameter. By default,
     *         this returns {@link Double#NaN}.
     */
    public double getMinimumDisplayValue()
    {
        return Double.NaN;
    }

    /**
     * Called by some tools that display parameter values on a plot. Override as needed.
     * 
     * @return A value that provides the largest value to include on an axis displaying this parameter. If
     *         {@link Double#NaN} is returned, then the axis maximum is not specified for this parameter. By default,
     *         this returns {@link Double#NaN}.
     */
    public double getMaximumDisplayValue()
    {
        return Double.NaN;
    }

    /**
     * Called by some tools that display parameter values on a plot. Override as needed.
     * 
     * @return True if the data display limits should always be square about 0. For example, if displaying a value that
     *         represents an additive difference or bias, then it may be desirable to display that such that the min and
     *         max values for an axis are centered on zero.
     */
    public boolean getSquarePlottedLimits()
    {
        return false;
    }

    @Override
    public int hashCode()
    {
        if(_optionalExtraIdentifier == null)
        {
            return new String(this.getClass().toString() + " @@ " + _parameterId + " @@ " + _parameterIndex).hashCode();
        }
        else
        {
            return new String(this.getClass().toString() + " @@ " + _parameterId + " @@ " + _parameterIndex + " @@ "
                + _optionalExtraIdentifier.toString()).hashCode();
        }
    }

    @Override
    public boolean equals(final Object o)
    {
        final ModelParameterType other = (ModelParameterType)o;
        return (o.getClass().equals(this.getClass()) && (_parameterId.equals(other.getParameterId()))
            && (_parameterIndex == other.getParameterIndex()) && Objects.equal(_optionalExtraIdentifier,
                                                                               other.getOptionalExtraIdentifier()));
    }
}
