package ohd.hseb.hefs.pe.model;

import java.util.Calendar;
import java.util.HashMap;
import java.util.List;

import ohd.hseb.hefs.pe.sources.ForecastSource;
import ohd.hseb.hefs.pe.sources.SourceDataHandler;

/**
 * Stores a single set of parameter values associated with a model for a particular {@link SourceDataHandler}. This
 * provides another view to {@link OneTypeParameterValues#getValue(int, int)} in that it will store all of the
 * parameters for a day of the year and secondary index. All parameters are stored as doubles for efficiency (trying to
 * avoid {@link Number} classes).<br>
 * <br>
 * IMPORTANT: Any method that takes a day of the year must start counting at 1; that is how {@link Calendar#DAY_OF_YEAR}
 * works as well.<br>
 * <br>
 * ALSO IMPORTANT: This set will always include all {@link ModelParameterType} instances specified by
 * {@link ParameterEstimationModel#getAllParametersRequiredForModel(ForecastSource)}, even if the source only requires
 * those specified in {@link ParameterEstimationModel#getObservationParametersRequiredForModel(ForecastSource)}. It is
 * up to outside methods to deal with this.
 * 
 * @author hank.herr
 */
public abstract class OneSetParameterValues
{

    private final HashMap<ModelParameterType, Integer> _typeToIndexMap = new HashMap<ModelParameterType, Integer>();
    private final List<ModelParameterType> _parameterTypes;
    private double[] _values;

    /**
     * The first index used for placing data in {@link OneTypeParameterValues} mapping. See
     * {@link OneTypeParameterValues#getValue(int, int)} and {@link OneTypeParameterValues#setValue(int, int, double)}
     * methods.
     */
    private final int _dayOfYearIndex;

    /**
     * The secondary index used for placing data in {@link OneTypeParameterValues} mapping. See
     * {@link OneTypeParameterValues#getValue(int, int)} and {@link OneTypeParameterValues#setValue(int, int, double)}
     * methods.
     */
    private final int _secondaryIndex;

    private final ForecastSource _forecastSource;

    /**
     * This is private, because I think only the other constructor should be needed.
     * 
     * @param dayOfYear Works like {@link Calendar#DAY_OF_YEAR}, so it is 1...365 (no 366 accunted for).
     * @param secondaryIndex
     * @param types List of types; the order matters and must match how other see the list of types.
     * @param handler The {@link SourceDataHandler} associated with the parameters.
     */
    private OneSetParameterValues(final int dayOfYear,
                                  final int secondaryIndex,
                                  final List<ModelParameterType> types,
                                  final ForecastSource source)
    {
        _parameterTypes = types;
        _values = new double[_parameterTypes.size()];
        _forecastSource = source;

        //Initializes so that the map is ordered!
        int index = 0;
        for(final ModelParameterType type: _parameterTypes)
        {
            _typeToIndexMap.put(type, index);
            index++;
        }

        _dayOfYearIndex = dayOfYear;

        _secondaryIndex = secondaryIndex;
    }

    /**
     * Constructs this using the return of
     * {@link ParameterEstimationModel#getAllParametersRequiredForModel(ForecastSource)}, so that it is meant to store
     * all parameters, not just observed parameters.
     * 
     * @param dayOfYear Works like {@link Calendar#DAY_OF_YEAR}, so it is 1...365 (no 366 accunted for).
     * @param secondaryIndex
     * @param model Provides a list of types via {@link ParameterEstimationModel#getAllParametersRequiredForModel()}.
     */
    public OneSetParameterValues(final int dayOfYear,
                                 final int secondaryIndex,
                                 final ParameterEstimationModel model,
                                 final ForecastSource source)
    {
        this(dayOfYear, secondaryIndex, model.getAllParametersRequiredForModel(source), source);
    }

    /**
     * Sets this up based on baseSet, but does NOT copy the values! It only sets up the meta-information.
     * 
     * @param baseSet
     */
    public OneSetParameterValues(final OneSetParameterValues baseSet)
    {
        this(baseSet.getDayOfYearIndex(),
             baseSet.getSecondaryIndex(),
             baseSet.getParameterTypes(),
             baseSet.getForecastSource());
    }

    public int getDayOfYearIndex()
    {
        return _dayOfYearIndex;
    }

    public int getSecondaryIndex()
    {
        return _secondaryIndex;
    }

    public List<ModelParameterType> getParameterTypes()
    {
        return _parameterTypes;
    }

    public int getTypeCount()
    {
        return _parameterTypes.size();
    }

    public ForecastSource getForecastSource()
    {
        return _forecastSource;
    }

    public void setValue(final ModelParameterType type, final double value)
    {
        _values[_typeToIndexMap.get(type)] = value;
    }

    public double getValue(final ModelParameterType type)
    {
        return _values[_typeToIndexMap.get(type)];
    }

    public void setValue(final int typeIndex, final double value)
    {
        _values[typeIndex] = value;
    }

    public double getValue(final int typeIndex)
    {
        if(typeIndex >= _values.length)
        {
            return Double.NaN;
        }
        return _values[typeIndex];
    }

    public double[] getValues()
    {
        return _values;
    }

    public void setValues(final double[] values)
    {
        _values = values;
    }
}
