package ohd.hseb.hefs.pe.model;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import ohd.hseb.hefs.utils.xml.XMLReadable;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLWritable;
import ohd.hseb.hefs.utils.xml.XMLWriter;

/**
 * Subclasses are meant to record all of the parameter values associated with a single parameter of the model. This
 * should be the values for all seasons or days of the year, at a minimum. It also allows for a secondary indexing,
 * because the associated get method is called to display diagnostics. The {@link ModelParameterType} specifying the
 * parameters is stored as the {@link OneTypeParameterValues#_modelParameterType} attribute. Everything else associated
 * with the values, including the values themselves, is handled by the subclass.<br>
 * <br>
 * Regardless of the data type of the parameter, sets and gets here assume it is a double.<br>
 * <br>
 * The default {@link XMLReader} and {@link XMLWriter} returned by the {@link #getReader()} and {@link #getWriter()}
 * methods is null. For subclasses reading/writing the values either to XML or other formats, the list
 * {@link #_daysOfTheYearForWhichToReadWriteParameters} provides the days for which parameters must be read, presumably
 * the day for which the model will be run. If the list is emtpy, read in the parameter for all days of the year.
 * 
 * @author hank.herr
 */
public abstract class OneTypeParameterValues implements XMLReadable, XMLWritable
{
    private final ModelParameterType _modelParameterType;

    /**
     * List of the days-of-the-year for which to read parameters. If empty, all parameters should be read regardless of
     * the day of the year.
     */
    private final List<Integer> _daysOfTheYearForWhichToReadWriteParameters = new ArrayList<Integer>();

    public OneTypeParameterValues(final ModelParameterType modelParameterType)
    {
        _modelParameterType = modelParameterType;
    }

    public ModelParameterType getType()
    {
        return _modelParameterType;
    }

    /**
     * Add a day of the year for which to read/write parameters.
     * 
     * @param dayOfYear
     */
    public void addDayOfYearToReadWriteParameters(final int dayOfYear)
    {
        _daysOfTheYearForWhichToReadWriteParameters.add(dayOfYear);
    }

    /**
     * Clear the days of the year so that all days are read/written.
     */
    public void clearDaysOfYearToReadWriteParameters()
    {
        _daysOfTheYearForWhichToReadWriteParameters.clear();
    }

    /**
     * @return A {@link List} of the days of the year for which to read/write parameters. If empty, all days are to be
     *         read/written.
     */
    public List<Integer> getDaysOfTheYearForWhichToReadWriteParameters()
    {
        return _daysOfTheYearForWhichToReadWriteParameters;
    }

    public abstract double getLargestValue();

    public abstract double getSmallestValue();

    /**
     * @param dayOfYear Starts at 1 so it is compatible with {@link Calendar#DAY_OF_YEAR}. The day of the year for which
     *            to get the parameter value. It is presumed that values are stored by day of year for this method. But
     *            it should be easy to map the day of year to months or other date based values.
     * @param secondaryIndex Allows for multiple parameter values to be recorded for each day. For example, in MEFP, for
     *            each day of year, there is one parameter value per canonical events; the number of canonical events is
     *            an upper bound on this parameter. If there is only one value stored per day of year, then ignore this
     *            parameter in the implementation of it and pass in -1 when calling the method.
     * @return The value always returned as a double.
     */
    public abstract double getValue(int dayOfYear, int secondaryIndex);

    /**
     * @param dayOfYear Starts at 1 so it is compatible with {@link Calendar#DAY_OF_YEAR}. The day of the year for which
     *            to get the parameter value. It is presumed that values are stored by day of year for this method. But
     *            it should be easy to map the day of year to months or other date based values.
     * @param secondaryIndex Allows for multiple parameter values to be recorded for each day. For example, in MEFP, for
     *            each day of year, there is one parameter value per canonical events; the number of canonical events is
     *            an upper bound on this parameter. If there is only one value stored per day of year, then ignore this
     *            parameter in the implementation of it and pass in -1 when calling the method.
     * @param value The value to set.
     */
    public abstract void setValue(int dayOfYear, int secondaryIndex, double value);

    /**
     * @return Corresponding to the getValue method, the number of days of year with values.
     */
    public abstract int getNumberOfDaysOfYearWithValues();

    /**
     * @param dayOfYear Starts at 1.
     * @return True if any parameter is not {@link Double#NaN}; false otherwise.
     */
    public abstract boolean areAllParametersMissingForDay(final int dayOfYear);

    /**
     * @return True if there are no non-missing values for the parameters throughout the year. Among other things, this
     *         means diagnostics cannot be displayed.
     */
    public abstract boolean areAllParameterValuesMissing();

    /**
     * Override if the default return of 1 (implying second arg of {@link #getValue(Integer, int)} is not used) does not
     * work for the model.
     * 
     * @return Returns number of indices used for the second argument in {@link #getValue(Integer, int)}. Counting is
     *         assumed to start at 0 and go up to this return value - 1.
     */
    public int getNumberOfIndicesWithComputedValuesPerDay()
    {
        return 1;
    }

    /**
     * Pulls the value for {@link #_modelParameterType} and records it according to the indices stored in values.
     * 
     * @param values
     */
    public void recordValueFrom(final OneSetParameterValues values)
    {
        setValue(values.getDayOfYearIndex(), values.getSecondaryIndex(), values.getValue(getType()));
    }

    /**
     * @param values
     * @return Returns the value for {@link #_modelParameterType} according to the indices stored in values.
     */
    public void putValueInto(final OneSetParameterValues values)
    {
        values.setValue(this._modelParameterType, getValue(values.getDayOfYearIndex(), values.getSecondaryIndex()));
    }

    @Override
    public XMLWriter getWriter()
    {
        return null;
    }

    @Override
    public XMLReader getReader()
    {
        return null;
    }

    @Override
    public String toString()
    {
        return "OneTypeParameterValues: " + this._modelParameterType;
    }

}
