package ohd.hseb.hefs.pe.model;

import java.util.List;

import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.estimation.options.ControlOptionSupplier;
import ohd.hseb.hefs.pe.estimation.options.EstimationControlOptions;
import ohd.hseb.hefs.pe.sources.ForecastSource;
import ohd.hseb.hefs.pe.sources.SourceModelParameters;

public interface ParameterEstimationModel extends ControlOptionSupplier
{

    /**
     * Called before calling
     * {@link #estimateParameters(FullModelParameters, ParameterEstimatorRunInfo, EstimationControlOptions, ForecastSource)}
     * to prepare the estimation control options based on the options in the provided parameters file, the run-time
     * options specified by the user, and the selected forecast source (or null for all sources). The prepared options
     * will then be placed within the provided modelParameters accessible via
     * {@link FullModelParameters#getEstimationControlOptions()}.<br>
     * <br>
     * Note that this method may call
     * {@link FullModelParameters#prepareEstimationOptionsForComputationOfOneSource(EstimationControlOptions, ForecastSource)}
     * , but may also perform other actions as needed.
     * 
     * @param modelParameters The model parameter file. If the forecast source is not null, then this should be set to
     *            the parameters already existing for the identifier to be estimated. Otherwise, it can be a blank set
     *            of parameters. The estimation options within the parameters will be set appropriately for a full model
     *            estimation (i.e., not a one source estimation).
     * @param runInfo Run time information specifying user selections for estimation options.
     * @param forecastSource The forecast source for which to estimate paramters, or null if all parameters are to be
     *            estimated.
     */
    public void prepareEstimationOptions(FullModelParameters modelParameters,
                                         final ParameterEstimatorRunInfo runInfo,
                                         final ForecastSource forecastSource);

    /**
     * Estimate parameter of the model. The resulting parameters are acquired via the get methods.
     * 
     * @param modelParameters {@link FullModelParameters} that are to be estimated. The estimation options to use for
     *            parameter estimation will already be specified in this object by the time this is called.
     * @param runInfo ParameterEstimatorRunInfo providing information for the run, including data handlers.
     * @param oneSource The one forecast source for which to estimate parameters, or null for all sources.
     * @throws ParameterEstimationException
     */
    public void estimateParameters(FullModelParameters modelParameters,
                                   ParameterEstimatorRunInfo runInfo,
                                   ForecastSource oneSource) throws ParameterEstimationException;

    /**
     * @return A list of StatisticParameterCalculator objects that specify all of the parameters required for a model.
     *         This is used for instances where both forecasts and observations are part of the data source.
     */
    public List<ModelParameterType> getAllParametersRequiredForModel(ForecastSource source);

    /**
     * @return A list of model parameter types that specify the parameters required for a model. This is used for
     *         instances where a data source includes only observations, meaning forecast-based parameters are unused.
     */
    public List<ModelParameterType> getObservationParametersRequiredForModel(ForecastSource source);

    /**
     * @return An instance of ModelParameters that contains the parameters required for the subclass of the model.
     */
    public AlgorithmModelParameters getAlgorithmModelParameters();

    /**
     * @param source The {@link ForecastSource} to which the parameters apply.
     * @param dayOfYear Day of year for which parameters will be stored in the returned object.
     * @param secondaryIndex The secondary index value. For example, for MEFP, this is the canonical event index.
     * @return A {@link OneSetParameterValues} instance that is ready to contain values for the given forecast source,
     *         day of year, and canonical event index. See {@link SourceModelParameters} for how the two indices are
     *         used.
     */
    public OneSetParameterValues initializeOneSetParameterValues(ForecastSource source,
                                                                 int dayOfYear,
                                                                 int secondaryIndex);
}
