package ohd.hseb.hefs.pe.setup;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JToolBar;
import javax.swing.JTree;
import javax.swing.JViewport;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.pe.core.ParameterEstimatorDiagnosticPanel;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepOptionsPanel;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.core.StepUnit;
import ohd.hseb.hefs.pe.notice.AvailableIdentifiersChangedNotice;
import ohd.hseb.hefs.pe.notice.SelectedIdentifiersChangedNotice;
import ohd.hseb.hefs.pe.sources.pixml.GenericPIXMLDataHandler;
import ohd.hseb.hefs.pe.tools.DataTypeIdentifier;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifierSupplier;
import ohd.hseb.hefs.pe.tools.LocationIdentifier;
import ohd.hseb.hefs.utils.gui.jtable.GenericTable;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.jobs.HJobMonitorDialog;
import ohd.hseb.hefs.utils.jobs.JobListener;
import ohd.hseb.hefs.utils.piservice.FewsPiServiceProvider;
import ohd.hseb.hefs.utils.piservice.FewsPiServiceProviderListener;
import ohd.hseb.hefs.utils.piservice.FewsPiServiceReconnect20x20Button;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tools.StringTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesHeaderInfo;
import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.HString;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.eventbus.Subscribe;

@SuppressWarnings("serial")
public class ExportPIXMLPanel extends ParameterEstimatorStepOptionsPanel implements ActionListener,
TreeSelectionListener, FewsPiServiceProviderListener, AvailableIdentifiersChangedNotice.Subscriber
{
    private GenericPIXMLDataHandler _piXMLDataHandler = null;
    private final JTree _sourcesToIdentifiersTree = new JTree();
    private final JTree _identifiersToSourcesTree = new JTree();
    private final HashMap<LocationAndDataTypeIdentifier, DefaultMutableTreeNode> _identifierToSourcesToIdentifiersTreeNode = new HashMap<LocationAndDataTypeIdentifier, DefaultMutableTreeNode>();
    private final HashMap<LocationAndDataTypeIdentifier, DefaultMutableTreeNode> _identifierToIdentifiersToSourceTreeNode = new HashMap<LocationAndDataTypeIdentifier, DefaultMutableTreeNode>();
    private JTabbedPane _treePane = null;
    private GenericTable _fileTable;
    private final String _diagnosticTemplateFileName;

    private final JButton _viewButton = HSwingFactory.createJButtonWithIcon("hefsIcons/chart20x20.png",
                                                                            "View Historical Data",
                                                                            this);
    private final JButton _reconnectButton = new FewsPiServiceReconnect20x20Button(this);
    private final JButton _exportButton = HSwingFactory.createJButtonWithIcon("hefsIcons/serverExport20x20.png",
                                                                              "Export Time Series from CHPS DB",
                                                                              this);
    private final JButton _refreshButton = HSwingFactory.createJButtonWithIcon("hefsIcons/refresh20x20.png",
                                                                               "Refresh List of Files and Tree",
                                                                               this);

    public ExportPIXMLPanel(final ParameterEstimatorRunInfo runInfo,
                            final ParameterEstimatorStepProcessor step,
                            final GenericPIXMLDataHandler handler,
                            final String diagnosticTemplateFileName)
    {
        super(runInfo, step);
        _piXMLDataHandler = handler;
        _diagnosticTemplateFileName = diagnosticTemplateFileName;
        initializeDisplay();
        refreshTree();
        updateEnablednessOfViewButton();
        FewsPiServiceProvider.addFewsPiServiceProviderListener(this);
        reactToReinitializedPIServiceConnection();
    }

    @Override
    protected void initializeDisplay()
    {
        setLayout(new BorderLayout());

        _viewButton.setEnabled(false);

        _sourcesToIdentifiersTree.getSelectionModel().setSelectionMode(TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION);
        _identifiersToSourcesTree.getSelectionModel().setSelectionMode(TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION);
        _sourcesToIdentifiersTree.addTreeSelectionListener(this);
        _identifiersToSourcesTree.addTreeSelectionListener(this);

        _treePane = new JTabbedPane(JTabbedPane.TOP);
        final JScrollPane scrollPane2 = new JScrollPane();
        _treePane.addTab("By Identifier", null, scrollPane2, null);
        scrollPane2.setViewportView(_identifiersToSourcesTree);
        final JScrollPane scrollPane1 = new JScrollPane();
        _treePane.addTab("By Source", null, scrollPane1, null);
        scrollPane1.setViewportView(this._sourcesToIdentifiersTree);
        _treePane.addChangeListener(new ChangeListener()
        {
            @Override
            public void stateChanged(final ChangeEvent e)
            {
                updateEnablednessOfViewButton();
            }
        });

        final PIXMLFilesFoundTableModel fileTableModel = new PIXMLFilesFoundTableModel(getRunInfo(), _piXMLDataHandler);
        _fileTable = new GenericTable(fileTableModel);

        final JScrollPane fileTablePane = new JScrollPane(_fileTable);
        fileTablePane.setPreferredSize(new Dimension(50, 100));

        final JPanel panel = new JPanel(new GridBagLayout());
        add(panel, BorderLayout.CENTER);

        final JLabel historicalPixmlFileLabel = new JLabel("Historical PI-XML Files:");
        final GridBagConstraints constraintsForHistoricalPixmlFileLabel = new GridBagConstraints();
        constraintsForHistoricalPixmlFileLabel.insets = new Insets(0, 0, 5, 0);
        constraintsForHistoricalPixmlFileLabel.anchor = GridBagConstraints.NORTHWEST;
        constraintsForHistoricalPixmlFileLabel.fill = GridBagConstraints.BOTH;
        constraintsForHistoricalPixmlFileLabel.gridx = 0;
        constraintsForHistoricalPixmlFileLabel.gridy = 0;
        constraintsForHistoricalPixmlFileLabel.weighty = 0;
        constraintsForHistoricalPixmlFileLabel.weightx = 1;
        panel.add(historicalPixmlFileLabel, constraintsForHistoricalPixmlFileLabel);

        final JLabel theDirectoryNameLabel = new JLabel("Default Directory: "
            + _piXMLDataHandler.getPIXMLDataDirectory().getAbsolutePath());
        theDirectoryNameLabel.setToolTipText(_piXMLDataHandler.getPIXMLDataDirectory().getAbsolutePath());
        final GridBagConstraints constraintsTheDirectoryNameLabel = new GridBagConstraints();
        constraintsTheDirectoryNameLabel.insets = new Insets(0, 0, 5, 0);
        constraintsTheDirectoryNameLabel.anchor = GridBagConstraints.NORTHWEST;
        constraintsTheDirectoryNameLabel.fill = GridBagConstraints.BOTH;
        constraintsTheDirectoryNameLabel.gridx = 0;
        constraintsTheDirectoryNameLabel.gridy = 1;
        constraintsTheDirectoryNameLabel.weighty = 0;
        constraintsTheDirectoryNameLabel.weightx = 1;
        panel.add(theDirectoryNameLabel, constraintsTheDirectoryNameLabel);

        final GridBagConstraints constraintsForFileTableScrollPane = new GridBagConstraints();
        constraintsForFileTableScrollPane.anchor = GridBagConstraints.NORTHWEST;
        constraintsForFileTableScrollPane.fill = GridBagConstraints.BOTH;
        constraintsForFileTableScrollPane.gridx = 0;
        constraintsForFileTableScrollPane.gridy = 2;
        constraintsForFileTableScrollPane.weighty = 1;
        constraintsForFileTableScrollPane.weightx = 1;
        panel.add(fileTablePane, constraintsForFileTableScrollPane);

        final GridBagConstraints constraintsForTabbedPane = new GridBagConstraints();
        constraintsForTabbedPane.anchor = GridBagConstraints.NORTHWEST;
        constraintsForTabbedPane.fill = GridBagConstraints.BOTH;
        constraintsForTabbedPane.gridx = 0;
        constraintsForTabbedPane.gridy = 3;
        constraintsForTabbedPane.weighty = 4;
        constraintsForTabbedPane.weightx = 1;
        panel.add(_treePane, constraintsForTabbedPane);

        final JToolBar toolBar = new JToolBar();
        toolBar.setFloatable(false);
        toolBar.add(_reconnectButton);
        toolBar.addSeparator();
        toolBar.add(_viewButton);
        toolBar.add(_exportButton);
        toolBar.add(HSwingFactory.createFillerJPanel());
        toolBar.add(_refreshButton);
        add(toolBar, BorderLayout.SOUTH);
    }

    @Override
    @Subscribe
    public void reactToAvailableIdentifiersChanged(final AvailableIdentifiersChangedNotice evt)
    {
        refreshTree();
        _fileTable.setVisible(false);
        _fileTable.setVisible(true);
    }

    private void refreshTree()
    {
        _sourcesToIdentifiersTree.setModel(new DefaultTreeModel(createJTreeTopNodeForSourceToIdentifierMap()));
        _identifiersToSourcesTree.setModel(new DefaultTreeModel(createJTreeTopNodeForIdentifiersToSourceMap()));
    }

    /**
     * @return A root node for displaying the source to identifier map.
     */
    private DefaultMutableTreeNode createJTreeTopNodeForSourceToIdentifierMap()
    {
        final DefaultMutableTreeNode topNode = new DefaultMutableTreeNode("Sources Found", true);
        final HashMap<String, DefaultMutableTreeNode> locationToNodeMap = new HashMap<String, DefaultMutableTreeNode>();

        //Build and sort list.
        final List<String> allSources = new ArrayList<String>();
        allSources.addAll(_piXMLDataHandler.getPIXMLFileNamesRead());
        Collections.sort(allSources);

        //Do it.
        for(final String dataSource: allSources)
        {
            //Construct file node.
            final File sourceFile = new File(dataSource);
            final DefaultMutableTreeNode node = new DefaultMutableTreeNode("File: " + sourceFile.getName() + " ("
                + sourceFile.getParent() + ")", true);
            topNode.add(node);
            locationToNodeMap.clear();

            //For each identifier...
            for(final LocationAndDataTypeIdentifier identifier: _piXMLDataHandler.getOriginalIdentifiersUsedFromSource(dataSource))
            {
                //If no location node exists yet, add it.
                DefaultMutableTreeNode locationNode = locationToNodeMap.get(identifier.getLocationId());
                if(locationNode == null)
                {
                    locationNode = new DefaultMutableTreeNode(new LocationIdentifier(identifier.getLocationId()), true);
                    locationToNodeMap.put(identifier.getLocationId(), locationNode);
                    node.add(locationNode);
                }

                //Add the parameter node to the location node and time series info to the parameter node.
                final DefaultMutableTreeNode parameterNode = new DefaultMutableTreeNode(new DataTypeIdentifier(identifier.getParameterId()),
                                                                                        true);
                addTimeSeriesInfoNodes(parameterNode, identifier);
                locationNode.add(parameterNode);
                _identifierToSourcesToIdentifiersTreeNode.put(identifier, parameterNode);
            }
        }
        return topNode;
    }

    /**
     * @return A root node displaying the identifiers to source map.
     */
    private DefaultMutableTreeNode createJTreeTopNodeForIdentifiersToSourceMap()
    {
        final DefaultMutableTreeNode topNode = new DefaultMutableTreeNode("Identifiers Found", true);
        final HashMap<String, DefaultMutableTreeNode> locationToNodeMap = new HashMap<String, DefaultMutableTreeNode>();

        //Sort the identifiers to display.
        final List<LocationAndDataTypeIdentifier> identifiers = Lists.newArrayList(_piXMLDataHandler.getAllOriginalIdentifiersForWhichDataWasFound());
        Collections.sort(identifiers, new Comparator<LocationAndDataTypeIdentifier>()
        {
            @Override
            public int compare(final LocationAndDataTypeIdentifier o1, final LocationAndDataTypeIdentifier o2)
            {
                return o1.buildStringToDisplayInTree().compareTo(o2.buildStringToDisplayInTree());
            }
        });

        //For all identifiers found, both obs and fcst...
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            //If no location node exists, yet, add it.
            DefaultMutableTreeNode locationNode = locationToNodeMap.get(identifier.getLocationId());
            if(locationNode == null)
            {
                locationNode = new DefaultMutableTreeNode(new LocationIdentifier(identifier.getLocationId()), true);
                locationToNodeMap.put(identifier.getLocationId(), locationNode);
                topNode.add(locationNode);
            }

            //Add the parameter node and then add the file and time series info.
            final DefaultMutableTreeNode parameterNode = new DefaultMutableTreeNode(new DataTypeIdentifier(identifier.getParameterId()),
                                                                                    true);
            final Object source = _piXMLDataHandler.getSourceProvidingDataForOriginalIdentifier(identifier);
            final String nodeName = "File: " + source;
            parameterNode.add(new DefaultMutableTreeNode(nodeName, false));
            addTimeSeriesInfoNodes(parameterNode, identifier);
            locationNode.add(parameterNode);
            _identifierToIdentifiersToSourceTreeNode.put(identifier, parameterNode);
        }
        return topNode;
    }

    /**
     * Add time series info nodes to a tree node.
     * 
     * @param node The node to which to add information.
     * @param originalIdentifier The original, raw identifier for which to acquire the header information and display
     *            it.
     */
    private void addTimeSeriesInfoNodes(final DefaultMutableTreeNode node,
                                        final LocationAndDataTypeIdentifier originalIdentifier)
    {
        final TimeSeriesHeaderInfo info = this._piXMLDataHandler.getUsedTimeSeriesInformationForOriginalIdentifier(originalIdentifier);
        node.add(new DefaultMutableTreeNode("Start Time: "
            + HCalendar.buildDateTimeTZStr(HCalendar.computeCalendarFromMilliseconds(info.getStartTime())), false));
        node.add(new DefaultMutableTreeNode("End Time: "
            + HCalendar.buildDateTimeTZStr(HCalendar.computeCalendarFromMilliseconds(info.getEndTime())), false));
        node.add(new DefaultMutableTreeNode("Time Step (hours): "
            + info.getTimeSeriesHeader().getTimeStep().getStepMillis() / HCalendar.MILLIS_IN_HR, false));
    }

    /**
     * Displays a diagnostic panel for selected time series. The display is built via a Job embedded below.
     */
    private void displayDiagnosticProduct()
    {
        final List<LocationAndDataTypeIdentifier> identifiers = getSelectedIdentifiersList();
        if(identifiers.size() == 0) // Don't do anything - only display one at a time right now.
        {
            return;
        }

        //Check the selected identifiers to determine if they are for the same location.
        for(int i = 1; i < identifiers.size(); i++)
        {
            if(!identifiers.get(i).isForSameLocation(identifiers.get(i - 1)))
            {
                final String message = "Not all selected entries are for the same location id.";
                fireDiagnostic("Unable to build chart displaying PI-xml data:", message);
                JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(ExportPIXMLPanel.this),
                                              StringTools.wordWrap(message, 80),
                                              "Unable to Build Diagnostic Panel",
                                              JOptionPane.ERROR_MESSAGE);
                return;
            }
            if(!identifiers.get(i).isSameTypeOfData(identifiers.get(i - 1)))
            {
                final String message = "Not all selected entries are of the same data type.";
                fireDiagnostic("Unable to build chart displaying PI-xml data:", message);
                JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(ExportPIXMLPanel.this),
                                              StringTools.wordWrap(message, 80),
                                              "Unable to Build Diagnostic Panel",
                                              JOptionPane.ERROR_MESSAGE);
                return;
            }
        }
        final LocationAndDataTypeIdentifier usedIdentifier = identifiers.get(0);

        final GenericJob buildPanelJob = new GenericJob()
        {
            @Override
            public void processJob()
            {
                setIndeterminate(true);
                updateNote("Loading time series from source...");

                //Create list of translated identifiers
                final List<LocationAndDataTypeIdentifier> translatedIdentifiers = _piXMLDataHandler.translateIdentifiers(identifiers);

                //Load the time series for the translated identifiers.  This may result in more time series than needed.
                try
                {
                    _piXMLDataHandler.loadOriginalTimeSeries(translatedIdentifiers);
                }
                catch(final Exception e)
                {
                    fireProcessJobFailure(new Exception("Unable to load selected time series for:" + e.getMessage()),
                                          true);
                    return;
                }

                //For each of the provided identifiers, translate it, get the time series loaded for the translated identifier, and add them to
                //the tsList only if the header for the loaded ts yields an identifier equal to the provided (pre-translated) identifier.
                final List<TimeSeriesArray> tsList = new ArrayList<TimeSeriesArray>();
                for(final LocationAndDataTypeIdentifier identifier: identifiers)
                {
                    final LocationAndDataTypeIdentifier transId = _piXMLDataHandler.translateIdentifier(identifier);
                    for(final TimeSeriesArray ts: ListTools.concat(_piXMLDataHandler.getLoadedObservedTimeSeries(transId),
                                                                   _piXMLDataHandler.getLoadedForecastTimeSeries(transId)))
                    {
                        if(LocationAndDataTypeIdentifier.get(ts.getHeader()).equals(identifier))
                        {
                            ListTools.addItemIfNotAlreadyInList(tsList, ts);
                        }
                    }
                }

                if(tsList.isEmpty())
                {
                    fireProcessJobFailure(new Exception("Time series found are either empty or all missing values."),
                                          true);
                    return;
                }

                //TimeSeriesArrays series = 
                //        TimeSeriesArraysTools.convertListOfTimeSeriesToTimeSeriesArrays(seriesList);
                //TimeSeriesArraysTools.writeToFile(new File("testdata/diagnosticProducts/historicalDisplayDiagnosticsTS.xml"), series);

                updateNote("Constructing chart...");
                fireDiagnostic(constructDiagnosticPanel(usedIdentifier, tsList));
                endTask();
            }
        };

        final HJobMonitorDialog jobDialog = new HJobMonitorDialog(this,
                                                                  "Building Chart to Display PI-xml Data",
                                                                  buildPanelJob,
                                                                  false);
        buildPanelJob.addListener(new JobListener()
        {
            @Override
            public void processJobFailure(final Exception exc, final GenericJob theJob, final boolean displayMessage)
            {
                //exc.printStackTrace();
                fireDiagnostic("Unable to build chart displaying PI-xml data:", exc.getMessage());
                JOptionPane.showMessageDialog(ExportPIXMLPanel.this,
                                              StringTools.wordWrap(exc.getMessage(), 80),
                                              "Unable to Build Diagnostic Panel",
                                              JOptionPane.ERROR_MESSAGE);
            }

            @Override
            public void processSuccessfulJobCompletion(final GenericJob theJob)
            {
            }
        });
        buildPanelJob.startJob();
        jobDialog.setMinimumSize(new Dimension(350, 10));
        jobDialog.setModal(true);
        jobDialog.setVisible(true);
    }

    protected ParameterEstimatorDiagnosticPanel constructDiagnosticPanel(final LocationAndDataTypeIdentifier usedIdentifier,
                                                                         final List<TimeSeriesArray> tsList)
    {
        return new ExportPIXMLDiagnosticPanel(usedIdentifier, tsList, getDiagnosticTemplateFileName());
    }

    private void exportHistoricalFiles()
    {
        //Check for PI-service connection first
        if(!FewsPiServiceProvider.isDefaultServiceConnected())
        {
            final String portString = JOptionPane.showInputDialog(SwingTools.getGlobalDialogParent(this),
                                                                  "Enter the port number to use to connect to the PI-service. "
                                                                      + "To identify the number, check the log panel at start-up for a line similar to this:\n\n"
                                                                      + "    10-14-2010 15:01:02 INFO - Started FewsPiServiceImpl on localHost : 8100\n\n"
                                                                      + "The number at the end of the line is the port number to enter here:",
                                                                  "Enter Port Number",
                                                                  JOptionPane.QUESTION_MESSAGE);
            if((portString == null) || (portString.length() == 0))
            {
                return;
            }
        }

        final PIXMLTimeSeriesExportJob exporter = new PIXMLTimeSeriesExportJob(getRunInfo(),
                                                                               _stepProcessor == null ? null : _stepProcessor.getClass(),
                                                                               _piXMLDataHandler);
        final HJobMonitorDialog jobDialog = new HJobMonitorDialog(this,
                                                                  "Exporting Historical Time Series from CHPS Database",
                                                                  exporter,
                                                                  false);
        exporter.addListener(new JobListener()
        {
            @Override
            public void processJobFailure(final Exception exc, final GenericJob theJob, final boolean displayMessage)
            {
                // Making sure the status dialog is closed before the error/message dialog is displayed.
                jobDialog.setVisible(false);
                if(displayMessage)
                {
                    JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(ExportPIXMLPanel.this),
                                                  StringTools.wordWrap(exc.getMessage(), 100),
                                                  "Error Exporting Time Series!",
                                                  JOptionPane.ERROR_MESSAGE);
                }
                if(exporter.wereAnyNewFilesExported())
                {
                    try
                    {
                        _piXMLDataHandler.readUnreadHistoricalFilesAndAddToLists();
                    }
                    catch(final Exception e)
                    {
                        e.printStackTrace();
                        System.err.println("####>> INTERNAL ERROR attempting to re-read exported time series files!");
                    }
                    getRunInfo().notifyAvailableIdentifiersChanged(ExportPIXMLPanel.this);
                }
                refreshTree();
                _fileTable.setVisible(false);
                _fileTable.setVisible(true);
            }

            @Override
            public void processSuccessfulJobCompletion(final GenericJob theJob)
            {
                // Making sure the status dialog is closed before the error/message dialog is displayed.
                jobDialog.setVisible(false);

                if(exporter.wereAnyNewFilesExported())
                {
                    try
                    {
                        _piXMLDataHandler.readUnreadHistoricalFilesAndAddToLists();
                    }
                    catch(final Exception e)
                    {
                        e.printStackTrace();
                        System.err.println("####>> INTERNAL ERROR attempting to re-read exported time series files!");
                    }
                    getRunInfo().notifyAvailableIdentifiersChanged(ExportPIXMLPanel.this);
                }
                if(!((PIXMLTimeSeriesExportJob)theJob).getEmptyIdentifiers().isEmpty())
                {
                    final String allLocationsStr = HString.buildStringFromList(LocationAndDataTypeIdentifier.createListOfIdentifierStrings(((PIXMLTimeSeriesExportJob)theJob).getEmptyIdentifiers()),
                                                                               ",");
                    JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(ExportPIXMLPanel.this),
                                                  StringTools.wordWrap("Job was successfully completed, but for the following selected identifiers, "
                                                                           + "CHPS provided time series that were empty (all missing values) and files were not exported:\n\n"
                                                                           + allLocationsStr
                                                                           + "\n\nCheck the PI-service configuration file "
                                                                           + "and the availability of the historical time series.",
                                                                       128),
                                                  "Missing Time Series Not Exported!",
                                                  JOptionPane.WARNING_MESSAGE);
                }
                refreshTree();
                _fileTable.setVisible(false);
                _fileTable.setVisible(true);
            }
        });
        exporter.setParentComponent(jobDialog);
        exporter.startJob();
        jobDialog.setMinimumSize(new Dimension(500, 10));
        jobDialog.setModal(true);
        jobDialog.setVisible(true);
    }

    /**
     * @return the active tree
     */
    private JTree getActiveTree()
    {
        JTree activeTree = null;
        try
        {
            final JScrollPane scrollPane = (JScrollPane)_treePane.getSelectedComponent();
            final JViewport vp = scrollPane.getViewport();
            activeTree = (JTree)vp.getView();
            if(activeTree != _identifiersToSourcesTree && activeTree != _sourcesToIdentifiersTree)
            {
                throw new Exception("Unknown active tree.");
            }
        }
        catch(final Exception e)
        {
            throw new RuntimeException("A timeseries selection tree is not currently active.", e);
        }
        return activeTree;
    }

    /**
     * @return the active tree
     */
    private HashMap<LocationAndDataTypeIdentifier, DefaultMutableTreeNode> getActiveIdentifierToNodeMap()
    {
        if(getActiveTree() == _identifiersToSourcesTree)
        {
            return this._identifierToIdentifiersToSourceTreeNode;
        }
        return this._identifierToSourcesToIdentifiersTreeNode;
    }

    /**
     * Adds the node's info to the identifier. Throws an exception if this would overwrite information.
     * 
     * @param node
     * @param id
     */
    private void addNodeInformationToIdentifierSupplier(final DefaultMutableTreeNode node,
                                                        final LocationAndDataTypeIdentifierSupplier id)
    {
        final Object userObject = node.getUserObject();
        if(userObject instanceof LocationIdentifier)
        {
            if(id.getLocationId() == null)
            {
                id.setLocationId(userObject.toString());
            }
            else
            {
                throw new RuntimeException("More than one location in path.");
            }
        }
        else if(userObject instanceof DataTypeIdentifier)
        {
            if(id.getParameterId() == null)
            {
                id.setParameterId(userObject.toString());
            }
            else
            {
                throw new RuntimeException("More than one datatype in path.");
            }
        }
    }

    /**
     * Set of every LocationId / ParameterId from this node and its children.
     * 
     * @param node the node to explore
     * @param base the original information we know about the identifier
     * @return set of all identifiers found
     */
    private Set<LocationAndDataTypeIdentifierSupplier> grabIdentifiersFromNode(final DefaultMutableTreeNode node,
                                                                               final LocationAndDataTypeIdentifierSupplier base)
    {
        addNodeInformationToIdentifierSupplier(node, base);
        final Set<LocationAndDataTypeIdentifierSupplier> supplierSet = Sets.newHashSet();

        // Check if we have location and parameter. If so, skip checking children.
        if(base.getLocationId() != null && base.getParameterId() != null)
        {
            supplierSet.add(new LocationAndDataTypeIdentifierSupplier(base));
        }
        else
        {
            for(int i = 0; i < node.getChildCount(); i++)
            {
                final DefaultMutableTreeNode child = (DefaultMutableTreeNode)node.getChildAt(i);
                supplierSet.addAll(grabIdentifiersFromNode(child, new LocationAndDataTypeIdentifierSupplier(base)));
            }
        }
        return supplierSet;
    }

    /**
     * Return set of identifiers for this path.
     * 
     * @param path
     * @return
     */
    private Set<LocationAndDataTypeIdentifier> grabIdentifiersFromPath(final TreePath path)
    {
        final LocationAndDataTypeIdentifierSupplier supplier = new LocationAndDataTypeIdentifierSupplier();
        for(int i = 0; i < path.getPathCount() - 1; i++)
        {
            addNodeInformationToIdentifierSupplier((DefaultMutableTreeNode)path.getPathComponent(i), supplier);
        }
        final Set<LocationAndDataTypeIdentifierSupplier> supplierSet = grabIdentifiersFromNode((DefaultMutableTreeNode)path.getLastPathComponent(),
                                                                                               supplier);
        final Set<LocationAndDataTypeIdentifier> idSet = Sets.newHashSet();

        for(final LocationAndDataTypeIdentifierSupplier sup: supplierSet)
        {
            idSet.add(sup.get());
        }
        return idSet;

    }

    /**
     * @return set of all selected identifiers in the active tree
     */
    private Set<LocationAndDataTypeIdentifier> getSelectedIdentifiers()
    {
        // Grab the active tree.
        final JTree activeTree = getActiveTree();

        // Add each selected path and all sub-nodes.
        final TreePath[] paths = activeTree.getSelectionPaths();
        final Set<LocationAndDataTypeIdentifier> idSet = new HashSet<LocationAndDataTypeIdentifier>();
        if(paths != null)
        {
            for(final TreePath tp: paths)
            {
                idSet.addAll(grabIdentifiersFromPath(tp));
            }
        }

        return idSet;
    }

    private List<LocationAndDataTypeIdentifier> getSelectedIdentifiersList()
    {
        final List<LocationAndDataTypeIdentifier> list = new ArrayList<LocationAndDataTypeIdentifier>();
        list.addAll(getSelectedIdentifiers());
        return list;
    }

    private void updateEnablednessOfViewButton()
    {
        final List<LocationAndDataTypeIdentifier> identifiers = this.getSelectedIdentifiersList();
        if(identifiers.size() >= 1)
        {
            _viewButton.setEnabled(true);
            _viewButton.setToolTipText("View the exported historical data for "
                + identifiers.get(0).buildStringToDisplayInTree() + ".");
        }
        else
        {
            _viewButton.setEnabled(false);
            _viewButton.setToolTipText("Select a single row to view the historical data for that location and data type.");
        }
    }

    @Override
    public List<LocationAndDataTypeIdentifier> getStepUnitsToPerform()
    {
        return new ArrayList<LocationAndDataTypeIdentifier>(getSelectedIdentifiers());
    }

    @Override
    public void reactToSelectedIdentifiersChanged(final SelectedIdentifiersChangedNotice evt)
    {
        // This panel displays all locations, so there is no need to react so a subset being based on data type.
    }

    @Override
    public void gotoUnit(final Collection<StepUnit> units)
    {
        if(units.isEmpty())
        {
            getActiveTree().clearSelection();
            return;
        }

        final TreePath[] selectedPaths = new TreePath[units.size()];
        int i = 0;
        for(final StepUnit unit: units)
        {
            final LocationAndDataTypeIdentifier identifier = (LocationAndDataTypeIdentifier)unit;
            final DefaultMutableTreeNode node = getActiveIdentifierToNodeMap().get(identifier);
            final TreePath path = new TreePath(node.getPath());
            selectedPaths[i] = path;
            i++;
        }
        getActiveTree().setSelectionPaths(selectedPaths);
    }

    @Override
    public void actionPerformed(final ActionEvent ev)
    {
        if(ev.getSource() == this._viewButton)
        {
            displayDiagnosticProduct();
        }
        else if(ev.getSource() == _exportButton)
        {
            exportHistoricalFiles();
        }
        else if(ev.getSource() == _refreshButton)
        {
            try
            {
                _piXMLDataHandler.initialize();
            }
            catch(final Exception e)
            {
                JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(this), "Problems refreshing display: "
                    + e.getMessage(), "Error Reading Exported Files", JOptionPane.ERROR_MESSAGE);
            }
            refreshTree();
            _fileTable.setVisible(false);
            _fileTable.setVisible(true);
            getRunInfo().notifyAvailableIdentifiersChanged(ExportPIXMLPanel.this);
        }
    }

    @Override
    public void valueChanged(final TreeSelectionEvent e)
    {
        updateEnablednessOfViewButton();

        //For all newly selected/unselected paths, if the path is selected and points to a leaf,
        //make sure its parent is selected.
        getActiveTree().removeTreeSelectionListener(this);
        for(final TreePath path: e.getPaths())
        {
            if(this.getActiveTree().isPathSelected(path))
            {
                final TreeNode node = (TreeNode)path.getLastPathComponent();
                if(node.isLeaf())
                {
                    getActiveTree().addSelectionPath(path.getParentPath());
                }
            }
        }
        getActiveTree().addTreeSelectionListener(this);
    }

    @Override
    public void reactToReinitializedPIServiceConnection()
    {
        _exportButton.setEnabled(FewsPiServiceProvider.isDefaultServiceConnected());
        if(_exportButton.isEnabled())
        {
            _exportButton.setToolTipText("Export Time Series from CHPS DB");
        }
        else
        {
            _exportButton.setToolTipText("<html>Currently not connected to a CHPS PI-service, so time series cannot be exported.<br>"
                + "Click on the Reconnect Button to establish a connection.</html>");
        }
    }

    protected String getDiagnosticTemplateFileName()
    {
        return _diagnosticTemplateFileName;
    }

    // XXX rewrite
//    public static void main(String args[])
//    {
//        LoggingTools.initializeStdoutLoggerForDebugging("ohd.hseb");
//        try
//        {
//            HEFSEnsPostPIXMLDataHandler handler = new HEFSEnsPostPIXMLDataHandler(new File("testdata/guiEnsPostTestBed"));
//            handler.initialize();
//
//            ExportPIXMLPanel panel = new ExportPIXMLPanel(handler);
//            panel.addOptionsPanelListener(new ParameterEstimatorStepOptionsPanelListener()
//            {
//
//                @Override
//                public void displayErrorMessage(String headerLine, String messageBody)
//                {
//                    System.err.println("ERROR MESSAGE HEADER: " + headerLine);
//                    System.err.println("ERROR MESSAGE BODY: " + messageBody);
//                }
//
//                @Override
//                public void displayDiagnosticPanel(ParameterEstimatorDiagnosticPanel panel)
//                {
//                    JFrame diagFrame = new JFrame("DIAG PANEL");
//                    diagFrame.setContentPane(panel);
//                    diagFrame.setSize(600, 400);
//                    diagFrame.setContentPane(panel);
//                    diagFrame.setVisible(true);
//                    diagFrame.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
//                }
//
//                @Override
//                public void reactToStepStatusChange()
//                {
//                    System.out.println("####>> REACTING TO STEP STATUS CHANGE");
//                }
//
//                @Override
//                public void reactToLocationSelectionChange()
//                {
//                    System.out.println("####>> REACTING TO LOC SELECT CHANGE");
//                }
//            });
//            JFrame frame = new JFrame("Options Panel");
//            frame.setSize(600, 400);
//            frame.setContentPane(panel);
//            frame.setVisible(true);
//            frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
//        }
//        catch(Exception e)
//        {
//            System.out.println("Unable to initialize handler.  Exception was " + e.getMessage());
//        }
//    }
}
