package ohd.hseb.hefs.pe.setup;

import static com.google.common.collect.Lists.newArrayList;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JToolBar;
import javax.swing.ListSelectionModel;

import nl.wldelft.util.Period;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.sources.pixml.GenericPIXMLDataHandler;
import ohd.hseb.hefs.pe.tools.GenericSummaryTablePanel;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.datetime.HDateChooserPanel;
import ohd.hseb.hefs.utils.gui.jtable.TableTools;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.piservice.FewsPiServiceProvider;
import ohd.hseb.hefs.utils.tools.StringTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesHeaderInfo;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesHeaderInfoList;
import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.HString;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.Lists;

public class PIXMLTimeSeriesExportJob extends GenericJob
{
    private static final Logger LOG = LogManager.getLogger(PIXMLTimeSeriesExportJob.class);

    //NEED A MECHANISM TO SPECIFY THE PORT NUMBER!!!
    private PIXMLLocationSummaryTableModel _model;
    private final ParameterEstimatorRunInfo _runInfo;
    private final Class<? extends ParameterEstimatorStepProcessor> _step;
    private final GenericPIXMLDataHandler _handler;
    private boolean _wereAnyNewFilesExported = false;
    private List<LocationAndDataTypeIdentifier> _selectedIdentifiers = null;
    private final List<LocationAndDataTypeIdentifier> _emptyIdentifiers = new ArrayList<LocationAndDataTypeIdentifier>();
    private Date _startDateTime = HCalendar.processDate("1940-01-01 00:00:00").getTime();
// Jim Ward changed 2000 to 2100 per https://vlab.noaa.gov/redmine/issues/97241
    private Date _endDateTime = HCalendar.processDate("2100-01-01 00:00:00").getTime();

    public PIXMLTimeSeriesExportJob(final ParameterEstimatorRunInfo runInfo,
                                    final Class<? extends ParameterEstimatorStepProcessor> step,
                                    final GenericPIXMLDataHandler handler)
    {
        _runInfo = runInfo;
        _step = step;
        _handler = handler;
    }

    public ParameterEstimatorRunInfo getRunInfo()
    {
        return _runInfo;
    }

    public List<LocationAndDataTypeIdentifier> getEmptyIdentifiers()
    {
        return this._emptyIdentifiers;
    }

    public List<LocationAndDataTypeIdentifier> getSelectedIdentifiers()
    {
        return this._selectedIdentifiers;
    }

    public boolean wereAnyNewFilesExported()
    {
        return this._wereAnyNewFilesExported;
    }

    @SuppressWarnings("unchecked")
    private TimeSeriesHeaderInfoList retrieveTimeSeriesHeaderInfoList()
    {
        TimeSeriesHeaderInfoList tsInfoList = null;
        try
        {
// Jim Ward changed 2000 to 2100 per https://vlab.noaa.gov/redmine/issues/97241
            LOG.debug("System time for retrieving time series info list is " + FewsPiServiceProvider.getSystemTime()
                + ".");
            tsInfoList = FewsPiServiceProvider.retrieveTimeSeriesHeaders(_handler.getPIServiceClientId(),
                                                                         _handler.getPIServiceQueryId(),
                                                                         HCalendar.processDate("1940-01-01 00:00:00")
                                                                                  .getTime(),
                                                                         FewsPiServiceProvider.getSystemTime(),
                                                                         HCalendar.processDate("2100-01-01 00:00:00")
                                                                                  .getTime(),
                                                                         null,
                                                                         null,
                                                                         null,
                                                                         -1,
                                                                         false);

            // LOG.info("JFW After FewsPiServiceProvider.retrieveTimeSeriesHeaders ...");

            if(tsInfoList != null)
            {
                Collections.sort(tsInfoList);
            }
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            fireProcessJobFailure(new Exception("Unable to load time series information from CHPS DB:\n"
                                      + e.getMessage()),
                                  true);
            return null;
        }
        return tsInfoList;
    }

    /**
     * This method will signal user if any unusable time series are found.
     * 
     * @param tsInfoList Time series header list from which list of identifiers is built.
     * @return List of usable identifiers based on {@link #_handler}'s
     *         {@link GenericPIXMLDataHandler#includeTimeSeries(LocationAndDataTypeIdentifier)} method.
     */
    private List<LocationAndDataTypeIdentifier> checkAndBuildListOfIdentifiers(final TimeSeriesHeaderInfoList tsInfoList)
    {
        //Build a list of usable identifiers and unused identifiers.  If any unused identifiers are found,
        //message the user.
        final List<LocationAndDataTypeIdentifier> identifiers = convertToListOfIdentifiers(tsInfoList);
        final List<LocationAndDataTypeIdentifier> usedIdentifiers = Lists.newArrayList();
        final List<LocationAndDataTypeIdentifier> unusedIdentifiers = Lists.newArrayList();
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            if(_handler.includeTimeSeries(identifier))
            {
                usedIdentifiers.add(identifier);
            }
            else
            {
                unusedIdentifiers.add(identifier);
            }
        }
        if(!unusedIdentifiers.isEmpty())
        {
            JOptionPane.showMessageDialog(getParentComponent(),
                                          "Time series returned from PI-service with following identifiers are not usable:\n"
                                              + StringTools.wordWrap(HString.buildStringFromList(LocationAndDataTypeIdentifier.createListOfIdentifierStrings(unusedIdentifiers),
                                                                                                 ", "),
                                                                     100) + ".",
                                          "Unusable Time Series Found",
                                          JOptionPane.WARNING_MESSAGE);
        }
        return usedIdentifiers;
    }

    /**
     * Opens a dialog to allow for the user to select time series to acquire from PI-service.
     * 
     * @param tsInfoList List of headers found in the time series returned from the PI-service query.
     */
    private void obtainSelectedLocationsFromUser(final TimeSeriesHeaderInfoList tsInfoList)
    {
        _selectedIdentifiers = null;
        _model = new PIXMLLocationSummaryTableModel(getRunInfo(), _step, _handler);

        //Create the panel and query the user.
        _model.setIdentifiers(checkAndBuildListOfIdentifiers(tsInfoList));
        final JButton checkAllButton = TableTools.createCheckAllButton(_model, "Check all rows for export");
        final JButton uncheckAllButton = TableTools.createUncheckAllButton(_model, "Uncheck all rows for export");
        final JButton checkSelectedButton = TableTools.createCheckSelectedButton(_model,
                                                                                 "Check selected rows for export");
        final JButton uncheckSelectedButton = TableTools.createUncheckSelectedButton(_model,
                                                                                     "Uncheck selected rows for export");
        final List<Component> extraButtons = newArrayList((Component)new JToolBar.Separator(null));
        extraButtons.add(checkAllButton);
        extraButtons.add(uncheckAllButton);
        extraButtons.add(checkSelectedButton);
        extraButtons.add(uncheckSelectedButton);

        final JPanel tableJPanel = new JPanel();

        final GenericSummaryTablePanel tablePanel = new GenericSummaryTablePanel(getRunInfo(),
                                                                                 "Select Historical Time Series to Export",
                                                                                 _model,
                                                                                 true,
                                                                                 extraButtons);
        tablePanel.getTable().setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
        _model.setContainingTable(tablePanel.getTable());
        tablePanel.getTable().setAutoCreateRowSorter(true);

//        toolBar.add(new HDateChooserPanel(new Date(System.currentTimeMillis()),
//                                          HDateChooserPanel.OPTION_DATE,
//                                          "Start Time"));

        tablePanel.setPreferredSize(new Dimension(400, 400));
        tablePanel.turnOffScrollingForMainPanel();
        tablePanel.getTable().getColumnModel().removeColumn(tablePanel.getTable().getColumnModel().getColumn(3));
        tablePanel.getTable().getColumnModel().removeColumn(tablePanel.getTable().getColumnModel().getColumn(2));
        tablePanel.getTable().getColumnModel().getColumn(0).setPreferredWidth(100);
        tablePanel.getTable().getColumnModel().getColumn(1).setPreferredWidth(100);
        tablePanel.getTable().getColumnModel().getColumn(2).setPreferredWidth(30);
        tablePanel.getTable().getColumnModel().getColumn(3).setPreferredWidth(30);
        tablePanel.getTable().moveColumn(3, 0);

        final HDateChooserPanel startDateChooserPanel = new HDateChooserPanel(HCalendar.processDate("1900-01-01 00:00:00 GMT")
                                                                                       .getTime(),
                                                                              HDateChooserPanel.OPTION_DATE_TIME,
                                                                              "Start Date",
                                                                              true);

        final HDateChooserPanel endDateChooserPanel = new HDateChooserPanel(new Date(System.currentTimeMillis()),
                                                                            HDateChooserPanel.OPTION_DATE_TIME,
                                                                            "End Date",
                                                                            true);

        final JPanel startDatePanel = new JPanel(new BorderLayout());
        final JPanel endDatePanel = new JPanel(new BorderLayout());
        startDatePanel.add(startDateChooserPanel, BorderLayout.WEST);
        endDatePanel.add(endDateChooserPanel, BorderLayout.WEST);

        tableJPanel.setLayout(new BoxLayout(tableJPanel, BoxLayout.Y_AXIS));
        tableJPanel.add(startDatePanel);
        tableJPanel.add(endDatePanel);

        tableJPanel.add(tablePanel);

        final int option = JOptionPane.showConfirmDialog(getParentComponent(),
                                                         tableJPanel,
                                                         "Select Time Series",
                                                         JOptionPane.OK_CANCEL_OPTION,
                                                         JOptionPane.QUESTION_MESSAGE);
        if(option == JOptionPane.OK_OPTION)
        {
            if(startDateChooserPanel.isEnabled())
            {
                _startDateTime = startDateChooserPanel.getDate();
            }
            if(endDateChooserPanel.isEnabled())
            {
                _endDateTime = endDateChooserPanel.getDate();
            }

            _selectedIdentifiers = _model.getIdentifiersToExport();
            if(_selectedIdentifiers.size() == 0)
            {
                fireProcessJobFailure(new Exception("No locations were selected."), true);
                _selectedIdentifiers = null;
            }
        }
        else
        {
            fireProcessJobFailure(new Exception("Export was canceled by user."), false);
        }
    }

    private TimeSeriesArrays retrieveTimeSeriesFromCHPSDB(final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        TimeSeriesArrays extractedTS = null;
        TimeSeriesArrays trimmedTS = null;
        try
        {
// Jim Ward changed 2000 to 2100 per https://vlab.noaa.gov/redmine/issues/97241
            LOG.debug("System time for retrieving time series is " + FewsPiServiceProvider.getSystemTime() + ".");

            extractedTS = FewsPiServiceProvider.retrieveTimeSeries(_handler.getPIServiceClientId(),
                                                                   _handler.getPIServiceQueryId(),
                                                                   HCalendar.processDate("1940-01-01 00:00:00")
                                                                            .getTime(),
                                                                   FewsPiServiceProvider.getSystemTime(),
                                                                   HCalendar.processDate("2100-01-01 00:00:00")
                                                                            .getTime(),
                                                                   new String[]{identifier.getParameterId()},
                                                                   new String[]{identifier.getLocationId()},
                                                                   null,
                                                                   -1,
                                                                   false);

            // LOG.info("JFW After FewsPiServiceProvider.retrieveTimeSeries ...");

            LOG.debug("Start time for the time series is " + HCalendar.buildDateTimeTZStr(_startDateTime.getTime()));
            LOG.debug("End time for the time series is " + HCalendar.buildDateTimeTZStr(_endDateTime.getTime()));

//            ts = FewsPiServiceProvider.retrieveTimeSeries(_handler.getPIServiceClientId(),
//                                                          _handler.getPIServiceQueryId(),
//                                                          _startDateTime,
//                                                          FewsPiServiceProvider.getSystemTime(),
//                                                          _endDateTime,
//                                                          new String[]{identifier.getParameterId()},
//                                                          new String[]{identifier.getLocationId()},
//                                                          null,
//                                                          -1,
//                                                          false);
//            ts.subArrays( new Period)

            trimmedTS = extractedTS.subArrays(new Period(_startDateTime, _endDateTime));

            if((trimmedTS == null) || (trimmedTS.size() == 0))
            {
                throw new Exception("INTERNAL ERROR: No time series was returned for "
                    + identifier.buildStringToDisplayInTree() + ".");
            }
            if(trimmedTS.size() > 1)
            {
                throw new Exception("Check the PiService configuration: " + trimmedTS.size()
                    + " time series should have been acquired for " + identifier.buildStringToDisplayInTree()
                    + " when only 1 should have been.");
            }
            identifier.setCoordinates(trimmedTS.get(0).getHeader());
            TimeSeriesArraysTools.trimMissingValuesFromBeginningAndEndOfTimeSeries(trimmedTS);
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            LOG.error("Unable to load time series for location " + identifier.buildStringToDisplayInTree() + ": "
                + e.getMessage());
            final int option = JOptionPane.showConfirmDialog(getParentComponent(),
                                                             "Unable to load time series for "
                                                                 + identifier.buildStringToDisplayInTree()
                                                                 + "."
                                                                 + "\nContinue to next location? (Click No to stop exporting files.)",
                                                             "Error Exporting Historical Data!",
                                                             JOptionPane.YES_NO_OPTION,
                                                             JOptionPane.WARNING_MESSAGE);
            if(option != JOptionPane.YES_OPTION)
            {
                throw new InterruptedException("Step was cancelled due to problems encountered.");
            }
        }
        return trimmedTS;
    }

    private void exportTSFile(final TimeSeriesArrays ts, final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        final File exportFile = new File(_handler.buildFullPathNameForPIXMLFileFromFileName(identifier.getFiFilename()));
        LOG.info("Exporting time series file " + exportFile.getName() + " (" + exportFile.getParent() + ")...");
        updateNote("Exporting time series file " + exportFile.getName() + " (" + exportFile.getParent() + ")...");

        if((!ts.isEmpty()) && (!ts.get(0).isEmpty()))
        {
            try
            {
                TimeSeriesArraysTools.writeToFile(exportFile, ts);

                //Set the flag
//                if(!_handler.getIdentifiersWithData().contains(identifier))
//                {
//                    _wereAnyNewIdentifiersExported = true;
//                }
                _wereAnyNewFilesExported = true;
                _handler.addFileToRead(identifier.getFiFilename(), true);
            }
            catch(final IOException e)
            {
                e.printStackTrace();
                LOG.error("Unable to create time series file for " + identifier.buildStringToDisplayInTree() + ": "
                    + e.getMessage());
                final int option = JOptionPane.showConfirmDialog(getParentComponent(),
                                                                 "Unable to export a time series file for "
                                                                     + identifier.buildStringToDisplayInTree()
                                                                     + "."
                                                                     + "\nContinue to next location? (Click No to stop exporting files.)",
                                                                 "Error Exporting Historical Data!",
                                                                 JOptionPane.YES_NO_OPTION,
                                                                 JOptionPane.WARNING_MESSAGE);
                if(option != JOptionPane.YES_OPTION)
                {
                    throw new InterruptedException("Step was cancelled due to problems encountered.");
                }
            }
        }
        else
        {
            LOG.info("Time series for " + identifier.buildStringToDisplayInTree()
                + " is empty or all missing, and was not exported.");
            _emptyIdentifiers.add(identifier);
        }
    }

    /**
     * @param tsInfoList
     * @return {@link List} of {@link LocationAndDataTypeIdentifier} instances, one per time series header found.
     */
    private List<LocationAndDataTypeIdentifier> convertToListOfIdentifiers(final TimeSeriesHeaderInfoList tsInfoList)
    {
        final List<LocationAndDataTypeIdentifier> results = new ArrayList<LocationAndDataTypeIdentifier>();
        for(final TimeSeriesHeaderInfo info: tsInfoList)
        {
            results.add(LocationAndDataTypeIdentifier.get(info.getTimeSeriesHeader()));
        }
        return results;
    }

    @Override
    public void processJob()
    {
        //Part 1 -- get ts info list
        try
        {

            LOG.info("Loading time series from CHPS DB...");
            setIndeterminate(true);
            updateNote("Loading time series information from CHPS DB...");
            final TimeSeriesHeaderInfoList tsInfoList = this.retrieveTimeSeriesHeaderInfoList();
            if(tsInfoList == null)
            {
                throw new Exception("Unable to load list of time series list.");
//                return;
            }

            //Part 2 -- display a list of locations to select.
            updateNote("Requesting user selection of identifiers to export...");
            setJobMonitorComponentVisibility(false);
            obtainSelectedLocationsFromUser(tsInfoList);
            setJobMonitorComponentVisibility(true);
            if(_selectedIdentifiers == null) //Indicates the obtain method failed or was cancelled.
            {
                return;
            }

            //Part 3 -- Loading and exporting all time series
            LOG.info("Loading time series and exporting file for each location and data type...");
            setMaximumNumberOfSteps(2 * _selectedIdentifiers.size());
            setIndeterminate(false);
            updateNote("Loading time series and exporting file for each location and data type...");
            for(final LocationAndDataTypeIdentifier identifier: _selectedIdentifiers)
            {
                //Get the time series.
                LOG.info("Loading time series for " + identifier.buildStringToDisplayInTree() + "...");
                updateNote("Loading time series for " + identifier.buildStringToDisplayInTree() + "...");
                final TimeSeriesArrays ts = this.retrieveTimeSeriesFromCHPSDB(identifier);
                madeProgress();

                if(ts != null)
                {
                    exportTSFile(ts, identifier);
                    _handler.addFileToRead(identifier.getFiFilename(), true);
                }
                madeProgress();
            }
        }
        catch(final InterruptedException e)
        {
            LOG.error("Time series export was interrupted: " + e.getMessage()
                + "\nThe step may have been performed for some locations prior to failure.");
            fireProcessJobFailure(new Exception("Time series export was interrupted: " + e.getMessage()
                + "\nThe step may have been performed for some locations prior to failure."), true);
            return;

        }
        catch(final Exception e)
        {
            e.printStackTrace();
            LOG.error("Time series export failed: " + e.getMessage());
            fireProcessJobFailure(new Exception("Error occurred while exporting time series: " + e.getMessage()), true);
            return;
        }

        LOG.info("All time series have been exported.");
        endTask();
    }
}
