package ohd.hseb.hefs.pe.sources;

import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;

/**
 * Abstract implementation of ForecastSource, it provides a {@link SourceDataHandler} so that the {@link ForecastSource}
 * can record it any return only one source from the {@link #getSourceDataHandler()} method. It also overrids
 * {@link #hashCode()} and {@link #equals(Object)} so that two {@link ForecastSource}s are considered the same if their
 * {@link #getName()} is the same. <br>
 * <br>
 * Each {@link ForecastSource} should always return the same {@link SourceDataHandler} instance whenever
 * {@link #getSourceDataHandler()} is called (not one instance per call).<br>
 * <br>
 * Note that the {@link #hashCode()} method will return {@link String#hashCode()} for the{@link #getName()}. return
 * value of the {@link #getClass()} String. This means two source instances of the same class will have the same hash
 * code.
 * 
 * @author hank.herr
 */
public abstract class AbstractForecastSource implements ForecastSource
{

    private SourceDataHandler _handler;

    private String _sourceId = this.getClass().getSimpleName().replace("ForecastSource", "");

    @Override
    public void setForecastSourceId(final String sourceId)
    {
        _sourceId = sourceId;
    }

    @Override
    public String getSourceId()
    {
        return _sourceId;
    }

    protected void setSourceDataHandler(final SourceDataHandler handler)
    {
        _handler = handler;
    }

    @Override
    public SourceDataHandler getSourceDataHandler()
    {
        return _handler;
    }

    /**
     * @return null, implying no instructions beyond class and source id are needed to identify the forecast source. If
     *         other instructions are required, this must be overridden.
     */
    @Override
    public ForecastSourceDefinitionXMLHandler getSourceDefinitionHandler(final ParameterEstimatorRunInfo runInfo)
    {
        return null;
    }

    @Override
    public String toString()
    {
        return getSourceId();
    }

    @Override
    public int hashCode()
    {
        return (this.getClass().getName() + " -- " + getSourceId()).hashCode();
    }

    @Override
    public boolean equals(final Object obj)
    {
        return this.hashCode() == obj.hashCode();
    }
}
