package ohd.hseb.hefs.pe.sources;

import java.io.File;

import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.estimation.options.ControlOptionSupplier;
import ohd.hseb.hefs.pe.model.AlgorithmModelParameters;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.notify.NoticePoster;

import com.google.common.eventbus.EventBus;

/**
 * Top level class for any forecast source, an implementer supplies other components needed by the parameter estimator
 * framework. It extends the {@link ControlOptionSupplier} interface.<br>
 * <br>
 * Note that {@link #equals(Object)} and {@link #hashCode()} are included here so that it is obvious, when looking at an
 * instance of this, that those two methods have been specifically overridden, as they must be. When overriden
 * {@link #hashCode()}, be sure that two instances of the same source always map to the same hash code.
 * 
 * @author hank.herr
 */
public interface ForecastSource extends ControlOptionSupplier
{
    /**
     * By default, a source id should be set based on the prefix to the class name. However, for plug-in sources, it is
     * necessary to set the source id based on plug-in settings. This id should then be used in parameter file names and
     * estimation options stuff.
     */
    public void setForecastSourceId(String sourceId);

    /**
     * @return The value set by {@link #setForecastSourceId(String)}. It should be a unique identifier without any
     *         white-space for this source. It will be used in XML tags and parameter file names.
     */
    public String getSourceId();

    /**
     * @return Name of the data source to be displayed in some GUIs.
     */
    public String getName();

    /**
     * @param identifier Identifier for which to construct source model parameters.
     * @param algorithmParameters The {@link AlgorithmModelParameters} that gets passed down to the
     *            {@link SourceDataHandler} constructor.
     * @return An instance of SourceModelParameters which can handle reading/writing the header portion for binary file
     *         parameters for this data source.
     */
    public SourceModelParameters getSourceModelParameters(LocationAndDataTypeIdentifier identifier,
                                                          AlgorithmModelParameters algorithmParameters);

    /**
     * @return The {@link SourceDataHandler} instance that handles preparing and loading data files for the forecast
     *         source. Whenever this method is called, it must return the same instance! An implementer of this should
     *         remember only construct the handler one time!
     */
    public SourceDataHandler getSourceDataHandler();

    /**
     * Initializes the stored {@link SourceDataHandler}. This initialized handler should be returned by
     * {@link #getSourceDataHandler()} so that the same handler is always used.
     * 
     * @param baseDirectory The base directory for the data handler allowing it to identify its own directory containing
     *            data files.
     * @param poster An object that allows for posting to a central {@link EventBus}. A typical implementation to use is
     *            a {@link ParameterEstimatorRunInfo} which supplies an {@link EventBus} used at the core of a parameter
     *            estimator.
     * @throws An exception if a problem occurs initializing the handler. If a problem does occur, then this forecast
     *             source cannot be used in parameter estimation. How the application reacts to that is up to the
     *             application.
     */
    public void initializeSourceDataHandler(File baseDirectory, NoticePoster poster) throws Exception;

    /**
     * Must return an instance of {@link ForecastSourceDefinitionXMLHandler} or one of its subclasses to be used for
     * reading the source definition from an XML file. Null can be returned if there is no further XML needed to define
     * a forecast source, as will be true for default forecast sources; for example, MEFP RFC, GEFS, CFSv2, and
     * Historical.
     * 
     * @param runInfo The run-information for this PE, which may be useful in some cases depending on the
     *            implementation. Note that this will be null when called in order to read XML operationally. It will
     *            only be non-null if called as part of a PE. Hence, operationally, it is up to this source and/or the
     *            handler to compensate for null runtime information.
     * @return Null if the XML need not be read (most likely the case during an operational run) or an appropriate
     *         {@link ForecastSourceDefinitionXMLHandler} for XML reading when run as part of a PE initialization.
     */
    public ForecastSourceDefinitionXMLHandler getSourceDefinitionHandler(final ParameterEstimatorRunInfo runInfo);

    /**
     * @param runInfo The {@link ParameterEstimatorRunInfo} that is likely needed to initialize the step processors.
     * @return The {@link ParameterEstimatorStepProcessor} associated with this source.
     */
    public ParameterEstimatorStepProcessor getSourceStepProcessor(ParameterEstimatorRunInfo runInfo);

    @Override
    public boolean equals(Object o);

    @Override
    public int hashCode();

}
