package ohd.hseb.hefs.pe.sources;

import java.io.File;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import nl.wldelft.fews.common.config.GlobalProperties;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.utils.xml.GenericXMLReadingHandlerException;
import ohd.hseb.hefs.utils.xml.PropertiesProcessingXMLReadingHandler;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderAdapter;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLWriter;
import ohd.hseb.hefs.utils.xml.XMLWriterAdapter;
import ohd.hseb.hefs.utils.xml.XMLWriterException;
import ohd.hseb.hefs.utils.xml.vars.XMLString;

/**
 * Tools applicable to all forecast sources in general, regardless of application (MEFP or EnsPost).
 * 
 * @author hankherr
 */
public abstract class ForecastSourceTools
{
    private static final Logger LOG = LogManager.getLogger(ForecastSourceTools.class);

    /**
     * @return An instance of the provided class with the specified source id.
     * @throws Exception If the class is not an {@link MEFPForecastSource} or for any other reason cannot be
     *             instantiated.
     */
    public static ForecastSource instantiateSource(final String sourceClassName, final String sourceId) throws Exception
    {
        try
        {
            final ForecastSource source = (ForecastSource)Class.forName(sourceClassName).newInstance();
            if(sourceId != null)
            {
                source.setForecastSourceId(sourceId);
            }
            return source;
        }
        catch(final Exception e)
        {
            throw new Exception("Unable to instantiate source object for class " + sourceClassName + " with id "
                + sourceId + ".", e);
        }
    }

    /**
     * @param runInfo The run-time information passed through to
     *            {@link ForecastSource#getSourceDefinitionHandler(ParameterEstimatorRunInfo)}.
     * @param sources The sources to list in the {@link XMLWriter}.
     * @param writeFullDefinition This flag is passed through to {@link ForecastSourceWriter}. If true, then the full
     *            definition provided via {@link ForecastSource#getSourceDefinitionHandler()} will be written.
     *            Otherwise, only a partial XML will be written appropriate for parameter files and adapter runs that
     *            only need to know which sources were used not how they are defined within the parameter estimator.
     * @return {@link XMLWriter} that outputs the list with a tag of 'forecastSources'.
     */
    public static XMLWriter createSourcesWriter(final ParameterEstimatorRunInfo runInfo,
                                                final List<? extends ForecastSource> sources,
                                                final boolean writeFullDefinition)
    {
        return new XMLWriterAdapter("forecastSources")
        {
            @Override
            public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
            {
                final Element mainElement = request.createElement(getXMLTagName());
                for(final ForecastSource source: sources)
                {
                    mainElement.appendChild(new ForecastSourceWriter(runInfo, source, writeFullDefinition).writePropertyToXMLElement(request));
                }
                return mainElement;
            }
        };

    }

    /**
     * Makes use of a {@link PropertiesProcessingXMLReadingHandler} to read forecast source definition allowing for
     * {@link GlobalProperties} to be employed. This method should be called to read XML if it is known that the XML to
     * read was written with a full definition, including the preparation steps (see
     * {@link #createSourcesWriter(ParameterEstimatorRunInfo, List, boolean)} last argumetn).
     * 
     * @param runInfo Run-time information.
     * @param sourcesFile The file to read.
     * @param sources The list of sources to populate.
     * @throws GenericXMLReadingHandlerException
     */
    public static void readSourcesFromXML(final ParameterEstimatorRunInfo runInfo,
                                          final File sourcesFile,
                                          final List<? extends ForecastSource> sources) throws GenericXMLReadingHandlerException
    {
        final PropertiesProcessingXMLReadingHandler handler = new PropertiesProcessingXMLReadingHandler(createSourcesReader(runInfo,
                                                                                                                            sources))
        {
        };
        handler.readXMLFromFile(sourcesFile, null);
    }

    /**
     * Note that, unless this reader is used within a {@link PropertiesProcessingXMLReadingHandler},
     * {@link GlobalProperties} will not be processed. That is appropriate if the XML to read is not a full definition
     * (see last argument of {@link #createSourcesWriter(ParameterEstimatorRunInfo, List, boolean)}). However, if it
     * includes a full definition, including preparation steps,
     * {@link #readSourcesFromXML(ParameterEstimatorRunInfo, File, List)} should be used.
     * 
     * @param runInfo The run-time information passed through to
     *            {@link ForecastSource#getSourceDefinitionHandler(ParameterEstimatorRunInfo)}. This can be null!
     * @param sources {@link List} of {@link ForecastSource} subclass instances that will be populated as part of
     *            reading.
     * @return An {@link XMLReader} that can read a list of MEFP forecast sources.
     */
    public static ForecastSourcesReader createSourcesReader(final ParameterEstimatorRunInfo runInfo,
                                                            final List<? extends ForecastSource> sources)
    {
        return new ForecastSourcesReader(runInfo, sources);
    }

    /**
     * A writer used within {@link ForecastSourceTools#createSourceWriter(List)} for writing the definition of a source.
     * 
     * @author hankherr
     */
    public static class ForecastSourceWriter extends XMLWriterAdapter
    {
        private final ParameterEstimatorRunInfo _runInfo;
        private final ForecastSource _source;

        private final boolean _writeFullDefinition;

        /**
         * @param runInfo This information will be passed through to the
         *            {@link ForecastSource#getSourceDefinitionHandler(ParameterEstimatorRunInfo)} method, but only if
         *            writeFullDefinition is true. It can be null.
         * @param source The source to write information about.
         * @param writeFullDefinition If true, then the source definition handler will be called to add elements to the
         *            XML. Otherwise, only an id and class are included.
         */
        public ForecastSourceWriter(final ParameterEstimatorRunInfo runInfo,
                                    final ForecastSource source,
                                    final boolean writeFullDefinition)
        {
            super("source");
            _source = source;
            _runInfo = runInfo;
            _writeFullDefinition = writeFullDefinition;
        }

        @Override
        public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
        {
            final Element mainElement = request.createElement(getXMLTagName());

            mainElement.setAttribute("class", _source.getClass().getName());
            mainElement.setAttribute("id", _source.getSourceId());

            if(_writeFullDefinition)
            {
                final XMLWriter sourceDefWriter = _source.getSourceDefinitionHandler(_runInfo).getWriter();
                if(sourceDefWriter != null)
                {
                    mainElement.appendChild(sourceDefWriter.writePropertyToXMLElement(request));
                }
            }

            return mainElement;
        }

    }

    /**
     * A reader used within {@link ForecastSourceTools#createSourceReader(List)} for processing the definition of a
     * source.
     * 
     * @author hankherr
     */
    public static class ForecastSourceReader extends XMLReaderAdapter
    {
        private final ParameterEstimatorRunInfo _runInfo;
        private final List _sources;
        
        /**
         * This is set when a source tag is found.  Its initialized based on attributes.  When the end of that
         * source tag is found, this is set back to null to indicate there is no active source.
         */
        private ForecastSource _activeSource = null;
        

        /**
         * @param runInfo This information will be passed through to the
         *            {@link ForecastSource#getSourceDefinitionHandler(ParameterEstimatorRunInfo)} method and can be
         *            null.
         * @param sources {@link List} to populate.
         */
        public ForecastSourceReader(final ParameterEstimatorRunInfo runInfo, final List sources)
        {
            super("source");
            _sources = sources;
            _runInfo = runInfo;
        }

        @Override
        public XMLReader readInPropertyFromXMLElement(final String elementName, final Attributes attr) throws XMLReaderException
        {
            ForecastSourceDefinitionXMLHandler activeSourceHandler = null;
            if (_activeSource != null)
            {
                activeSourceHandler = _activeSource.getSourceDefinitionHandler(_runInfo);
            }
            
            if(elementName.equals(getXMLTagName()))
            {
                final XMLString className = new XMLString("class");
                final XMLString sourceId = new XMLString("id");

                //Class attribute
                final String classAttrValue = attr.getValue(className.getXMLTagName());
                if(classAttrValue == null)
                {
                    throw new XMLReaderException("Required attribute 'class' was not defined for element "
                        + getXMLTagName() + ".");
                }
                className.set(classAttrValue);

                //id attribute
                final String idAttrValue = attr.getValue(sourceId.getXMLTagName());
                if(idAttrValue == null)
                {
                    throw new XMLReaderException("Required attribute 'id' was not defined for element "
                        + getXMLTagName() + ".");
                }
                sourceId.set(idAttrValue);

                try
                {
                    _activeSource = instantiateSource(className.get(), sourceId.get());
                    _sources.add(_activeSource);
                    
                    //Ensure that all initialization within the source definition handler is prepped.
                    //I wanted to put this in PluginForecastSource, but I'm not sure it belong there.
                    activeSourceHandler = _activeSource.getSourceDefinitionHandler(_runInfo);
                }
                catch(final Exception e)
                {
                    e.printStackTrace();
                    LOG.warn("Ignoring source that cannot be instantiated with class name " + className.get()
                        + " and source id " + sourceId.get() + ": " + e.getMessage());
                }
            }
            
            //Only return the active handler if its XML tag is found.
            else if (elementName.equals(activeSourceHandler.getReader().getXMLTagName()))
            {
                return activeSourceHandler.getReader();
            }

            return null;
        }

        @Override
        public void finalizeReading() throws XMLReaderException
        {
            _activeSource = null;
        }
    }

    /**
     * Reader for a list of {@link ForecastSource} instances. It populates the list in place as it reads.
     * 
     * @author hankherr
     */
    public static class ForecastSourcesReader extends XMLReaderAdapter
    {
        private final ParameterEstimatorRunInfo _runInfo;
        private final List<? extends ForecastSource> _sources;

        public ForecastSourcesReader(final ParameterEstimatorRunInfo runInfo,
                                     final List<? extends ForecastSource> sources)
        {
            super("forecastSources");
            _sources = sources;
            _runInfo = runInfo;
        }

        @Override
        public final XMLReader readInPropertyFromXMLElement(final String elementName, final Attributes attr) throws XMLReaderException
        {
            final ForecastSourceReader reader = new ForecastSourceReader(_runInfo, _sources);
            if(elementName.equals(getXMLTagName()))
            {
                _sources.clear();
            }
            else if(elementName.equals(reader.getXMLTagName()))
            {
                return reader;
            }
            else
            {
                throw new XMLReaderException("While reading " + getXMLTagName()
                    + " element, invalid element with name " + elementName + " found.");
            }
            return null;
        }
    }

}
