package ohd.hseb.hefs.pe.sources;

import java.io.File;
import java.util.Collection;
import java.util.List;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.notify.Notifier;

/**
 * Framework specifies basic functionality all MEFP forecast source data handlers must provide.<br>
 * <br>
 * A call to {@link #loadOriginalTimeSeries(List)} or {@link #loadPreparedTimeSeries(List)} will load the appropriate
 * time series from whatever source is used. The observed time series loaded are acquired by
 * {@link #getLoadedObservedTimeSeries(LocationAndDataTypeIdentifier)} while the forecast time series are acquired via
 * {@link #getLoadedForecastTimeSeries(LocationAndDataTypeIdentifier)}. The returned time series are ready for use in
 * computations. Note that since pairing is performed between observed and forecast time series, it is important that
 * both are on the same cardinal time steps.
 * 
 * @author hank.herr
 */
public interface SourceDataHandler extends Notifier
{
    /**
     * Will be called at start up. It specifies the base directory used for all data handlers so that the data handler
     * implementation can navigate to the files containing the data which it is to read and the directories where the
     * prepared data files are to be written.
     * 
     * @param directory A directory on the file system usually ..._sa/Models/epp3pe.
     */
    public void setDataHandlerBaseDirectory(File directory);

    /**
     * @return The overall base directory for the data handler. The prepared directory is determined based on it.
     */
    public File getDataHandlerBaseDirectory();

    /**
     * @return The base directory for all files prepared by this data handler.
     */
    public File getPreparedDataFilesDirectory();

    /**
     * Converts an identifier to one that can be used by the particular implementation of this.
     * 
     * @param identifier The identifier to translate.
     * @return The translated identifier.
     */
    public LocationAndDataTypeIdentifier translateIdentifier(final LocationAndDataTypeIdentifier identifier);

    /**
     * @param identifier Identifier for which we want to identified the prepared files.
     * @return A list of all prepared data files, both observed and forecast, that should be generated by the data
     *         handler for the given identifier.
     */
    public List<File> generateListOfPreparedDataFiles(LocationAndDataTypeIdentifier translatedIdentifier);

    /**
     * Load the time series for the specified location and data type from the original/base file (not the prepared
     * files). Implementer must know where to go to acquire the data. Time series loaded are returned by
     * getLoaded*TimeSeries() methods. Loaded time series should be maintained in memory until the next load method call
     * is made.
     * 
     * @param identifiers List of translated identifiers.
     */
    public void loadOriginalTimeSeries(List<LocationAndDataTypeIdentifier> translatedIdentifiers) throws Exception;

    /**
     * Load the time series for the specified location and data type from the prepared files. Implementer must know
     * where to go to acquire the data. Time series loaded are returned by getLoaded*TimeSeries() methods. Loaded time
     * series should be maintained in memory until the next load method call is made.
     * 
     * @param locationAndDataType
     */
    public void loadPreparedTimeSeries(List<LocationAndDataTypeIdentifier> translatedIdentifiers) throws Exception;

    /**
     * Clear the time series that have been loaded, usually in preparation for loading the next batch.
     */
    public void clearLoadedTimeSeries();

    /**
     * @return A list of LocationAndDataTypeIdentifier instance identifying the time series most recently loaded.
     */
    public Collection<LocationAndDataTypeIdentifier> getIdentifiersWithData();

    /**
     * Perform data preparation required of the data handler for those time series which have been loaded via
     * loadTimeSeries. Most likely, this method will be called multiple times for small portions of loaded time series
     * in order to conserve memory.
     */
    public void prepareDataFiles() throws Exception;

    /**
     * @param identifier Location and data type for which data is desired.
     * @return A collection of TimeSeriesArray instances that specify the loaded observed time series. The returned
     *         collection of time series will be handled via the TimeSeriesSorter class in order to identify the data
     *         type of each series and pair them with forecasts, so pay close attention to the parameter ids assumed.
     */
    public Collection<TimeSeriesArray> getLoadedObservedTimeSeries(LocationAndDataTypeIdentifier translatedIdentifier);

    /**
     * @param identifier Location and data type for which data is desired.
     * @return A collection of TimeSeriesArray instances that specify the loaded forecast time series. The returned
     *         collection of time series will be handled via the TimeSeriesSorter class in order to identify the data
     *         type of each series and pair them with observations, so pay close attention to the parameter ids assumed.
     */
    public Collection<TimeSeriesArray> getLoadedForecastTimeSeries(LocationAndDataTypeIdentifier translatedIdentifier);

    /**
     * @return All observed time series loaded via loadTimeSeries.
     */
    public Collection<TimeSeriesArray> getAllLoadedObservedTimeSeries();

    /**
     * @return All forecast time series loaded via loadTimeSeries.
     */
    public Collection<TimeSeriesArray> getAllLoadedForecastTimeSeries();

    /**
     * Perform any initialization necessary for the data handler.
     */
    public void initialize() throws Exception;

    /**
     * @param identifier Identifier to check.
     * @return True if a prepared data file exists.
     */
    public boolean havePreparedDataFilesBeenCreatedAlready(LocationAndDataTypeIdentifier translatedIdentifier);

    /**
     * @param identifier Identifier to check.
     * @return True if the prepared data file is up-to-date relative to the source (original) data file.
     */
    public boolean arePreparedDataFilesUpToDate(LocationAndDataTypeIdentifier translatedIdentifier);
}
