package ohd.hseb.hefs.pe.sources;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;

import ohd.hseb.hefs.pe.estimation.DefaultModelParameterDiagnosticDisplay;
import ohd.hseb.hefs.pe.estimation.ParameterSummaryPanel;
import ohd.hseb.hefs.pe.estimation.ParametersFoundTableModel;
import ohd.hseb.hefs.pe.model.AlgorithmModelParameters;
import ohd.hseb.hefs.pe.model.ModelParameterType;
import ohd.hseb.hefs.pe.model.OneSetParameterValues;
import ohd.hseb.hefs.pe.model.OneTypeParameterValues;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.xml.CompositeXMLReader;
import ohd.hseb.hefs.utils.xml.CompositeXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReadable;
import ohd.hseb.hefs.utils.xml.XMLWritable;

import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;

/**
 * Abstract class records the parameters in a map from {@link ModelParameterType} to {@link OneTypeParameterValues}. It
 * also records the SourceDataHandler associated with the parameters, along with the identifier and algorithm model
 * parameters.<br>
 * <br>
 * Some methods are meant to provide information to the generic {@link ParameterSummaryPanel} and
 * {@link ParametersFoundTableModel}. To do so, the SourceModelParameters assumes that the underlying models record
 * parameters by day of year (which should be easy to map to seasons, if used) and possibly multiple instances per day
 * (e.g., MEFP stores parameters for canonical events). In the later case, the multiple instances are displayed as
 * ensemble members in a time series dialog, so the method here provide information about the number of members to
 * display and the column name in the ParametersFoundTableModel for the ensemble member size as well as the day-of-year
 * index. <br>
 * <br>
 * Along with everything else dealing with model parameters, this may need to be cleaned up once we store the parameters
 * in XML.
 * 
 * @author hank.herr
 */
public abstract class SourceModelParameters implements XMLReadable, XMLWritable
{
    /**
     * This must be known by the time parameters are being written or read. This is neither written to nor read from
     * XML.
     */
    private LocationAndDataTypeIdentifier _identifier;

    /**
     * This must be specified by the time parameters are being written or read. This is neither written to nor read from
     * XML.
     */
    private final ForecastSource _forecastSource;

    /**
     * By the time this {@link SourceModelParameters} is being written or read,
     * {@link SourceModelParameters#_algorithmModelParameters} should have already been written or red and can serve as
     * a resource, if needed, for the {@link SourceModelParameters} to be written or read. This is neither written to
     * nor read from XML.
     */
    private AlgorithmModelParameters _algorithmModelParameters;

    /**
     * Records the parameter values in a map of {@link ModelParameterType} to {@link OneTypeParameterValues}. The order
     * of the map MUST be kept in synch with the order of parameters specified in _algorithmModelParameters. This is
     * what needs to be written to and read from XML.
     */
    private LinkedHashMap<ModelParameterType, OneTypeParameterValues> _parameterTypeToValuesMap = null;

    /**
     * A list of the days of the year for which to read parameters. This allows for streamlining reading, as long as the
     * {@link OneTypeParameterValues} used by the subclass properly implement the days-of-the-year features.
     */
    private final List<Integer> _daysOfTheYearForWhichToReadWriteParameters = new ArrayList<Integer>();

    /////////////////////////////////////

    public SourceModelParameters(final LocationAndDataTypeIdentifier identifier,
                                 final ForecastSource source,
                                 final AlgorithmModelParameters algorithmParameters)
    {
        _identifier = identifier;
        _forecastSource = source;
        _algorithmModelParameters = algorithmParameters;
        _parameterTypeToValuesMap = new LinkedHashMap<ModelParameterType, OneTypeParameterValues>();
    }

    /**
     * Override if you need to perform additional validation. By default, it validates the returns of getParameterValues
     * by using the method in ModelParameterType.
     * 
     * @throws Exception If validation fails, with the message providing the reason.
     */
    public void validateParameters() throws Exception
    {
        for(final ModelParameterType type: getModelParameterTypesWithStoredValues())
        {
            type.validateParameters(getParameterValues(type));
        }
    }

    /**
     * @return False if {@link #_parameterTypeToValuesMap} was never initialized, implying that the source has not been
     *         initialized for use.
     */
    public boolean wasParameterStorageInitializedForSource()
    {
        return !_parameterTypeToValuesMap.isEmpty();
    }

    /**
     * Override as appropriate. If this method is not important for an application, leave it untouched. By default, it
     * returns the value returned by {@link #wasParameterStorageInitializedForSource()}.
     * 
     * @return True if parameters were estimated for the source.
     */
    public boolean wereParametersEstimatedForSource()
    {
        return wasParameterStorageInitializedForSource();
    }

    /**
     * @param dayOfYear Starts counting at 1.
     * @return True if all parameters are missing, making use of
     *         {@link OneTypeParameterValues#areAllParametersMissingForDay(int)}. False if any are not missing.
     */
    public boolean areAllParametersMissing(final int dayOfYear)
    {
        for(final ModelParameterType type: this.getModelParameterTypesWithStoredValues())
        {
            if(!this.getParameterValues(type).areAllParametersMissingForDay(dayOfYear))
            {
                return false;
            }
        }
        return true;
    }

    /**
     * Updates the stored values to reflect those in the provided {@link OneSetParameterValues}. Note that any
     * {@link ModelParameterType} instances specified by the provided values will only be recorded if a corresponding
     * instance is used by this. For example, if this set of parameters only uses observed {@link ModelParameterType}s,
     * then forecast values will not be recorded.
     * 
     * @param values Values for a single day and value of the secondary index. The values
     *            {@link #getDaysOfTheYearForWhichToReadWriteParameters()} must correspond to the date for which to
     *            record the values.
     */
    public void recordValuesFrom(final OneSetParameterValues values)
    {
        for(final ModelParameterType type: values.getParameterTypes())
        {
            final OneTypeParameterValues tmp = _parameterTypeToValuesMap.get(type);
            if(tmp != null)
            {
                tmp.recordValueFrom(values);
            }
        }
    }

    /**
     * Populates the given {@link OneSetParameterValues} with values from the map stored here. The values are extracted
     * based on {@link OneSetParameterValues#getDayOfYearIndex()} and {@link OneSetParameterValues#getSecondaryIndex()}.
     * It is possible for the provided {@link OneSetParameterValues} to allow for more parameter types to be stored than
     * is required here. In those cases, the parameter value will be set to {@link Double#NaN} in the provided values.<br>
     * <br>
     * The parameters for days for which parameters were not estimated are nearest neighbor, based on feedback from
     * Limin. This is because EPT parameters cannot be linearly interpolated (there are distribution shape/scale parms
     * within it for which linear interpolation may not be a good idea), so we use nearest neighbor to be consistent.
     * 
     * @param values {@link OneSetParameterValues} instance specifying the day of year and secondary index.
     * @param nearestComputationDay The nearest computation day, that for which the parameters are stored.
     */
    public void putValuesInto(final OneSetParameterValues values, final int nearestComputationalDay)
    {
//XXX all code commented out below implements either debug prints -or- linear interpolation to determine parameters.  The algorithm below currently
//uses nearest neighbor, but that is passed in as argument nearestComputationalDay, since FullModelParameters is able to compute this value, not
//SourceModelParameters.

//        final List<Integer> boundingDays = Lists.newArrayList(MEFPTools.determineDaysForWhichToReadParameters(getAlgorithmModelParameters(),
//                                                                                                              Lists.newArrayList(values.getDayOfYearIndex())));
//        int dayToUseInLinearIterpolation = boundingDays.get(1);
//        if(dayToUseInLinearIterpolation == 1)
//        {
//            dayToUseInLinearIterpolation = 366;
//        }
//        final int nearestComputationalDay = _algorithmModelParameters.findNearestComputationalDay(values.getDayOfYearIndex());
//
//        System.err.println("####>> IN HERE --- " + boundingDays.get(0) + ", " + dayToUseInLinearIterpolation + ", "
//            + values.getDayOfYearIndex());

        //For each parameter type specified in values...
        for(final ModelParameterType type: values.getParameterTypes())
        {
            final OneTypeParameterValues tmp = _parameterTypeToValuesMap.get(type);

            //If no corresponding OneTypeParameterValues can be acquired, just specify it as NaN.
            if(tmp == null)
            {
                values.setValue(type, Double.NaN);
            }
            //Otherwise, acquire the values from the bounding days and linear interpolate the desired value.
            else
            {
                //Try to get the desired value.
                double desiredValue = tmp.getValue(values.getDayOfYearIndex(), values.getSecondaryIndex());

                //If the value is not found, meaning parameters are only stored for computation days, then...
                //(Fortran code would store the parameters for all days, even those that were interpolated)
                if(Double.isNaN(desiredValue))
                {
                    desiredValue = tmp.getValue(nearestComputationalDay, values.getSecondaryIndex());

//                    final double firstValue = tmp.getValue(boundingDays.get(0), values.getSecondaryIndex());
//                    final double secondValue = tmp.getValue(boundingDays.get(1), values.getSecondaryIndex());
//
//                    //Linear interpolation is used here.  
//                    desiredValue = NumberTools.linearlyInterpolate(boundingDays.get(0),
//                                                                   firstValue,
//                                                                   dayToUseInLinearIterpolation,
//                                                                   secondValue,
//                                                                   values.getDayOfYearIndex());
                }

                values.setValue(type, desiredValue);

//                System.err.println("####>> " + i + " SET VALUE TO "
//                    + values.getValue(values.getParameterTypes().get(i)));
            }
        }
    }

    /**
     * Add a day-of-the-year to read/write parameters.
     * 
     * @param dayOfYear Counting starts at 1.
     */
    public void addDayOfYearToReadWriteParameters(final int dayOfYear)
    {
        _daysOfTheYearForWhichToReadWriteParameters.add(dayOfYear);
    }

    /**
     * Clear the days of the year.
     */
    public void clearDaysOfYearToReadWriteParameters()
    {
        _daysOfTheYearForWhichToReadWriteParameters.clear();
    }

    /**
     * Add the days specified.
     * 
     * @param daysOfYear Counting starts at 1.
     */
    public void addAllDaysOfYearToReadWrite(final Collection<Integer> daysOfYear)
    {
        _daysOfTheYearForWhichToReadWriteParameters.addAll(daysOfYear);
    }

    /**
     * @return {@link List} of days of the year for which to read/write parameters. Useful if the subclass needs to do
     *         its own reading/writing.
     */
    public List<Integer> getDaysOfTheYearForWhichToReadWriteParameters()
    {
        return _daysOfTheYearForWhichToReadWriteParameters;
    }

    public LocationAndDataTypeIdentifier getIdentifier()
    {
        return _identifier;
    }

    public void setIdentifier(final LocationAndDataTypeIdentifier identifier)
    {
        _identifier = identifier;
    }

    protected void setParameterTypeToValuesMap(final LinkedHashMap<ModelParameterType, OneTypeParameterValues> map)
    {
        this._parameterTypeToValuesMap = map;
    }

    protected void addParameterValues(final ModelParameterType type, final OneTypeParameterValues values)
    {
        this._parameterTypeToValuesMap.put(type, values);
    }

    public void setAlgorithmModelParameters(final AlgorithmModelParameters parameters)
    {
        _algorithmModelParameters = parameters;
    }

    protected AlgorithmModelParameters getAlgorithmModelParameters()
    {
        return _algorithmModelParameters;
    }

    public ForecastSource getForecastSource()
    {
        return _forecastSource;
    }

    public OneTypeParameterValues getParameterValues(final ModelParameterType type)
    {
        return this._parameterTypeToValuesMap.get(type);
    }

    public List<ModelParameterType> getModelParameterTypesWithStoredValues()
    {
        final List<ModelParameterType> types = new ArrayList<ModelParameterType>();
        if(this._parameterTypeToValuesMap != null)
        {
            types.addAll(this._parameterTypeToValuesMap.keySet());
        }
        return types;
    }

    /**
     * Override if getNumberOfEnsembleMembersForDiagnosticDisplay() returns a value greater than 1.
     * 
     * @param memberIndex
     * @return
     */
    public String getLegendEntryForEnsembleMember(final int memberIndex)
    {
        return null;
    }

    /**
     * @return Corresponding to the getValue method, the number of days of year with values.
     */
    public int getNumberOfDaysOfYearWithValues()
    {
        if(!_parameterTypeToValuesMap.isEmpty())
        {
            final ModelParameterType oneType = new ArrayList<ModelParameterType>(_parameterTypeToValuesMap.keySet()).get(0);
            return _parameterTypeToValuesMap.get(oneType).getNumberOfDaysOfYearWithValues();
        }
        return 0;
    }

    /**
     * Override if you need to display more than one parameter value per day of year. Example: MEFP displays one parm
     * value for canonical event per day of year.
     * 
     * @return 1 by default. The number of ensemble members to display on a diagnostic chart.
     */
    public int getNumberOfEnsembleMembersForDiagnosticDisplay()
    {
        return 1;
    }

    /**
     * By default, null is returned. By returning null, the table model will not display the column.
     * 
     * @return The column name to use in the {@link ParameterSummaryPanel} ParametersFoundTable corresponding to the
     *         ensemble size for displaying the collected values.
     */
    public String getColumnNameForEnsembleSizeIndex()

    {
        return null;
    }

    /**
     * @return The column name to use in the ParameterSummaryPanel ParametersFoundTable corresponding to the
     *         days-of-year indexing for the collected values.
     */
    public abstract String getColumnNameForDaysOfYearIndex();

    /**
     * Override if you do not want to use the default title. This title will appear in a plot of the parameters created
     * by {@link DefaultModelParameterDiagnosticDisplay}.
     * 
     * @return Title to place for the legend, or null if the default is used.
     */
    public abstract String getLegendTitle();

    /**
     * @return The XML tag name assigned to XML readers and writers of this.
     */
    public String getXMLTagName()
    {
        return "sourceModelParameters";
    }

    /**
     * @return True if the entry file is the name of the first entry for a source.
     */
    public String buildFirstSourceParametersFileEntryName()
    {
        throw new IllegalStateException("The default behavior of " + getClass().getSimpleName()
            + " does not allow for reading from tar archives.");
    }

    /**
     * Override to allow for source model parameters subclass to have its parameter written to a tar archive.
     * 
     * @param outputStream Stream to which to write the tar entries for these parameters.
     * @param performanceMode True if the tar ball should be written with speed and size in mind; for example
     *            fastInfoset files should be written to the tar-ball; false indicates testing, which means all files
     *            should be human readable XML.
     * @throws Exception
     */
    public void writeToTarArchive(final TarArchiveOutputStream outputStream, final boolean performanceMode) throws Exception
    {
        //Do nothing by default.
        throw new IllegalStateException("The default behavior of " + getClass().getSimpleName()
            + " does not allow for writing to tar archives.");
    }

    /**
     * Override to allow for source model parameters subclass to have its parameters read from a tar archive. Note that
     * the nature of the {@link TarArchiveInputStream} software forces this to assume that the files are in specific
     * order Hence, when reading from the stream, the order of files read must match exactly the order in which they
     * were written (see {@link #writeToTarArchive(TarArchiveOutputStream, boolean)} and make sure the order matches
     * when coding a reader). <br>
     * <br>
     * By default, this method throws an exception indicating it has not been overridden so that tar archives cannot be
     * read from.
     * 
     * @param file File from which to read parameters. Since the tar archive may be closed after each use (see above),
     *            this file allows for opening and searching for the entry to read for each file in the tar.
     * @param operationalRead If true, then the parameters are being read for the purposes of an operational run. If
     *            false, parameters are being read during parameter estimation and the read should be a full one.
     * @param skipSourceRead If true, then the overridden method can skip the files to read, since parameters need not
     *            be stored. However, it must skip an appropriate number of entries in the tar stream.
     * @throws Exception
     */
    public void readFromTarArchive(final TarArchiveInputStream tarStream,
                                   final boolean operationalRead,
                                   final boolean skipSourceRead) throws Exception
    {
        //Do nothing by default.
        throw new IllegalStateException("The default behavior of " + getClass().getSimpleName()
            + " does not allow for reading from tar archives.");
    }

    /**
     * @return A {@link CompositeXMLWriter} that can be used to write a complete, all encompassing XML, specifying
     *         everything for this source model parameters instance. The attribute
     *         {@link #_daysOfTheYearForWhichToReadWriteParameters} specifies the days of the year for which to write
     *         parameter values, with that information being passed along to the {@link OneTypeParameterValues} objects
     *         being written. Note that the {@link OneTypeParameterValues#clearDaysOfYearToReadWriteParameters()} method
     *         is called for each of the {@link OneTypeParameterValues} objects even if no day restriction is provided
     *         (the clearing without adding tells it to write all days).
     */
    @Override
    public CompositeXMLWriter getWriter()
    {
        final List<OneTypeParameterValues> writeList = new ArrayList<OneTypeParameterValues>(_parameterTypeToValuesMap.values());
        for(final OneTypeParameterValues values: writeList)
        {
            values.clearDaysOfYearToReadWriteParameters();
            for(int i = 0; i < _daysOfTheYearForWhichToReadWriteParameters.size(); i++)
            {
                values.addDayOfYearToReadWriteParameters(_daysOfTheYearForWhichToReadWriteParameters.get(i));
            }

        }
        return new CompositeXMLWriter(getXMLTagName(), writeList);
    }

    /**
     * @return A {@link CompositeXMLReader} that can be used to read a complete, all encompassing XML file, specifying
     *         everything for this source model parameters instance. The attribute
     *         {@link #_daysOfTheYearForWhichToReadWriteParameters} specifies the days of the year for which to write
     *         parameter values, with that information being passed along to the {@link OneTypeParameterValues} objects
     *         being written. Note that the {@link OneTypeParameterValues#clearDaysOfYearToReadWriteParameters()} method
     *         is called for each of the {@link OneTypeParameterValues} objects even if no day restriction is provided
     *         (the clearing without adding tells it to read all days).
     */
    @Override
    public CompositeXMLReader getReader()
    {
        //By this point, the map must already be initialized to contain a value for every parameter type. 
        //Note that there may be more parameters in the map than are in the XML.  So it is critical that the parameters
        //are at the end, because the CompositeXMLReader is willing to allow some parameters to not be set, but it is not
        //willing to read in parameters out of order.  That may happen if the parameters are in the middle of the list
        //but not all of them are in the XML.
        final List<OneTypeParameterValues> readList = new ArrayList<OneTypeParameterValues>(_parameterTypeToValuesMap.values());
        for(final OneTypeParameterValues values: readList)
        {
            values.clearDaysOfYearToReadWriteParameters();
            for(int i = 0; i < _daysOfTheYearForWhichToReadWriteParameters.size(); i++)
            {
                values.addDayOfYearToReadWriteParameters(_daysOfTheYearForWhichToReadWriteParameters.get(i));
            }

        }
        return new CompositeXMLReader(getXMLTagName(), true, readList);
    }
}
