package ohd.hseb.hefs.pe.sources.pixml;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import nl.wldelft.util.FileUtils;
import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import nl.wldelft.util.timeseries.TimeSeriesHeader;
import ohd.hseb.hefs.pe.sources.SourceDataHandler;
import ohd.hseb.hefs.pe.tools.HEFSTools;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifierList;
import ohd.hseb.hefs.utils.notify.Notice;
import ohd.hseb.hefs.utils.notify.NoticePoster;
import ohd.hseb.hefs.utils.notify.NotifierBase;
import ohd.hseb.hefs.utils.tools.IterableTools;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesHeaderInfo;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesHeaderInfoList;
import ohd.hseb.hefs.utils.xml.XMLTools;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.HashBasedTable;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Table;

/**
 * Load time series from PI-xml time series output. The subclass specifies the directory that stores the PI-xml files
 * underneath of the standard PE baseDirectory for the application using this class. The found files provide time series
 * that are sorted based on forecast/observed data type, according to HEFSTools. This means that the time series found
 * must be for data types (parameterIds) HEFSTools can understand (see that class for more info).However, note that
 * there is an overridable translation method, {@link #translateExternalParameter(String)}, that allows for converting
 * external parameters that are known to be used but invalid into something that can be used internally.
 * 
 * @author hank.herr
 */
public abstract class GenericPIXMLDataHandler extends NotifierBase implements SourceDataHandler
{

    private static final Logger LOG = LogManager.getLogger(GenericPIXMLDataHandler.class);

    /**
     * This stores a map of the translated identifier to the original identifiers that were translated to it.
     */
    private final HashMultimap<LocationAndDataTypeIdentifier, LocationAndDataTypeIdentifier> _translatedIdentifierToOriginalMap = HashMultimap.create();

    /**
     * This table records which time series header was loaded from the given source for the given identifier. The
     * identifier cannot be the translated identifier; it must be the original.
     */
    private final Table<Object, LocationAndDataTypeIdentifier, TimeSeriesHeaderInfo> _tsInfoTable = HashBasedTable.create();

    /**
     * This mapping stores all time series headers loaded for the given translated identifier. The headers are loaded
     * from the sources specified in {@link #_usedSources}. This must be kept in sync with {@link #_usedSources}.
     */
    private final ArrayListMultimap<LocationAndDataTypeIdentifier, TimeSeriesHeaderInfo> _translatedIdentifierToHeadersMap = ArrayListMultimap.create();

    /**
     * The identifier in this map is NOT translated. This points to the source currently being used for the time series
     * and must be kept in sync with {@link #_translatedIdentifierToHeadersMap}.
     */
    private final Map<LocationAndDataTypeIdentifier, Object> _usedSources = Maps.newHashMap();

    /**
     * Stores loaded time series, allowing for multiple per translated identifier.
     */
    private final ArrayListMultimap<LocationAndDataTypeIdentifier, TimeSeriesArray> _translatedIdentifierToTSMap = ArrayListMultimap.create();

    /**
     * Stores loaded time series by the original (not translated) identifier associated with the time series.
     */
    private final Map<LocationAndDataTypeIdentifier, TimeSeriesArray> _originalIdentifeirToTSMap = Maps.newHashMap();

    /**
     * List of identifiers for which enough data was found for use and the parameter id indicates a forecast time
     * series. These are translated identifiers.
     */
    private final List<LocationAndDataTypeIdentifier> _forecastIdentifiersForWhichReqdDataWasFound = new ArrayList<LocationAndDataTypeIdentifier>();

    /**
     * List of identifiers for which enough data was found and the parameter id indicates a observed time series. These
     * are translated identifiers.
     */
    private final List<LocationAndDataTypeIdentifier> _observedIdentifiersForWhichReqdDataWasFound = new ArrayList<LocationAndDataTypeIdentifier>();

    private File _dataHandlerBaseDirectory = null;

    /**
     * Set based on the parameters to the constructor.
     */
    private final String _piXMLSubDirName;

    /**
     * Based on the _dataHandlerBaseDirectory + _piXMLSubDirName.
     */
    private File _piXMLDataDirectory = null;

    /**
     * Stores the observed time series contained within {@link #_translatedIdentifierToTSMap}. This must be kept in
     * synch with {@link #_translatedIdentifierToTSMap}, which can be done by called {@link #putTimeSeriesIntoLists()}
     * AFTER populating the other map.
     */
    private final List<TimeSeriesArray> _listOfObservedTimeSeries = new ArrayList<TimeSeriesArray>();

    /**
     * Stores the forecast time series contained within {@link #_translatedIdentifierToTSMap}. This must be kept in
     * synch with {@link #_translatedIdentifierToTSMap}, which can be done by called {@link #putTimeSeriesIntoLists()}
     * AFTER populating the other map.
     */
    private final List<TimeSeriesArray> _listOfForecastTimeSeries = new ArrayList<TimeSeriesArray>();

    private final List<String> _piXMLFileNamesRead = new ArrayList<String>();
    private final List<String> _piXMLFileNamesToRead = new ArrayList<String>();

    private final NoticePoster _noticePoster;

    /**
     * Sets the base directory and then calls addAllFilesInHistoricalDataDirectoryToUnreadList() followed by
     * initialize(). The result is an instance that has been initialized based on files found in the historical file
     * directory.
     * 
     * @param baseDirectory
     * @throws Exception
     */
    public GenericPIXMLDataHandler(final File baseDirectory,
                                   final String piXMLSubDirectoryName,
                                   final NoticePoster noticePoster) throws Exception
    {
        _piXMLSubDirName = piXMLSubDirectoryName;
        setDataHandlerBaseDirectory(baseDirectory);
        _noticePoster = noticePoster;
        initialize();
    }

    /**
     * Posts events via the {@link #_noticePoster}.
     * 
     * @param event
     */
    protected void post(final Notice event)
    {
        if(_noticePoster != null)
        {
            _noticePoster.post(event);
        }
    }

    /**
     * Initialize list of files to read to be those in the historical data directory (see setDataHandlerBaseDirectory).
     */
    public void addAllFilesInHistoricalDataDirectoryToUnreadList() throws Exception
    {
        final List<File> listOfFiles = Arrays.asList(_piXMLDataDirectory.listFiles());
        for(final File file: listOfFiles)
        {
            this.addFileToRead(file.getName(), true);
        }
    }

    /**
     * Override as needed. Currently, this returns what it is provided. This method is called to translate the
     * parameterId of any read in time series header immediately after reading. It can be used to convert a normally
     * unrecognizable external parameter into something reasonable internally. For example, TAMN/TAMX can be translated
     * to TMIN/TMAX
     * 
     * @param parameter The {@link ParameterId} that must be translated from what is read in to the id used internally.
     * @return The appropriate internal id to use.
     */
    public String translateExternalParameter(final String parameterId)
    {
        return parameterId;
    }

    /**
     * Called as soon as any time series header info is read in from the historical data. This allows for parameters
     * that are not usually allowed to be translated into internal parameters that are understood. For example,
     * TAMN/TAMX can be translated to TMIN/TMAX here.
     * 
     * @param header The header in which to perform the in-place translation.
     */
    private void translateHeaderExternalParameterId(final TimeSeriesHeader header)
    {
        ((DefaultTimeSeriesHeader)header).setParameterId(translateExternalParameter(header.getParameterId()));
    }

    /**
     * Reads all files in the list of unread files and adds appropriate sources via
     * {@link #addToSourceIdentifierTimeSeriesInformationMaps(Object, LocationAndDataTypeIdentifier, TimeSeriesHeaderInfo)}
     * .
     * 
     * @throws Exception If a problem occurs.
     */
    public void readUnreadHistoricalFilesAndAddToLists() throws Exception
    {
        //Check for existence of directory.
        if(!_piXMLDataDirectory.exists())
        {
            LOG.warn("PI-xml data directory does not exist: " + _piXMLDataDirectory.getAbsolutePath() + ".");
            LOG.warn("Any files specified without a complete path (i.e., starting with '/') will not be read.");
        }

        //For each file in the directory
        for(final String filename: this._piXMLFileNamesToRead)
        {
            //Build the full-path file name.  If the returned filename is not an absolute path, skip the file.
            //This means that the original file name was relative but the historical data directory is not specified.
            final String fullPathName = this.buildFullPathNameForPIXMLFileFromFileName(filename);
            if(!new File(fullPathName).isAbsolute())
            {
                continue;
            }

            //Read the file
            try
            {
                //Use this if you want to throw out the time series after loading.
                //Parse time series from the file
                LOG.info("Loading time series header information from file " + fullPathName + "...");
                final TimeSeriesHeaderInfoList tsInfoList = new TimeSeriesHeaderInfoList();
                XMLTools.readXMLFromFile(new File(fullPathName), tsInfoList);

                //Add to the historical files read.  If I reached this point, the file was of valid format
                //and should be remembered.
                ListTools.addItemIfNotAlreadyInList(_piXMLFileNamesRead, fullPathName);

                //PiTimeSeriesReader reader = new PiTimeSeriesReader(fullPathName);
                //listOfTimeSeries = reader.read();
                LOG.info("Done loading.  Loaded " + tsInfoList.size() + " time series headers.");

                //Add information to the lists and map.
                for(int i = 0; i < tsInfoList.size(); i++)
                {
                    translateHeaderExternalParameterId(tsInfoList.get(i).getTimeSeriesHeader());
                    final LocationAndDataTypeIdentifier id = LocationAndDataTypeIdentifier.get(tsInfoList.get(i)
                                                                                                         .getTimeSeriesHeader());
                    if(includeTimeSeries(id))
                    {
                        addToSourceIdentifierTimeSeriesInformationMaps(fullPathName, id, tsInfoList.get(i));
                    }
                }

                //XXX This code dumps the read in time series to files with the specified names in writeToFile.  This is for debug.
//                final TimeSeriesArrays tss = TimeSeriesArraysTools.readFromFile(new File(fullPathName));
//                for(int i = 0; i < tss.size(); i++)
//                {
//                    TimeSeriesArray ts = tss.get(i);
//                    ts = TimeSeriesArrayTools.trimMissingValuesFromBeginningAndEndOfTimeSeries(ts);
//                    ((DefaultTimeSeriesHeader)ts.getHeader()).setLocationId(ts.getHeader()
//                                                                              .getLocationId()
//                                                                              .toUpperCase());
//                    TimeSeriesArraysTools.writeToFile(new File("/Users/hankherr/matTimeSeries/"
//                        + ts.getHeader().getLocationId() + "." + ts.getHeader().getParameterId() + ".xml"), ts);
//                }

                LOG.info("Done recording time series.");
            }
            catch(final Exception e)
            {
                //e.printStackTrace();
                LOG.warn("Unable to parse time series in file " + fullPathName + ".  Message: '" + e.getMessage()
                    + "'.  Skipping...");
            }
        }

        _piXMLFileNamesToRead.clear();
    }

    /**
     * Adds to {@link #_piXMLFileNamesToRead} unless the file has already been read.
     */
    public void addFileToRead(final String filename, final boolean forceReRead)
    {
        if(!hasFileBeenRead(filename) || forceReRead)
        {
            _piXMLFileNamesToRead.add(filename);
        }
    }

    /**
     * Checks to see if {@link #buildFullPathNameForPIXMLFileFromFileName(String)} given the file name indicates a file
     * within {@link #_piXMLFileNamesRead}.
     * 
     * @param filename
     * @return
     */
    public boolean hasFileBeenRead(String filename)
    {
        filename = buildFullPathNameForPIXMLFileFromFileName(filename);
        return _piXMLFileNamesRead.contains(filename);
    }

    /**
     * It removes the source,removes the file name from {@link #_piXMLFileNamesRead} and {@link #_piXMLFileNamesToRead},
     * and removes the file from the file system.
     * 
     * @param filename
     */
    public void deleteFileRead(final String filename)
    {
        //This must match the conditions in the read method for ignoring or adding a directory to a file.
        final String fullPathName = buildFullPathNameForPIXMLFileFromFileName(filename);

        //Construct the source.
        removeSource(fullPathName);
        this._piXMLFileNamesRead.remove(fullPathName);
        this._piXMLFileNamesToRead.remove(fullPathName);

        try
        {
            FileUtils.delete(fullPathName);
        }
        catch(final Exception e)
        {
            LOG.error("Unable to delete file " + fullPathName + ": " + e.getMessage());
        }
    }

    /**
     * Adds {@link #_piXMLDataDirectory} to the beginning of the file name using the {@link File#separator} separator.
     * 
     * @param filename
     * @return
     */
    public String buildFullPathNameForPIXMLFileFromFileName(String filename)
    {
        final File file = new File(filename);
        if(!file.isAbsolute())
        {
            if(_piXMLDataDirectory.exists())
            {
                filename = _piXMLDataDirectory.getAbsolutePath() + File.separator + filename;
            }
            else
            {
//                LOG.debug("The PI-XML Data Directory, " + _piXMLDataDirectory
//                    + ", does not exist!  Using root for PI-xml file name.");
            }
        }
        return filename;
    }

    /**
     * @param identifier The identifier to translate.
     * @return The translated identifier. By default, the passed in identifier is returned.
     */
    @Override
    public LocationAndDataTypeIdentifier translateIdentifier(final LocationAndDataTypeIdentifier identifier)
    {
        return identifier;
    }

    /**
     * @param originalIdentifiers
     * @return List of translated identifiers corresponding to the original identifiers. This list may repeat entries if
     *         two originals map to the same translated.
     */
    public List<LocationAndDataTypeIdentifier> translateIdentifiers(final Collection<LocationAndDataTypeIdentifier> originalIdentifiers)
    {
        final List<LocationAndDataTypeIdentifier> results = Lists.newArrayList();
        for(final LocationAndDataTypeIdentifier id: originalIdentifiers)
        {
            results.add(translateIdentifier(id));
        }
        return results;
    }

    /**
     * @param translatedIdentifier The identifier for which to check if enough time series are available.
     * @param availableTimeSeries The time series that are currently available for that time series based on
     *            {@link #_identifierToTimeSeriesMap}.
     * @return True if enough time series are available for the identifier to be used, false otherwise.
     */
    public boolean areAllRequiredTimeSeriesPresent(final LocationAndDataTypeIdentifier translatedIdentifier,
                                                   final List<TimeSeriesHeaderInfo> availableTimeSeriesHeaders)
    {
        return !availableTimeSeriesHeaders.isEmpty();
    }

    /**
     * Adds the source, identifier, and tsInfo to the handler's storage attributes. This uses a translated identifier
     * for most attributes, except for {@link #_usedSources} which uses the original identifier.
     * 
     * @param source
     * @param originalIdentifier
     * @param tsInfo
     */
    private void addToSourceIdentifierTimeSeriesInformationMaps(final Object source,
                                                                final LocationAndDataTypeIdentifier originalIdentifier,
                                                                final TimeSeriesHeaderInfo tsInfo)
    {
        //Get the translated identifier for storage.
        final LocationAndDataTypeIdentifier translatedIdentifier = translateIdentifier(originalIdentifier);
        _translatedIdentifierToOriginalMap.put(translatedIdentifier, originalIdentifier);

        //Add the information to the table using the original identifier.
        _tsInfoTable.put(source, originalIdentifier, tsInfo);

        //If the original identifier is already mapped to a source within _usedSource, then the identifier's time series will not be
        //used by MEFPPE.  However, if not, then we need to add to the _usedSources map, as well as ot the identifiers to headers map.
        if(!_usedSources.containsKey(originalIdentifier))
        {
            _usedSources.put(originalIdentifier, source);
            if(_translatedIdentifierToHeadersMap.get(translatedIdentifier) != null)
            {
                _translatedIdentifierToHeadersMap.get(translatedIdentifier).add(tsInfo);
            }
            else
            {
                final List<TimeSeriesHeaderInfo> infos = Lists.newArrayList();
                infos.add(tsInfo);
                _translatedIdentifierToHeadersMap.put(translatedIdentifier, tsInfo);
            }
        }

        //Store the used source based on the original identifier.

        //Record identifier for which data was found, but only if we now have enough time series to use the identifier.
        if(areAllRequiredTimeSeriesPresent(translatedIdentifier,
                                           _translatedIdentifierToHeadersMap.get(translatedIdentifier)))
        {
            if(translatedIdentifier.isForecastDataType())
            {
                ListTools.addItemIfNotAlreadyInList(_forecastIdentifiersForWhichReqdDataWasFound, translatedIdentifier);
            }
            else
            {
                ListTools.addItemIfNotAlreadyInList(_observedIdentifiersForWhichReqdDataWasFound, translatedIdentifier);
            }
        }

        //Record name of file.
        if(source instanceof String)
        {
            ListTools.addItemIfNotAlreadyInList(_piXMLFileNamesRead, (String)source);
        }

        Collections.sort(_piXMLFileNamesRead);
        Collections.sort(_forecastIdentifiersForWhichReqdDataWasFound);
        Collections.sort(_observedIdentifiersForWhichReqdDataWasFound);
    }

    /**
     * @param source Remove the source and all identifiers and time series dependent on it.
     */
    private void removeSource(final Object source)
    {
        // For every original identifier mapped from that source...
        for(final LocationAndDataTypeIdentifier originalIdentifier: _tsInfoTable.row(source).keySet())
        {
            final LocationAndDataTypeIdentifier translatedIdentifier = translateIdentifier(originalIdentifier);

            // Get the header info map for that identifier, showing which sources are available specifying it.
            final Map<Object, TimeSeriesHeaderInfo> sourceToInfoMap = _tsInfoTable.column(originalIdentifier);
            final TimeSeriesHeaderInfo sourceIdentifierHeaderInfo = sourceToInfoMap.get(source);

            //Remove the source line from the table via the map for the identifier.  If the map now has size 0,
            //remove all traces of the original identifier because there is no longer a source for it.
            sourceToInfoMap.remove(source);
            if(sourceToInfoMap.size() == 0)
            {
                //Get rid of the original identifier from the translated identifier map.  The multimap should handle
                //clearing out all trace of the translatedIdentifier if ther are no longer any originals to map to it.
                _translatedIdentifierToOriginalMap.remove(translatedIdentifier, originalIdentifier);

                //Remove the original identifier from the headers map
                _translatedIdentifierToHeadersMap.remove(translatedIdentifier, sourceIdentifierHeaderInfo);

                //Used sources connection to the source must also be removed... there is not longer a source for the identifier.
                _usedSources.remove(originalIdentifier);
            }
            //So there are still sources that point to this identifier...
            else
            {
                //If the original identifier's source header info, the one to be removed, is in the identifiers to headers map...
                if(_translatedIdentifierToHeadersMap.containsEntry(translatedIdentifier, sourceIdentifierHeaderInfo))
                {
                    //Remove it and use the first entry int he sourceToInfoMap values.
                    _translatedIdentifierToHeadersMap.remove(translatedIdentifier, sourceIdentifierHeaderInfo);
                    _translatedIdentifierToHeadersMap.put(translatedIdentifier,
                                                          IterableTools.first(sourceToInfoMap.values()));

                    //Remap the used sources link
                    _usedSources.put(originalIdentifier, IterableTools.first(sourceToInfoMap.values()));
                }
            }

            // If we no longer have enough time series to use the translatedIdentifier, remove it from the appropriate list.
            if(!areAllRequiredTimeSeriesPresent(translatedIdentifier,
                                                _translatedIdentifierToHeadersMap.get(translatedIdentifier)))
            {
                if(translatedIdentifier.isForecastDataType())
                {
                    _forecastIdentifiersForWhichReqdDataWasFound.remove(translatedIdentifier);
                }
                else
                {
                    _observedIdentifiersForWhichReqdDataWasFound.remove(translatedIdentifier);
                }
            }

            //Now for loaded time series.
            final TimeSeriesArray loadedTS = _originalIdentifeirToTSMap.get(originalIdentifier);
            if(loadedTS != null)
            {
                _originalIdentifeirToTSMap.remove(originalIdentifier);
                _translatedIdentifierToTSMap.remove(translatedIdentifier, loadedTS);
            }
        }
    }

    /**
     * Extracts all times series from the given source.
     * 
     * @param source the source to extract from. Currently a (String) filename; nothing else is acceptable.
     * @return all time series in the given source
     */
    private TimeSeriesArrays extractListOfTrimmedTimeSeries(final Object source) throws Exception
    {
        TimeSeriesArrays readSeries;

        // Determine source type and then read in all time series from the source.
        if(source instanceof String)
        {
            final File sourceFile = new File((String)source);
            try
            {
                readSeries = TimeSeriesArraysTools.readFromFile(sourceFile);
            }
            catch(final Exception e)
            {
                e.printStackTrace();
                throw new Exception("Unable to parse time series in file " + sourceFile.getAbsolutePath()
                    + ".  Message: " + e.getMessage());
            }

//Why did I manually do the reading below instead of using the tools?  XXX Can this be removed?            
//            try
//            {
//                LOG.info("Loading time series from the file " + sourceFile.getAbsolutePath() + "...");
//                final PiTimeSeriesReader reader = new PiTimeSeriesReader(sourceFile);
//                readSeries = reader.read();
//                reader.close();
//            }
//            catch(final Exception e)
//            {
//                e.printStackTrace();
//                throw new Exception("Unable to parse time series in file " + sourceFile.getAbsolutePath()
//                    + ".  Message: " + e.getMessage());
//            }
        }
        else
        {
            throw new Exception("INTERNAL ERROR: Cannot extract time series from given source.");
        }

        //Translate the parameters
        for(int i = 0; i < readSeries.size(); i++)
        {
            translateHeaderExternalParameterId(readSeries.get(i).getHeader());
        }

        TimeSeriesArraysTools.trimMissingValuesFromBeginningAndEndOfTimeSeries(readSeries);
        TimeSeriesArraysTools.convertTimeSeriesUnits(readSeries, true); //to metric
        LOG.info("Done loading time series.  Loaded " + readSeries.size() + " matching time series.");
        return readSeries;
    }

    /**
     * @param identifiers list of identifiers to search for
     * @param series the series to filter out
     * @return a list of time series having any of the given identifiers
     */
    private List<TimeSeriesArray> filterTimeSeriesByIdentifiers(final List<LocationAndDataTypeIdentifier> translatedIdentifiers,
                                                                final TimeSeriesArrays series)
    {
        final List<TimeSeriesArray> matchingTimeSeries = new ArrayList<TimeSeriesArray>();

        //Identifiers to use are based on untranslating the translated identifiers.
        final List<LocationAndDataTypeIdentifier> originalIdentifiers = Lists.newArrayList();
        for(final LocationAndDataTypeIdentifier translatedId: translatedIdentifiers)
        {
            originalIdentifiers.addAll(this._translatedIdentifierToOriginalMap.get(translatedId));
        }

        // Pick only matching time series.
        LOG.info("Throwing out unmatching time series ... ");
        for(int i = 0; i < series.size(); i++)
        {
            final TimeSeriesArray ts = series.get(i);
            if(LocationAndDataTypeIdentifier.isTimeSeriesForAtLeastOneIdentifier(originalIdentifiers, ts))
            {
                matchingTimeSeries.add(ts);
            }
        }

        LOG.info("Filtered to a total of " + matchingTimeSeries.size() + " matching time series.");
        return matchingTimeSeries;
    }

    /**
     * @param fcstIdentifier Identifier with a forecast data type.
     * @return List of identifiers with an observed data type that shares the same locationId and general type of data
     *         as the fcstIdentifier.
     */
    public LocationAndDataTypeIdentifierList findCorrespondingObservedIdentifier(final LocationAndDataTypeIdentifier fcstIdentifier)
    {
        final List<LocationAndDataTypeIdentifier> results = new ArrayList<LocationAndDataTypeIdentifier>();
        for(final LocationAndDataTypeIdentifier obsIdentifier: this._observedIdentifiersForWhichReqdDataWasFound)
        {
            if(fcstIdentifier.getLocationId().equalsIgnoreCase(obsIdentifier.getLocationId())
                && fcstIdentifier.isSameTypeOfData(obsIdentifier))
            {
                results.add(obsIdentifier);
            }
        }
        return new LocationAndDataTypeIdentifierList(results);
    }

    /**
     * @param fcstIdentifier Forecast identifier to be searched for.
     * @return A list of LocationAndDataTypeIdentifier containing the provided fcstIdentifier and all observed
     *         identifiers that correspond to the forecast identifier. The return of this method can be passed into the
     *         load methods to acquire all time series, forecast and observed, corresponding to a forecast identifier.
     */
    public List<LocationAndDataTypeIdentifier> generateListOfIdentifiersForForecastIdentifier(final LocationAndDataTypeIdentifier fcstIdentifier)
    {
        final List<LocationAndDataTypeIdentifier> results = Lists.newArrayList(fcstIdentifier);
        results.addAll(findCorrespondingObservedIdentifier(fcstIdentifier));
        return results;
    }

    /**
     * @return The identifier found in either the {@link #_forecastIdentifiersForWhichReqdDataWasFound} or
     *         {@link #_observedIdentifiersForWhichReqdDataWasFound} lists with the given location id and parameter id.
     */
    public LocationAndDataTypeIdentifier findIdentifier(final String locationId, final String parameterId)
    {
        if(HEFSTools.isForecastDataType(parameterId))
        {
            for(final LocationAndDataTypeIdentifier identifier: _forecastIdentifiersForWhichReqdDataWasFound)
            {
                if(identifier.isThisIdentifierForLocationAndDataType(locationId, parameterId))
                {
                    return identifier;
                }
            }
        }
        else
        {
            for(final LocationAndDataTypeIdentifier identifier: _observedIdentifiersForWhichReqdDataWasFound)
            {
                if(identifier.isThisIdentifierForLocationAndDataType(locationId, parameterId))
                {
                    return identifier;
                }
            }
        }
        return null;
    }

    /**
     * @return List of file names of PI-timeseries XML/FI files that have been read.
     */
    public List<String> getPIXMLFileNamesRead()
    {
        return this._piXMLFileNamesRead;
    }

    /**
     * @return {@link List} of the original identifiers for which time series are found for the given source. This does
     *         not translate any of the identifiers.
     */
    public List<LocationAndDataTypeIdentifier> getOriginalIdentifiersFoundForSource(final Object source)
    {
        return Lists.newArrayList(_tsInfoTable.row(source).keySet());
    }

    /**
     * @return The original, pre-translation identifiers used from the source.
     */
    public List<LocationAndDataTypeIdentifier> getOriginalIdentifiersUsedFromSource(final Object source)
    {
        final List<LocationAndDataTypeIdentifier> results = Lists.newArrayList();

        for(final LocationAndDataTypeIdentifier identifier: _tsInfoTable.row(source).keySet())
        {
            if(_usedSources.get(identifier).equals(source))
            {
                results.add(identifier);
            }
        }

        return results;
    }

    /**
     * If the subclass does not override {@link #translateIdentifier(LocationAndDataTypeIdentifier)}, then this method
     * can be called instead of {@link #getSourcesProvidingDataForIdentifier(LocationAndDataTypeIdentifier)}, since we
     * know that each identifier can correspond to only one time series and therefore one source.
     * 
     * @return The source containing the data for the original identifier.
     */
    public Object getSourceProvidingDataForOriginalIdentifier(final LocationAndDataTypeIdentifier originalIdentifier)
    {
        return _usedSources.get(originalIdentifier);
    }

    /**
     * @return The source from which the time series for the given identifier will be loaded. The identifier must be
     *         that before translation via {@link #translateIdentifier(LocationAndDataTypeIdentifier)}.
     */
    public List<Object> getSourcesProvidingDataForIdentifier(final LocationAndDataTypeIdentifier translatedIdentifier)
    {
        final List<Object> sources = Lists.newArrayList();
        for(final LocationAndDataTypeIdentifier originalIdentifier: _usedSources.keySet())
        {
            if(translateIdentifier(originalIdentifier).equals(translatedIdentifier))
            {
                sources.add(_usedSources.get(originalIdentifier));
            }
        }
        return sources;
    }

    /**
     * Calls {@link #getTimeSeriesInformationForSourceAndOriginalIdentifier(Object, LocationAndDataTypeIdentifier)},
     * passing in the source returned by {@link #getSourceProvidingDataForIdentifier(LocationAndDataTypeIdentifier)}.
     */
    public TimeSeriesHeaderInfo getUsedTimeSeriesInformationForOriginalIdentifier(final LocationAndDataTypeIdentifier originalIdentifier)
    {
        final Object source = getSourceProvidingDataForOriginalIdentifier(originalIdentifier);
        return getTimeSeriesInformationForSourceAndOriginalIdentifier(source, originalIdentifier);
    }

    /**
     * @return The time series header info pulled from the given source and original, pre-translation identifier.
     */
    private TimeSeriesHeaderInfo getTimeSeriesInformationForSourceAndOriginalIdentifier(final Object source,
                                                                                        final LocationAndDataTypeIdentifier originaldentifier)
    {
        return _tsInfoTable.get(source, originaldentifier);
    }

    public File getPIXMLDataDirectory()
    {
        return this._piXMLDataDirectory;
    }

    /**
     * Parses read in time series into the forecast and observed lists.
     */
    protected void putTimeSeriesIntoLists()
    {
        for(final TimeSeriesArray ts: _translatedIdentifierToTSMap.values())
        {
            if(HEFSTools.isForecastDataType(ts.getHeader().getParameterId()))
            {
                _listOfForecastTimeSeries.add(ts);
            }
            else
            {
                _listOfObservedTimeSeries.add(ts);
            }
        }
    }

    protected List<TimeSeriesArray> getListOfForecastTimeSeries()
    {
        return _listOfForecastTimeSeries;
    }

    protected List<TimeSeriesArray> getListOfObservedTimeSeries()
    {
        return _listOfObservedTimeSeries;
    }

    public Collection<LocationAndDataTypeIdentifier> getAllOriginalIdentifiersForWhichDataWasFound()
    {
        return _usedSources.keySet();
    }

    /**
     * @return All time series found for the given translated, usable identifier.
     */
    protected List<TimeSeriesArray> getLoadedTimeSeriesForIdentifer(final LocationAndDataTypeIdentifier translatedIdentifier)
    {
        return _translatedIdentifierToTSMap.get(translatedIdentifier);
    }

    /**
     * @return True if any time series have been read in for the identifier.
     */
    protected boolean areLoadedTimeSeriesPresent(final LocationAndDataTypeIdentifier translatedIdentifier)
    {
        return _translatedIdentifierToTSMap.containsKey(translatedIdentifier);
    }

    /**
     * Puts the loaded time series into the map after constructing an identifier via
     * {@link LocationAndDataTypeIdentifier#get(TimeSeriesArray)} and translating via
     * {@link #translateIdentifier(LocationAndDataTypeIdentifier)}.
     * 
     * @param ts The time series to add.
     */
    protected void addLoadedTimeSeries(final TimeSeriesArray ts)
    {
        _translatedIdentifierToTSMap.put(translateIdentifier(LocationAndDataTypeIdentifier.get(ts)), ts);
    }

    /**
     * @return The identifiers returned are translated via {@link #translateIdentifier(LocationAndDataTypeIdentifier)},
     *         not raw.
     */
    public List<LocationAndDataTypeIdentifier> getForecastIdentifiersForWhichReqdDataWasFound()
    {
        return this._forecastIdentifiersForWhichReqdDataWasFound;
    }

    /**
     * @return The identifiers returned are translated, not raw.
     */
    public List<LocationAndDataTypeIdentifier> getObservedIdentifiersForWhichReqdDataWasFound()
    {
        return this._observedIdentifiersForWhichReqdDataWasFound;
    }

    /**
     * Override as needed.
     * 
     * @param identifier Identifier to check.
     * @return True if the identifier specifies data that is to be loaded by the data handler and possibly displayed in
     *         the MEFPPE diagnostics. This is applied when the PI-service is accessed for time series (time series not
     *         included are thwon out). It is also applied when determining what to put in the tree displayed in the
     *         setup panel. This is NOT applied to determine new identifiers for which to estimate parameters.
     */
    public boolean includeTimeSeries(final LocationAndDataTypeIdentifier identifier)
    {
        return true;
    }

    /**
     * @return Query id used to query the PI-service and acquire time series that correspond to the files this
     *         DataHandler must process. This is useful for the export setup panel. By default, it returns an empty
     *         string.
     */
    public String getPIServiceQueryId()
    {
        return "";
    }

    /**
     * @return Client id used to query the PI-service and acquire time series that correspond to the files this
     *         DataHandler must process. This is useful for the export setup panel. By default, it returns an empty
     *         string.
     */
    public String getPIServiceClientId()
    {
        return "";
    }

    @Override
    public void setDataHandlerBaseDirectory(final File directory)
    {
        _dataHandlerBaseDirectory = directory;
        _piXMLDataDirectory = new File(_dataHandlerBaseDirectory.getAbsolutePath() + File.separator + _piXMLSubDirName);
    }

    @Override
    public void loadOriginalTimeSeries(final List<LocationAndDataTypeIdentifier> translatedIdentifiers) throws Exception
    {
        LOG.info("Retrieving all time series for a list of " + translatedIdentifiers.size()
            + " location and data type identifiers.");

        //XXX Uncomment the following line if you want it to forget loaded time series.
        clearLoadedTimeSeries();

        final List<Object> listOfProcessedSources = new ArrayList<Object>();

        //Loop through all identifiers.
        for(final LocationAndDataTypeIdentifier identifier: translatedIdentifiers)
        {

            //If the time series has already been loaded.  This will always be null if the map clear
            //above is used (i.e., if time series area to be loaded from scratch).
            if(!areLoadedTimeSeriesPresent(identifier))
            {

                final List<Object> sources = getSourcesProvidingDataForIdentifier(identifier);
                if((sources == null) || sources.isEmpty())
                {
                    throw new Exception("No historical data file exists for that location and data type");
                }

                for(final Object source: sources)
                {
                    if(!listOfProcessedSources.contains(source))
                    {
                        listOfProcessedSources.add(source);

                        // Use all identifiers here so we only need to load each source once.
                        for(final TimeSeriesArray ts: filterTimeSeriesByIdentifiers(translatedIdentifiers,
                                                                                    extractListOfTrimmedTimeSeries(source)))
                        {
                            //Translate the parameter
                            addLoadedTimeSeries(ts);
                        }
                    }
                }
            }
        }

        // Convert the values in the results map into a list of time series
        putTimeSeriesIntoLists();
        if(_listOfForecastTimeSeries.isEmpty() && _listOfObservedTimeSeries.isEmpty())
        {
            throw new Exception("No time series found match any in the list of location and data type identifiers.");
        }

        LOG.info("Done retrieving all time series for the list of location and data type identifiers.  Found "
            + _translatedIdentifierToTSMap.values().size() + " time series that matched location and data types.");
    }

    @Override
    public void clearLoadedTimeSeries()
    {
        _translatedIdentifierToTSMap.clear();
        _listOfForecastTimeSeries.clear();
        _listOfObservedTimeSeries.clear();
    }

    /**
     * Scans the {@link #_identifierToTimeSeriesMap} and returns all time series for the same location and data type as
     * that passed in, so long as the parameter id for the time series is an observed data type based on
     * {@link LocationAndDataTypeIdentifier#isObservedDataType()}.
     * 
     * @param identifier
     * @return All loaded time series the given identifier information that is an observed data type time series.
     */
    @Override
    public Collection<TimeSeriesArray> getLoadedObservedTimeSeries(final LocationAndDataTypeIdentifier identifier)
    {
        final List<TimeSeriesArray> results = new ArrayList<TimeSeriesArray>();
        for(final LocationAndDataTypeIdentifier id: _translatedIdentifierToTSMap.keySet())
        {
            if((id.isForSameLocation(identifier)) && (id.isSameTypeOfData(identifier)))
            {
                final List<TimeSeriesArray> tsToAdd = _translatedIdentifierToTSMap.get(id);
                for(final TimeSeriesArray ts: tsToAdd)
                {
                    if(LocationAndDataTypeIdentifier.get(ts).isObservedDataType())
                    {
                        results.add(ts);
                    }
                }
            }
        }
        return results;
    }

    /**
     * Scans the {@link #_identifierToTimeSeriesMap} and returns all time series for the same location and data type as
     * that passed in, so long as the parameter id for the time series is a forecast data type based on
     * {@link LocationAndDataTypeIdentifier#isForecastDataType()}.
     * 
     * @param identifier
     * @return All loaded time series the given identifier information that is an observed data type time series.
     */
    @Override
    public Collection<TimeSeriesArray> getLoadedForecastTimeSeries(final LocationAndDataTypeIdentifier identifier)
    {
        final List<TimeSeriesArray> results = new ArrayList<TimeSeriesArray>();
        for(final LocationAndDataTypeIdentifier id: _translatedIdentifierToTSMap.keySet())
        {
            if((id.isForSameLocation(identifier)) && (id.isSameTypeOfData(identifier)))
            {
                final List<TimeSeriesArray> tsToAdd = _translatedIdentifierToTSMap.get(id);
                for(final TimeSeriesArray ts: tsToAdd)
                {
                    if(LocationAndDataTypeIdentifier.get(ts).isForecastDataType())
                    {
                        results.add(ts);
                    }
                }
            }
        }
        return results;
    }

    @Override
    public void initialize() throws Exception
    {
        LOG.info("Initializing PI-xml data handler...");
        _piXMLFileNamesRead.clear();
        _piXMLFileNamesToRead.clear();
        _forecastIdentifiersForWhichReqdDataWasFound.clear();
        _observedIdentifiersForWhichReqdDataWasFound.clear();
        _tsInfoTable.clear();
        _usedSources.clear();
        _translatedIdentifierToOriginalMap.clear();
        _translatedIdentifierToHeadersMap.clear();
        _originalIdentifeirToTSMap.clear();
        clearLoadedTimeSeries();

        //Add all files, but throw out the exception.
        try
        {
            addAllFilesInHistoricalDataDirectoryToUnreadList();
        }
        catch(final Exception e)
        {
            //Ignore: The only kind of exception that will occur is if a file to add to the to-read list
            //has already been read.  That cannot be the case here because the lists are cleared above.
        }

        readUnreadHistoricalFilesAndAddToLists();
        LOG.info("Done initializing PI-xml data handler.");
    }

    @Override
    public List<TimeSeriesArray> getAllLoadedObservedTimeSeries()
    {
        return this._listOfObservedTimeSeries;
    }

    @Override
    public List<TimeSeriesArray> getAllLoadedForecastTimeSeries()
    {
        return this._listOfForecastTimeSeries;
    }

    @Override
    public File getDataHandlerBaseDirectory()
    {
        return this._dataHandlerBaseDirectory;
    }

//THE BELOW ILLUSTRATES HOW TO CREATE MY OWN HASHMAP IN WHICH FILES CAN BE KEYS.  FILES CAUSE PROBLEMS BECAUSE TWO FILE OBJECTS
//CAN POINT TO THE SAME FILE BUT HAVE DIFFERENT HASHCODE VALUES.
//    /**
//     * Special inner class is needed to convert from File to String (absolute path) because the File hashCode method
//     * does not allow for two File objects that point to the same file to have the same key. This will map the File to
//     * its absolute path when putting it in the map and vice versa when pulling it out.
//     * 
//     * @author hank.herr
//     * @param <K> Usually Object
//     * @param <V> Whatever
//     */
//    private class SourceKeyLinkedHashMap<K, V> extends HashMap<K, V>
//    {
//        private static final long serialVersionUID = 1L;
//
//        @Override
//        public V get(Object key)
//        {
//            if(key instanceof File)
//            {
//                return super.get(((File)key).getAbsolutePath());
//            }
//            return super.get(key);
//        }
//
//        @Override
//        public V put(K key, V value)
//        {
//            this.keySet();
//            if(key instanceof File)
//            {
//                return super.put((K)((File)key).getAbsolutePath(), value);
//            }
//            return super.put(key, value);
//        }
//
//        @Override
//        public Set<K> keySet()
//        {
//            LinkedHashMap<K, V> keyMap = new LinkedHashMap<K, V>();
//
//            Set<K> ks = super.keySet();
//            for(K source: ks)
//            {
//                if(source instanceof String)
//                {
//                    keyMap.put((K)(new File((String)source)), null);
//                }
//                else
//                {
//                    keyMap.put(source, null);
//                }
//            }
//
//            return keyMap.keySet();
//        }
//    }

}
