package ohd.hseb.hefs.pe.tools;

import static com.google.common.collect.Lists.newArrayList;

import java.awt.Color;
import java.io.File;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeSet;

import nl.wldelft.fews.common.config.GlobalProperties;
import nl.wldelft.util.Period;
import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.charter.ChartEngine;
import ohd.hseb.charter.ChartTools;
import ohd.hseb.charter.datasource.XYChartDataSource;
import ohd.hseb.charter.datasource.instances.TimeSeriesArraysXYChartDataSource;
import ohd.hseb.charter.parameters.DataSourceDrawingParameters;
import ohd.hseb.charter.parameters.SeriesDrawingParameters;
import ohd.hseb.charter.parameters.ThresholdListParameters;
import ohd.hseb.charter.parameters.ThresholdParameters;
import ohd.hseb.graphgen.GraphGenSettingsController;
import ohd.hseb.graphgen.chartseries.ChartSeriesParameters;
import ohd.hseb.graphgen.core.GraphicsGeneratorEngine;
import ohd.hseb.graphgen.core.GraphicsGeneratorProductParameters;
import ohd.hseb.graphgen.inputseries.TimeSeriesSelectionParameters;
import ohd.hseb.hefs.mefp.sources.MEFPSourceDataHandler;
import ohd.hseb.hefs.mefp.tools.QuestionableMessageMap;
import ohd.hseb.hefs.mefp.tools.QuestionableTools;
import ohd.hseb.hefs.utils.datetime.HEFSDateTools;
import ohd.hseb.hefs.utils.tools.GeneralTools;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;
import ohd.hseb.hefs.utils.xml.GenericXMLReadingHandler;
import ohd.hseb.util.Pair;
import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.HString;

import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

/**
 * Draws a chart based on the given parameter file and time series... this only works with time series based
 * diagnostics!.<br>
 * <br>
 * This looks at the chart series parameters and corresponding data source drawing parameters within the given parameter
 * file. Based on each of these, it draws all of its given series in the same format as the first series of each data
 * source.<br>
 * <br>
 * Also assumes that any given parameter file has the following execution parameters (arguments): displayedLocationId,
 * displayedParameterId, sourceId.<br>
 * <br>
 * This will also color any data source which has a forecast time and is unemphasized according to a sequence of colors,
 * to help distinguish them.<br>
 * <br>
 * If a data source uses the AreaBetweenLines plotter, this separates all given series into separate chart series / data
 * source drawing parameters by forecast time.<br>
 * <br>
 * Currently, subclasses specify which parameter file to load.
 * 
 * @author alexander.garbarino
 */
public class DiagnosticChartBuilder
{
    protected final LocationAndDataTypeIdentifier _displayedIdentifier;
    protected final TimeSeriesSorter _loadedSeries = new TimeSeriesSorter();
    private final TimeSeriesSorter _displayedSeries = new TimeSeriesSorter();
    protected final SortedSet<Long> _forecastTimes = new TreeSet<Long>();
    protected GraphicsGeneratorProductParameters _parameters;
    protected final List<ChartSeriesParameters> _defaultSeriesParameters;
    protected final List<DataSourceDrawingParameters> _defaultDrawingParameters;

    protected Color[] _solidColors;
    protected Color[] _transparentColors;

    protected Integer _year = null;
    protected ChartEngine _engine = null;

    protected ThresholdParameters _templateParameters = null;

    /**
     * A questionable map set via a call to {@link #setQuestionableMap(HashMap)} after the constructor is built and
     * before calling {@link #buildChartEngine()}.
     */
    private QuestionableMessageMap _questionableMap;

    public DiagnosticChartBuilder(final LocationAndDataTypeIdentifier identifier,
                                  final String parameterFile,
                                  final String sourceId,
                                  final Iterable<? extends TimeSeriesArray>... series) throws Exception
    {
        this(identifier, parameterFile, sourceId, ListTools.<TimeSeriesArray>concat(series));
    }

    public DiagnosticChartBuilder(final LocationAndDataTypeIdentifier identifier,
                                  final String parameterFile,
                                  final String sourceId,
                                  final Iterable<? extends TimeSeriesArray> series) throws Exception
    {
        _displayedIdentifier = identifier;
        loadParameters(parameterFile);

        // Grab default parameters.
        _defaultSeriesParameters = new ArrayList<ChartSeriesParameters>(_parameters.getTemplateParameters()
                                                                                   .getChartSeriesParameters());
        _defaultDrawingParameters = new ArrayList<DataSourceDrawingParameters>(_parameters.getTemplateParameters()
                                                                                          .getChartDrawingParameters()
                                                                                          .getDataSourceParameters());

        // Set execution parameters.
        _parameters.getExecutionParameters().addParameter("displayedLocationId", _displayedIdentifier.getLocationId());
        _parameters.getExecutionParameters()
                   .addParameter("displayedParameterId", _displayedIdentifier.getParameterId());
        _parameters.getExecutionParameters().addParameter("sourceId", sourceId);

        //XXX Do I want this to come from run info?  If so, I need to pass the run info down into every single step,
        //options panel, and into here.
        _parameters.getExecutionParameters().addParameter("configDirectory",
                                                          GlobalProperties.get("REGION_HOME") + File.separator
                                                              + "Config");

        // Initialize colors.
        _solidColors = Arrays.copyOfRange(ChartTools.buildESPADPColorPalette(9), 0, 7);
        _transparentColors = new Color[_solidColors.length];
        for(int i = 0; i < _solidColors.length; i++)
        {
            final Color color = _solidColors[i];
            _transparentColors[i] = new Color(color.getRed(), color.getGreen(), color.getBlue(), 20);
        }

        for(final TimeSeriesArray array: series)
        {
            TimeSeriesArrayTools.fillNaNs(array);
            _loadedSeries.add(array);
        }

        // I think the (de)emphasizing causes the two sorters to get out of sync.
        _loadedSeries.clearEmphasis();
        _displayedSeries.addAll(_loadedSeries);
    }

    /**
     * Call this method, if desired, after constructing this and before calling {@link #prepareChart()}. This adds a
     * threshold for each chunk of questionable data found based on
     * {@link TimeSeriesArraysTools#isAnyValueQuestionableAtTime(Collection, long, boolean)}. This calls
     * {@link #addDomainThreshold(String, long, long, Color, Color)}.<br>
     * <br>
     * This is useful only for displaying long historical time series. It checks all observed time series to be
     * displayed to see if any contain questionable values. The first time checked is the overall start time of all time
     * series, while the last time checked is the overall end time. Since reforecasts are not have many varying start
     * times and end times, checking the overall start-end time period is not reasonable; each reforecast time series
     * should be checked independently. In that case, however, the thresholds could be very cluttered with many
     * overlaying each other.
     * 
     * @param Predicate<TimeSeriesArray> useTimeSeriesPredicate
     * @param questionableMarkColor - The color to mark a single questionable value.
     * @param questionableZoneColor - The color to mark regions of questionable data.
     */
    private void addDomainThresholdsForQuestionableValues(final Predicate<TimeSeriesArray> useTimeSeriesPredicate,
                                                          final Color questionableMarkColor,
                                                          final Color questionableZoneColor)
    {
        //No questionable map available... just return.
        if((_questionableMap == null) || (_questionableMap.isEmpty()))
        {
            return;
        }

        // Clear the QC threshold parameters in the Threshold List. This is so when we add a new QC Template they 
        // don't keep displaying on top of each other.
        final ThresholdListParameters tlp = _parameters.getTemplateParameters()
                                                       .getChartDrawingParameters()
                                                       .getThresholdList();
        final ThresholdParameters qctp = tlp.retrieveThresholdParameters("QC TEMPLATE");
        tlp.clearParameters();

        //Record the template parameters if they were found, but do NOT put the template parameters back in.  
        //That causes problems for zooming out from a plot, since that will not make use of the helper Zhengtao created.
        if(qctp != null)
        {
            _templateParameters = qctp;
        }

        Long startTimeOfBlock = Long.MIN_VALUE;
        Long endTimeOfBlock = Long.MIN_VALUE;
        Long currentTime = Long.MIN_VALUE;
        final List<String> messages = Lists.newArrayList();

        //Get the collection and range of times that covers all time series.
        final Collection<TimeSeriesArray> collection = Lists.newArrayList(Iterables.filter(_displayedSeries,
                                                                                           useTimeSeriesPredicate));

        final long[] timeRange = TimeSeriesArraysTools.getRangeOfTimes(collection);
        if(timeRange == null)
        {
            return; //Do nothing... no times to worry about.
        }

        //Loop over the time range.
        for(currentTime = timeRange[0]; currentTime <= timeRange[1]; currentTime += TimeSeriesArraysTools.findSmallestTimeStep(collection))
        {
            //No questionable data found for the time, so end the currently building threshold if one exists.
//            if(!QuestionableTools.isAnyValueQuestionable(_questionableMap, currentTime)) // not questionable
            if(!_questionableMap.isAnyValueQuestionable(currentTime))
            {
                // Add the previous block, if any, and clear the messages.
                if(startTimeOfBlock != Long.MIN_VALUE)
                {
                    addDomainThreshold(startTimeOfBlock,
                                       endTimeOfBlock,
                                       questionableMarkColor,
                                       questionableZoneColor,
                                       messages);
                    messages.clear();
                }
                startTimeOfBlock = Long.MIN_VALUE;
                endTimeOfBlock = Long.MIN_VALUE;
            }
            //Otherwise, questionable data was found, so build on the previous threshold or make a new one.
            else
            {
                final List<String> subMessages = _questionableMap.getMessagesForAllTimeSeries(currentTime);

                //Determine the message to add, making sure to add only ONE line per time.
                String message = "";
                if(!subMessages.isEmpty())
                {
                    message += "<b><u>" + HCalendar.buildDateStr(currentTime, HCalendar.DEFAULT_DATE_FORMAT)
                        + "</b></u> - " + subMessages.get(0);
                }
                if(subMessages.size() > 1)
                {
                    message += " (+ " + (subMessages.size() - 1) + " more)";
                }
                messages.add(message);

                //Start the new block if needed.
                if(startTimeOfBlock == Long.MIN_VALUE)
                {
                    startTimeOfBlock = currentTime;
                }
                endTimeOfBlock = currentTime;
            }
        }

        // Check if the last time checked contains questionable data.
        if(startTimeOfBlock != Long.MIN_VALUE)
        {
            addDomainThreshold(startTimeOfBlock, endTimeOfBlock, questionableMarkColor, questionableZoneColor, messages);
        }
    }

    /**
     * Calls {@link #addDomainThresholdsForQuestionableValues(Collection, Color, Color)} passing in the time series to
     * check.
     */
    public void addDomainThresholdsForQuestionableValues(final Color questionableMarkColor,
                                                         final Color questionableZoneColor)
    {
        addDomainThresholdsForQuestionableValues(Predicates.<TimeSeriesArray>alwaysTrue(),
                                                 questionableMarkColor,
                                                 questionableZoneColor);
    }

    /**
     * Adds a time-based domain threshold to display either a mark or zone. Whether its a mark or zone is controlled by
     * if the start and end time are identical. The threshold is built using {@link #_templateParameters} as a starting
     * point.
     * 
     * @param startTime The start time of the threshold.
     * @param endTime The end time of the threshold. If this does not equal the startTime, then it is assumed the
     *            threshold is a zone.
     * @param colorIfMark The color to use if this is a mark.
     * @param colorIfZone The color to use if this is a zone.
     */
    public void addDomainThreshold(final long startTime,
                                   final long endTime,
                                   final Color colorIfMark,
                                   final Color colorIfZone,
                                   final List<String> messages)
    {
        //Get the template threshold parameters and copy it.
        if(_templateParameters == null)
        {
            throw new IllegalArgumentException("No template threshold parameters available.");
        }
        final ThresholdParameters thresholdParms = (ThresholdParameters)_templateParameters.clone();

        //Make changes as needed.
        thresholdParms.setIsZone(startTime != endTime); //Zone if the start and end are not the same.
        thresholdParms.setDateAxisValueStart(HCalendar.buildDateTimeStr(startTime));
        thresholdParms.setDateAxisValueEnd(HCalendar.buildDateTimeStr(endTime));
        if(thresholdParms.getIsZone())
        {
            thresholdParms.setColor(colorIfZone);
        }
        else
        {
            thresholdParms.setColor(colorIfMark);
            thresholdParms.setLineWidth(3.0f); //Marks are 3 wide, however many pixels that is.
        }

        //Give it a unique identifier, make it visible and add it to the list.
        thresholdParms.setIdentifier("threshold "
            + _parameters.getTemplateParameters()
                         .getChartDrawingParameters()
                         .getThresholdList()
                         .getThresholdParametersList()
                         .size());
        thresholdParms.setVisible(true);
        thresholdParms.setShowLabelInPlot(false);
        thresholdParms.getLabel().setText("<html>Questionable Data:<br>"
            + HString.buildStringFromList(messages, "<br>") + "</html>");
        _parameters.getTemplateParameters().getChartDrawingParameters().getThresholdList().addThreshold(thresholdParms);
    }

    /**
     * Set the questionable map used for constructing thresholds of questionable data.
     * 
     * @param questionableMap Map generated by {@link QuestionableTools#toHash(File)} probably via a call to
     *            {@link MEFPSourceDataHandler#loadQuestionableHash(LocationAndDataTypeIdentifier)}.
     */
    public void setQuestionableMap(final QuestionableMessageMap questionableMap)
    {
        _questionableMap = questionableMap;
    }

    /**
     * Sets the year to be displayed by this chart. Set to null to allow all years.
     * 
     * @param year the year to display, or null for all years
     */
    public void setYear(final Integer year)
    {
        final Integer oldYear = _year;
        _year = year;
        if(!GeneralTools.checkForFullEqualityOfObjects(year, oldYear))
        {
            _displayedSeries.clear();
            _engine = null;

            if(year == null)
            {
                _displayedSeries.addAll(_loadedSeries);
            }
            else
            {
                //Observed data is restricted based on forecasts displayed
                final long[] times = TimeSeriesArraysTools.getRangeOfTimes(_loadedSeries.forecastInYear(year));
                if(times != null)
                {
                    final Period period = new Period(times[0], times[1]);
                    for(final TimeSeriesArray tsa: _loadedSeries.restrictViewToObserved())
                    {
                        _displayedSeries.add(tsa.subArray(period));
                    }
                }
                else
                {
                    for(final TimeSeriesArray tsa: _loadedSeries.restrictViewToObserved())
                    {
                        final Period period = new Period(HCalendar.computeFirstOfYear(year).getTime(),
                                                         HCalendar.computeLastOfYear(year).getTime());
                        _displayedSeries.add(tsa.subArray(period));
                    }
                }

                // Add all series forecast in the given year.
                _displayedSeries.addAll(_loadedSeries.forecastInYear(year));
            }

            // Track forecast times.
            _forecastTimes.clear();
            for(final TimeSeriesArray tsa: _displayedSeries)
            {
                final long time = tsa.getHeader().getForecastTime();
                if(time > Long.MIN_VALUE)
                {
                    _forecastTimes.add(time);
                }
            }

        }

    }

    /**
     * Sets the emphasized dates. Call after setting the year, not before.
     * 
     * @param dates the dates to be emphasized
     */
    public void setEmphasizedDates(final Collection<Long> dates)
    {
        _displayedSeries.setEmphasizedDates(dates);
        _engine = null;
    }

    /**
     * @return A set of {@link GraphicsGeneratorProductParameters} that can be used to modify the appearance of the
     *         chart.
     */
    public GraphicsGeneratorProductParameters getParameters()
    {
        return _parameters;
    }

    /**
     * Prepares the chart to be drawn. Call if you've made any changes since the chart was last drawn.
     */
    //XXX This method appears to have problems if the AreaBetweenLines is used AND time series are given with different 
    //time steps, even if the area is not filled for the odd ball time step.  I tried to use this for historical temperature
    //diagnostic panel, but it failed to fill in the area for the min/max.  Fortunately, I don't need the area: I switched it
    //to a step plot.
    protected void prepareChart() throws Exception
    {
        _parameters.getTemplateParameters().clearChartSeriesParameters();
        _parameters.getTemplateParameters().getChartDrawingParameters().clearDataSourceParameters();

        for(int i = 0; i < _defaultDrawingParameters.size(); i++)
        {
            final ChartSeriesParameters seriesParams = _defaultSeriesParameters.get(i);
            final DataSourceDrawingParameters drawingParams = _defaultDrawingParameters.get(i);

            //We need to restrict the series to only those that match any one of the time series selection parameters.
            //The old code used a list of parameters and TimeSeriesSorter to restrict view (check earlier svn revision).
            final List<TimeSeriesArray> series = Lists.newArrayList();
            for(final TimeSeriesSelectionParameters tssp: seriesParams.getTSSelectionParameters())
            {
                ListTools.addAllUniqueItemsToList(series, _displayedSeries.extractMatchingTimeSeries(tssp));
            }

            final String plotter = drawingParams.getPlotterName();

            // Simple case.
            Pair<List<ChartSeriesParameters>, List<DataSourceDrawingParameters>> pair;
            if(plotter == null || plotter.equals("AreaUnderLines") || plotter.equals("LineAndScatter")
                || plotter.equals("Step") || plotter.equals("StepAtBeginning") || plotter.equals(""))
            {
                pair = prepareSimpleTimeSeries(series, seriesParams, drawingParams);
            }
            else if(plotter.equals("AreaBetweenLines"))
            {
                // If there's only one pair of time series, we can just add them normally.
                //If there are no time series, then we need to call prepareSimpleTimeSeries
                //to ensure that a data source is still created (prepareFilled* will not create
                //that data source).  This will make the number of data sources consistent,
                //which is useful for external users to manipulate the resulting chart engine.
                if((series.size() == 2) || (series.isEmpty()))
                {
                    pair = prepareSimpleTimeSeries(series, seriesParams, drawingParams);
                }
                else
                {
                    pair = prepareFilledTimeSeries(series, seriesParams, drawingParams);
                }
            }
            else
            {
                throw new RuntimeException("Plotter not yet supported: " + plotter);
            }

            // If forecast is not emphasized, then color.  XXX Future NOTE: This may pull out the time series
            // in the wrong order, if multiple parameterIds are used.  This is because the ts will come out
            //ordered by their LocationAndDataTypeIdentifier (blame Alexander), so if different parameterIds are used,
            //the ordering may not be identical to the base ordering within series.
            final TimeSeriesSorter tsSorter = new TimeSeriesSorter(series);
            if(!tsSorter.restrictViewToUnemphasized().restrictViewToForecast().isEmpty())
            {
                for(final DataSourceDrawingParameters p: pair.second())
                {
                    colorSeries(series, p, _solidColors, _transparentColors);
                }
            }
        }

        final GraphicsGeneratorEngine metaEngine = new GraphicsGeneratorEngine(GraphGenSettingsController.getGlobalGraphGenSettings(),
                                                                               _parameters);
        metaEngine.useExternallyLoadedTimeSeries(TimeSeriesArraysTools.convertListOfTimeSeriesToTimeSeriesArrays(DefaultTimeSeriesHeader.class,
                                                                                                                 getTimeSeriesInDisplayedOrder()));
        metaEngine.prepareChart();

        for(final XYChartDataSource source: metaEngine.getChartEngine().getDataSources())
        {
            if(source instanceof TimeSeriesArraysXYChartDataSource)
            {
                ((TimeSeriesArraysXYChartDataSource)source).setReturnForecastTimeAsTableColumnHeader(true);
            }
        }

        _engine = metaEngine.getChartEngine();
    }

    protected List<TimeSeriesArray> getTimeSeriesInDisplayedOrder()
    {
        return _displayedSeries.toList();
    }

    /**
     * Loads a parameter xml file.
     * 
     * @param resourceName the xml file
     * @throws Exception
     */
    protected void loadParameters(final String resourceName) throws Exception
    {
        _parameters = new GraphicsGeneratorProductParameters();
        final GenericXMLReadingHandler reader = new GenericXMLReadingHandler(_parameters);
        final InputStream stream = ClassLoader.getSystemResourceAsStream(resourceName);
        if(stream == null)
        {
            throw new Exception("Unable to find diagnostic display resource with name " + resourceName);
        }
        reader.readXMLFromStreamAndClose(stream, false);
    }

    /**
     * Adds parameters to allow the specified time series to be displayed. This method copies the drawing parameters for
     * the first series to the other series for which time series exist.
     * 
     * @param seriesColl the list of series to add
     * @param chartParams the chart series parameters to duplicate
     * @param drawParams the data source drawing parameters to duplicate
     */
    protected Pair<List<ChartSeriesParameters>, List<DataSourceDrawingParameters>> prepareSimpleTimeSeries(final Collection<TimeSeriesArray> seriesColl,
                                                                                                           final ChartSeriesParameters chartParams,
                                                                                                           final DataSourceDrawingParameters drawParams)
    {
        final SeriesDrawingParameters seriesParams = drawParams.getSeriesDrawingParametersForSeriesIndex(0);

        // Copy templates.
        final ChartSeriesParameters newChartParams = (ChartSeriesParameters)chartParams.clone();
        final DataSourceDrawingParameters newDrawParams = (DataSourceDrawingParameters)drawParams.clone();
        newDrawParams.setDataSourceOrderIndex(_parameters.getTemplateParameters()
                                                         .getChartDrawingParameters()
                                                         .getDataSourceParametersCount());
        newDrawParams.removeAllSeriesDrawingParameters();

        // Add to main parameters.
        _parameters.getTemplateParameters().addChartSeriesParameters(newChartParams);
        _parameters.getTemplateParameters().getChartDrawingParameters().addDataSourceParameters(newDrawParams);

        // Add parameters for each series.
        final List<TimeSeriesArray> tss = Lists.newArrayList(seriesColl);
        for(int i = 0; i < tss.size(); i++)
        {
            if(tss.size() > drawParams.getSeriesParametersCount())
            {
                final SeriesDrawingParameters newSeriesParams = new SeriesDrawingParameters(i);
                newSeriesParams.copyOverriddenParameters(seriesParams);
                newDrawParams.addSeriesDrawingParameters(newSeriesParams);
            }
            else
            {
                final SeriesDrawingParameters foundSeriesParams = drawParams.getSeriesDrawingParametersForSeriesIndex(i);
                if(foundSeriesParams == null)
                {
                    throw new IllegalArgumentException("Expected series parameters to be defined for index " + i
                        + " but found none.  Check the diagnostic parameter XML GraphGen product files.");
                }
                newDrawParams.addSeriesDrawingParameters(foundSeriesParams);
            }
        }

        return new Pair<List<ChartSeriesParameters>, List<DataSourceDrawingParameters>>(newArrayList(newChartParams),
                                                                                        newArrayList(newDrawParams));
    }

    /**
     * Adds parameters to allow the specified time series to be displayed.<br>
     * Use where you want pairs of min/max in their own data series, for example with the AreaBetweenLines plotter. All
     * series will be slotted into their own forecast times.
     * 
     * @param seriesColl collection of series to draw
     * @param chartParams the chart series parameters to duplicate
     * @param drawParams the data source drawing parameters to duplicate
     */
    // TODO Has a problem with duplicating legend entries.
    private Pair<List<ChartSeriesParameters>, List<DataSourceDrawingParameters>> prepareFilledTimeSeries(final Collection<TimeSeriesArray> seriesColl,
                                                                                                         final ChartSeriesParameters chartParams,
                                                                                                         final DataSourceDrawingParameters drawParams)
    {
        final Map<Long, List<TimeSeriesArray>> seriesByTime = TimeSeriesArrayTools.splitByForecastTime(seriesColl);
        final ChartSeriesParameters newChartParams = (ChartSeriesParameters)chartParams.clone();
        final DataSourceDrawingParameters newDrawParams = (DataSourceDrawingParameters)drawParams.clone();

        final List<ChartSeriesParameters> chartList = new ArrayList<ChartSeriesParameters>();
        final List<DataSourceDrawingParameters> drawList = new ArrayList<DataSourceDrawingParameters>();

        // Loop through each time slot.
        for(final Map.Entry<Long, List<TimeSeriesArray>> entry: seriesByTime.entrySet())
        {
            // Format time.
            final Calendar date = HCalendar.computeCalendarFromMilliseconds(entry.getKey());
            final String format = HEFSDateTools.getGUIDateTimeFormat();
            final String timeString = HCalendar.buildDateStr(date, format);

            // Add time to all selection parameters.
            for(final TimeSeriesSelectionParameters tssp: newChartParams.getTSSelectionParameters())
            {
                tssp.setT0Str(timeString);
            }

            // Set each time series' forecast to its header's forecast.
            for(final TimeSeriesArray tsa: entry.getValue())
            {
                tsa.setForecastTime(tsa.getHeader().getForecastTime());
            }

            newDrawParams.setDataSourceOrderIndex(_parameters.getTemplateParameters()
                                                             .getChartDrawingParameters()
                                                             .getDataSourceParametersCount());
            Pair<List<ChartSeriesParameters>, List<DataSourceDrawingParameters>> pair;
            pair = prepareSimpleTimeSeries(entry.getValue(), newChartParams, newDrawParams);

            chartList.addAll(pair.first());
            drawList.addAll(pair.second());
        }

        return new Pair<List<ChartSeriesParameters>, List<DataSourceDrawingParameters>>(chartList, drawList);
    }

    /**
     * Colors the series in a data source so that ones with the same forecast time have the same color. Assumes the
     * seriesColl is the same collection used to add the parameters in the first place, so the same order is maintained.
     * 
     * @param seriesColl the collection of series used to add the series
     * @param drawParams the draw parameters for this data source
     * @param lineColors the array of line colors to alternate between
     * @param fillColors the array of fill colors to alternate between
     */
    protected void colorSeries(final Collection<TimeSeriesArray> seriesColl,
                               final DataSourceDrawingParameters drawParams,
                               final Color[] lineColors,
                               final Color[] fillColors)
    {
        // Index forecast times.
        int i = 0;
        final Map<Long, Integer> colorMap = new HashMap<Long, Integer>();
        for(final Long time: _forecastTimes)
        {
            colorMap.put(time, i);
            i++;
        }

        i = 0;
        for(final TimeSeriesArray tsa: seriesColl)
        {
            final int colorIndex = colorMap.get(tsa.getHeader().getForecastTime());
            final SeriesDrawingParameters sdp = drawParams.getSeriesDrawingParameters().get(i);
            if(lineColors != null)
            {
                sdp.setLineColor(lineColors[colorIndex % lineColors.length]);
            }
            if(fillColors != null)
            {
                sdp.setFillColor(fillColors[colorIndex % fillColors.length]);
            }
            i++;
        }
    }

    /**
     * Once you've set all the settings, builds the chart.
     * 
     * @return
     * @throws Exception
     */
    public ChartEngine buildChartEngine() throws Exception
    {
        if(_engine == null)
        {
            prepareChart();
        }
        return _engine;
    }
}
